/*
 * Copyright (c) 2021, 2024, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.oracle.truffle.js.nodes.intl;

import java.util.List;

import com.oracle.truffle.api.CompilerDirectives.TruffleBoundary;
import com.oracle.truffle.api.dsl.Cached;
import com.oracle.truffle.api.dsl.Specialization;
import com.oracle.truffle.api.strings.TruffleString;
import com.oracle.truffle.js.nodes.JavaScriptBaseNode;
import com.oracle.truffle.js.nodes.access.PropertyGetNode;
import com.oracle.truffle.js.nodes.cast.JSToBooleanNode;
import com.oracle.truffle.js.nodes.cast.JSToStringNode;
import com.oracle.truffle.js.runtime.Errors;
import com.oracle.truffle.js.runtime.JSConfig;
import com.oracle.truffle.js.runtime.JSContext;
import com.oracle.truffle.js.runtime.objects.Undefined;

/**
 * GetStringOrBooleanOption() operation.
 */
public abstract class GetStringOrBooleanOptionNode extends JavaScriptBaseNode {
    private final List<String> values;
    private final Object trueValue;
    private final Object falsyValue;
    private final Object fallback;
    @Child PropertyGetNode propertyGetNode;

    protected GetStringOrBooleanOptionNode(JSContext context, TruffleString property, List<String> values, Object trueValue, Object falsyValue, Object fallback) {
        this.values = values;
        this.trueValue = trueValue;
        this.falsyValue = falsyValue;
        this.fallback = fallback;
        this.propertyGetNode = PropertyGetNode.create(property, false, context);
    }

    public abstract Object executeValue(Object options);

    public static GetStringOrBooleanOptionNode create(JSContext context, TruffleString property, List<String> values, Object trueValue, Object falsyValue, Object fallback) {
        return GetStringOrBooleanOptionNodeGen.create(context, property, values, trueValue, falsyValue, fallback);
    }

    @Specialization
    public Object getOption(Object options,
                    @Cached(inline = true) JSToBooleanNode toBooleanNode,
                    @Cached JSToStringNode toStringNode,
                    @Cached TruffleString.ToJavaStringNode toJavaStringNode) {
        Object value = propertyGetNode.getValue(options);
        if (value == Undefined.instance) {
            return fallback;
        }
        if (value == Boolean.TRUE) {
            return trueValue;
        }
        boolean valueBoolean = toBooleanNode.executeBoolean(this, value);
        if (propertyGetNode.getContext().getEcmaScriptVersion() < JSConfig.ECMAScript2023) {
            return valueBoolean ? trueValue : falsyValue;
        }
        if (!valueBoolean) {
            return falsyValue;
        }
        String stringValue = toJavaStringNode.execute(toStringNode.executeString(value));
        if ("true".equals(stringValue) || "false".equals(stringValue)) {
            return fallback;
        }
        checkIfAllowed(stringValue);
        return stringValue;
    }

    @TruffleBoundary
    private void checkIfAllowed(String value) {
        if (!values.contains(value)) {
            throw Errors.createRangeError(String.format("invalid option %s", value));
        }
    }

}
