/*
 * Decompiled with CFR 0.152.
 */
package org.junit.platform.engine.support.hierarchical;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Deque;
import java.util.EnumMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.Semaphore;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiFunction;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;
import org.apiguardian.api.API;
import org.jspecify.annotations.Nullable;
import org.junit.platform.commons.logging.Logger;
import org.junit.platform.commons.logging.LoggerFactory;
import org.junit.platform.commons.util.ClassLoaderUtils;
import org.junit.platform.commons.util.ExceptionUtils;
import org.junit.platform.commons.util.Preconditions;
import org.junit.platform.commons.util.ToStringBuilder;
import org.junit.platform.engine.UniqueId;
import org.junit.platform.engine.support.hierarchical.BlockingAwareFuture;
import org.junit.platform.engine.support.hierarchical.ExclusiveResource;
import org.junit.platform.engine.support.hierarchical.HierarchicalTestExecutorService;
import org.junit.platform.engine.support.hierarchical.Node;
import org.junit.platform.engine.support.hierarchical.ParallelExecutionConfiguration;
import org.junit.platform.engine.support.hierarchical.ResourceLock;

@API(status=API.Status.EXPERIMENTAL, since="6.1")
public final class WorkerThreadPoolHierarchicalTestExecutorService
implements HierarchicalTestExecutorService {
    private static final Logger LOGGER = LoggerFactory.getLogger(WorkerThreadPoolHierarchicalTestExecutorService.class);
    private final WorkQueue workQueue = new WorkQueue();
    private final ExecutorService threadPool;
    private final int parallelism;
    private final WorkerLeaseManager workerLeaseManager;

    WorkerThreadPoolHierarchicalTestExecutorService(ParallelExecutionConfiguration configuration) {
        this(configuration, ClassLoaderUtils.getDefaultClassLoader());
    }

    WorkerThreadPoolHierarchicalTestExecutorService(ParallelExecutionConfiguration configuration, ClassLoader classLoader) {
        WorkerThreadFactory threadFactory = new WorkerThreadFactory(classLoader);
        this.parallelism = configuration.getParallelism();
        this.workerLeaseManager = new WorkerLeaseManager(this.parallelism, this::maybeStartWorker);
        LeaseAwareRejectedExecutionHandler rejectedExecutionHandler = new LeaseAwareRejectedExecutionHandler(this.workerLeaseManager);
        this.threadPool = new ThreadPoolExecutor(configuration.getCorePoolSize(), configuration.getMaxPoolSize(), configuration.getKeepAliveSeconds(), TimeUnit.SECONDS, new SynchronousQueue<Runnable>(), threadFactory, rejectedExecutionHandler);
        LOGGER.trace(() -> "initialized thread pool for parallelism of " + configuration.getParallelism());
    }

    @Override
    public void close() {
        LOGGER.trace(() -> "shutting down thread pool");
        this.threadPool.shutdownNow();
    }

    @Override
    public Future<@Nullable Void> submit(HierarchicalTestExecutorService.TestTask testTask) {
        LOGGER.trace(() -> "submit: " + String.valueOf(testTask));
        WorkerThread workerThread = WorkerThread.get();
        if (workerThread == null) {
            return this.enqueue(testTask, 0).future();
        }
        if (testTask.getExecutionMode() == Node.ExecutionMode.SAME_THREAD) {
            workerThread.executeTask(testTask);
            return CompletableFuture.completedFuture(null);
        }
        WorkQueue.Entry entry = this.enqueue(testTask, workerThread.nextChildIndex());
        workerThread.trackSubmittedChild(entry);
        return new WorkStealingFuture(entry);
    }

    @Override
    public void invokeAll(List<? extends HierarchicalTestExecutorService.TestTask> testTasks) {
        LOGGER.trace(() -> "invokeAll: " + String.valueOf(testTasks));
        WorkerThread workerThread = WorkerThread.get();
        Preconditions.condition((workerThread != null && workerThread.executor() == this ? 1 : 0) != 0, (String)"invokeAll() must be called from a worker thread that belongs to this executor");
        workerThread.invokeAll(testTasks);
    }

    private WorkQueue.Entry enqueue(HierarchicalTestExecutorService.TestTask testTask, int index) {
        WorkQueue.Entry entry = this.workQueue.add(testTask, index);
        this.maybeStartWorker();
        return entry;
    }

    private void forkAll(Collection<WorkQueue.Entry> entries) {
        if (entries.isEmpty()) {
            return;
        }
        this.workQueue.addAll(entries);
        for (int i = 0; i < Math.min(this.parallelism - 1, entries.size()); ++i) {
            this.maybeStartWorker();
        }
    }

    private void maybeStartWorker() {
        this.maybeStartWorker(() -> false);
    }

    private void maybeStartWorker(BooleanSupplier doneCondition) {
        if (this.threadPool.isShutdown() || this.workQueue.isEmpty() || doneCondition.getAsBoolean()) {
            return;
        }
        WorkerLease workerLease = this.workerLeaseManager.tryAcquire();
        if (workerLease == null) {
            return;
        }
        this.threadPool.execute(new RunLeaseAwareWorker(workerLease, () -> WorkerThread.getOrThrow().processQueueEntries(workerLease, doneCondition), () -> this.maybeStartWorker(doneCondition)));
    }

    private static class WorkQueue
    implements Iterable<Entry> {
        private final Set<Entry> queue = new ConcurrentSkipListSet<Entry>(Entry.QUEUE_COMPARATOR);

        private WorkQueue() {
        }

        Entry add(HierarchicalTestExecutorService.TestTask task, int index) {
            Entry entry = new Entry(task, index);
            LOGGER.trace(() -> "forking: " + String.valueOf(entry.task));
            return this.doAdd(entry);
        }

        void addAll(Collection<Entry> entries) {
            entries.forEach(this::doAdd);
        }

        void reAdd(Entry entry) {
            LOGGER.trace(() -> "re-enqueuing: " + String.valueOf(entry.task));
            this.doAdd(entry);
        }

        private Entry doAdd(Entry entry) {
            boolean added = this.queue.add(entry);
            if (!added) {
                throw new IllegalStateException("Could not add entry to the queue for task: " + String.valueOf(entry.task));
            }
            return entry;
        }

        boolean remove(Entry entry) {
            return this.queue.remove(entry);
        }

        boolean isEmpty() {
            return this.queue.isEmpty();
        }

        @Override
        public Iterator<Entry> iterator() {
            return this.queue.iterator();
        }

        private static final class Entry {
            private static final Comparator<Entry> QUEUE_COMPARATOR = Comparator.comparing(Entry::level).reversed().thenComparing(Entry::isContainer).thenComparing(Entry::index).thenComparing(Entry::uniqueId, new SameLengthUniqueIdComparator());
            private static final Comparator<Entry> CHILD_COMPARATOR = Comparator.comparing(Entry::isContainer).reversed().thenComparing(Entry::index);
            private final HierarchicalTestExecutorService.TestTask task;
            private final CompletableFuture<@Nullable Void> future = new CompletableFuture();
            private final int index;

            Entry(HierarchicalTestExecutorService.TestTask task, int index) {
                this.future.whenComplete((__, t) -> {
                    if (t == null) {
                        LOGGER.trace(() -> "completed normally: " + String.valueOf(task));
                    } else {
                        LOGGER.trace(t, () -> "completed exceptionally: " + String.valueOf(task));
                    }
                });
                this.task = task;
                this.index = index;
            }

            private int index() {
                return this.index;
            }

            private int level() {
                return this.uniqueId().getSegments().size();
            }

            private boolean isContainer() {
                return this.task.getTestDescriptor().isContainer();
            }

            private UniqueId uniqueId() {
                return this.task.getTestDescriptor().getUniqueId();
            }

            CompletableFuture<@Nullable Void> future() {
                return this.future;
            }

            public boolean equals(Object obj) {
                if (obj == this) {
                    return true;
                }
                if (obj == null || obj.getClass() != this.getClass()) {
                    return false;
                }
                Entry that = (Entry)obj;
                return Objects.equals(this.uniqueId(), that.uniqueId()) && this.index == that.index;
            }

            public int hashCode() {
                return Objects.hash(this.uniqueId(), this.index);
            }

            public String toString() {
                return new ToStringBuilder((Object)this).append("task", (Object)this.task).append("index", (Object)this.index).toString();
            }

            private static class SameLengthUniqueIdComparator
            implements Comparator<UniqueId> {
                private SameLengthUniqueIdComparator() {
                }

                @Override
                public int compare(UniqueId a, UniqueId b) {
                    Iterator<UniqueId.Segment> aIterator = a.getSegments().iterator();
                    Iterator<UniqueId.Segment> bIterator = b.getSegments().iterator();
                    while (aIterator.hasNext()) {
                        UniqueId.Segment bCurrent;
                        UniqueId.Segment aCurrent = aIterator.next();
                        int result = SameLengthUniqueIdComparator.compareBy(aCurrent, bCurrent = bIterator.next());
                        if (result == 0) continue;
                        return result;
                    }
                    return 0;
                }

                private static int compareBy(UniqueId.Segment a, UniqueId.Segment b) {
                    int result = a.getType().compareTo(b.getType());
                    if (result != 0) {
                        return result;
                    }
                    return a.getValue().compareTo(b.getValue());
                }
            }
        }
    }

    private class WorkerThreadFactory
    implements ThreadFactory {
        private static final AtomicInteger POOL_NUMBER = new AtomicInteger(1);
        private final AtomicInteger threadNumber = new AtomicInteger(1);
        private final int poolNumber;
        private final ClassLoader classLoader;

        WorkerThreadFactory(ClassLoader classLoader) {
            this.classLoader = classLoader;
            this.poolNumber = POOL_NUMBER.getAndIncrement();
        }

        @Override
        public Thread newThread(Runnable runnable) {
            WorkerThread thread = new WorkerThread(runnable, "junit-%d-worker-%d".formatted(this.poolNumber, this.threadNumber.getAndIncrement()));
            thread.setContextClassLoader(this.classLoader);
            return thread;
        }
    }

    static class WorkerLeaseManager {
        private final int parallelism;
        private final Semaphore semaphore;
        private final Consumer<BooleanSupplier> compensation;

        WorkerLeaseManager(int parallelism, Consumer<BooleanSupplier> compensation) {
            this.parallelism = parallelism;
            this.semaphore = new Semaphore(parallelism);
            this.compensation = compensation;
        }

        @Nullable WorkerLease tryAcquire() {
            boolean acquired = this.semaphore.tryAcquire();
            if (acquired) {
                LOGGER.trace(() -> "acquired worker lease for new worker (available: %d)".formatted(this.semaphore.availablePermits()));
                return new WorkerLease(this::release);
            }
            return null;
        }

        private ReacquisitionToken release(boolean compensate, BooleanSupplier doneCondition) {
            this.semaphore.release();
            LOGGER.trace(() -> "release worker lease (available: %d)".formatted(this.semaphore.availablePermits()));
            if (compensate) {
                this.compensation.accept(doneCondition);
            }
            return new ReacquisitionToken();
        }

        public boolean isAtLeastOneLeaseTaken() {
            return this.semaphore.availablePermits() < this.parallelism;
        }

        public String toString() {
            return new ToStringBuilder((Object)this).append("parallelism", (Object)this.parallelism).append("semaphore", (Object)this.semaphore).toString();
        }

        private class ReacquisitionToken {
            private boolean used = false;

            private ReacquisitionToken() {
            }

            void reacquire() throws InterruptedException {
                Preconditions.condition((!this.used ? 1 : 0) != 0, (String)"Lease was already reacquired");
                this.used = true;
                WorkerLeaseManager.this.semaphore.acquire();
                LOGGER.trace(() -> "reacquired worker lease (available: %d)".formatted(WorkerLeaseManager.this.semaphore.availablePermits()));
            }
        }
    }

    private record LeaseAwareRejectedExecutionHandler(WorkerLeaseManager workerLeaseManager) implements RejectedExecutionHandler
    {
        @Override
        public void rejectedExecution(Runnable r, ThreadPoolExecutor executor) {
            if (!(r instanceof RunLeaseAwareWorker)) {
                return;
            }
            RunLeaseAwareWorker worker = (RunLeaseAwareWorker)r;
            worker.workerLease.release(false);
            if (executor.isShutdown() || this.workerLeaseManager.isAtLeastOneLeaseTaken()) {
                return;
            }
            throw new RejectedExecutionException("Task with " + String.valueOf(this.workerLeaseManager) + " rejected from " + String.valueOf(executor));
        }
    }

    private class WorkerThread
    extends Thread {
        private final Deque<State> stateStack;
        @Nullable WorkerLease workerLease;

        WorkerThread(Runnable runnable, String name) {
            super(runnable, name);
            this.stateStack = new ArrayDeque<State>();
        }

        static @Nullable WorkerThread get() {
            Thread thread = Thread.currentThread();
            if (thread instanceof WorkerThread) {
                WorkerThread workerThread = (WorkerThread)thread;
                return workerThread;
            }
            return null;
        }

        static WorkerThread getOrThrow() {
            WorkerThread workerThread = WorkerThread.get();
            if (workerThread == null) {
                throw new IllegalStateException("Not on a worker thread");
            }
            return workerThread;
        }

        WorkerThreadPoolHierarchicalTestExecutorService executor() {
            return WorkerThreadPoolHierarchicalTestExecutorService.this;
        }

        void processQueueEntries(WorkerLease workerLease, BooleanSupplier doneCondition) {
            this.workerLease = workerLease;
            while (!WorkerThreadPoolHierarchicalTestExecutorService.this.threadPool.isShutdown()) {
                if (doneCondition.getAsBoolean()) {
                    LOGGER.trace(() -> "yielding resource lock");
                    break;
                }
                if (WorkerThreadPoolHierarchicalTestExecutorService.this.workQueue.isEmpty()) {
                    LOGGER.trace(() -> "no queue entries available");
                    break;
                }
                this.processQueueEntries();
            }
        }

        private void processQueueEntries() {
            WorkStealResult result;
            ArrayList<WorkQueue.Entry> entriesRequiringResourceLocks = new ArrayList<WorkQueue.Entry>();
            for (WorkQueue.Entry entry : WorkerThreadPoolHierarchicalTestExecutorService.this.workQueue) {
                result = this.tryToStealWork(entry, BlockingMode.NON_BLOCKING);
                if (result == WorkStealResult.EXECUTED_BY_THIS_WORKER) {
                    return;
                }
                if (result != WorkStealResult.RESOURCE_LOCK_UNAVAILABLE) continue;
                entriesRequiringResourceLocks.add(entry);
            }
            for (WorkQueue.Entry entry : entriesRequiringResourceLocks) {
                result = this.tryToStealWork(entry, BlockingMode.BLOCKING);
                if (result != WorkStealResult.EXECUTED_BY_THIS_WORKER) continue;
                return;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        <T> T runBlocking(BooleanSupplier doneCondition, BlockingAction<T> blockingAction) throws InterruptedException {
            WorkerLease workerLease = Objects.requireNonNull(this.workerLease);
            workerLease.release(doneCondition);
            try {
                T t = blockingAction.run();
                return t;
            }
            finally {
                try {
                    workerLease.reacquire();
                }
                catch (InterruptedException e) {
                    this.interrupt();
                }
            }
        }

        void invokeAll(List<? extends HierarchicalTestExecutorService.TestTask> testTasks) {
            if (testTasks.isEmpty()) {
                return;
            }
            if (testTasks.size() == 1) {
                this.executeTask(testTasks.get(0));
                return;
            }
            ArrayList isolatedTasks = new ArrayList(testTasks.size());
            ArrayList<HierarchicalTestExecutorService.TestTask> sameThreadTasks = new ArrayList<HierarchicalTestExecutorService.TestTask>(testTasks.size());
            List<WorkQueue.Entry> queueEntries = this.forkConcurrentChildren(testTasks, isolatedTasks::add, sameThreadTasks);
            this.executeAll(sameThreadTasks);
            Map<WorkStealResult, List<WorkQueue.Entry>> queueEntriesByResult = this.tryToStealWorkWithoutBlocking(queueEntries);
            this.tryToStealWorkWithBlocking(queueEntriesByResult);
            this.waitFor(queueEntriesByResult);
            this.executeAll(isolatedTasks);
        }

        private List<WorkQueue.Entry> forkConcurrentChildren(List<? extends HierarchicalTestExecutorService.TestTask> children, Consumer<HierarchicalTestExecutorService.TestTask> isolatedTaskCollector, List<HierarchicalTestExecutorService.TestTask> sameThreadTasks) {
            ArrayList<WorkQueue.Entry> queueEntries = new ArrayList<WorkQueue.Entry>(children.size());
            for (HierarchicalTestExecutorService.TestTask testTask : children) {
                if (WorkerThread.requiresGlobalReadWriteLock(testTask)) {
                    isolatedTaskCollector.accept(testTask);
                    continue;
                }
                if (testTask.getExecutionMode() == Node.ExecutionMode.SAME_THREAD) {
                    sameThreadTasks.add(testTask);
                    continue;
                }
                queueEntries.add(new WorkQueue.Entry(testTask, this.nextChildIndex()));
            }
            if (!queueEntries.isEmpty()) {
                queueEntries.sort(WorkQueue.Entry.CHILD_COMPARATOR);
                if (sameThreadTasks.isEmpty()) {
                    WorkQueue.Entry firstEntry = (WorkQueue.Entry)queueEntries.remove(0);
                    sameThreadTasks.add(firstEntry.task);
                }
                WorkerThreadPoolHierarchicalTestExecutorService.this.forkAll(queueEntries);
            }
            return queueEntries;
        }

        private Map<WorkStealResult, List<WorkQueue.Entry>> tryToStealWorkWithoutBlocking(Iterable<WorkQueue.Entry> queueEntries) {
            EnumMap<WorkStealResult, List<WorkQueue.Entry>> queueEntriesByResult = new EnumMap<WorkStealResult, List<WorkQueue.Entry>>(WorkStealResult.class);
            this.tryToStealWork(queueEntries, BlockingMode.NON_BLOCKING, queueEntriesByResult);
            return queueEntriesByResult;
        }

        private void tryToStealWorkWithBlocking(Map<WorkStealResult, List<WorkQueue.Entry>> queueEntriesByResult) {
            List<WorkQueue.Entry> entriesRequiringResourceLocks = queueEntriesByResult.remove((Object)WorkStealResult.RESOURCE_LOCK_UNAVAILABLE);
            if (entriesRequiringResourceLocks == null) {
                return;
            }
            this.tryToStealWork(entriesRequiringResourceLocks, BlockingMode.BLOCKING, queueEntriesByResult);
        }

        private void tryToStealWork(Iterable<WorkQueue.Entry> entries, BlockingMode blocking, Map<WorkStealResult, List<WorkQueue.Entry>> queueEntriesByResult) {
            for (WorkQueue.Entry entry : entries) {
                WorkStealResult result = this.tryToStealWork(entry, blocking);
                queueEntriesByResult.computeIfAbsent(result, __ -> new ArrayList()).add(entry);
            }
        }

        private WorkStealResult tryToStealWork(WorkQueue.Entry entry, BlockingMode blockingMode) {
            if (entry.future.isDone()) {
                return WorkStealResult.EXECUTED_BY_DIFFERENT_WORKER;
            }
            boolean claimed = WorkerThreadPoolHierarchicalTestExecutorService.this.workQueue.remove(entry);
            if (claimed) {
                LOGGER.trace(() -> "stole work: " + String.valueOf(entry.task));
                boolean executed = this.executeStolenWork(entry, blockingMode);
                if (executed) {
                    return WorkStealResult.EXECUTED_BY_THIS_WORKER;
                }
                WorkerThreadPoolHierarchicalTestExecutorService.this.workQueue.reAdd(entry);
                return WorkStealResult.RESOURCE_LOCK_UNAVAILABLE;
            }
            return WorkStealResult.EXECUTED_BY_DIFFERENT_WORKER;
        }

        private void waitFor(Map<WorkStealResult, List<WorkQueue.Entry>> queueEntriesByResult) {
            List<WorkQueue.Entry> children = queueEntriesByResult.get((Object)WorkStealResult.EXECUTED_BY_DIFFERENT_WORKER);
            if (children == null) {
                return;
            }
            CompletableFuture<?> future = WorkerThread.toCombinedFuture(children);
            try {
                if (future.isDone()) {
                    future.join();
                } else {
                    this.runBlocking(future::isDone, () -> {
                        LOGGER.trace(() -> "blocking for forked children : %s".formatted(children));
                        return future.join();
                    });
                }
            }
            catch (InterruptedException e) {
                WorkerThread.currentThread().interrupt();
            }
        }

        private static boolean requiresGlobalReadWriteLock(HierarchicalTestExecutorService.TestTask testTask) {
            return testTask.getResourceLock().getResources().contains(ExclusiveResource.GLOBAL_READ_WRITE);
        }

        private void executeAll(List<? extends HierarchicalTestExecutorService.TestTask> children) {
            if (children.isEmpty()) {
                return;
            }
            LOGGER.trace(() -> "running %d children directly".formatted(children.size()));
            if (children.size() == 1) {
                this.executeTask(children.get(0));
                return;
            }
            for (HierarchicalTestExecutorService.TestTask testTask : children) {
                this.executeTask(testTask);
            }
        }

        private boolean executeStolenWork(WorkQueue.Entry entry, BlockingMode blockingMode) {
            return switch (blockingMode.ordinal()) {
                default -> throw new IncompatibleClassChangeError();
                case 0 -> this.tryExecute(entry);
                case 1 -> {
                    this.execute(entry);
                    yield true;
                }
            };
        }

        private boolean tryExecute(WorkQueue.Entry entry) {
            try {
                boolean executed = this.tryExecuteTask(entry.task);
                if (executed) {
                    entry.future.complete(null);
                }
                return executed;
            }
            catch (Throwable t) {
                entry.future.completeExceptionally(t);
                return true;
            }
        }

        private void execute(WorkQueue.Entry entry) {
            try {
                this.executeTask(entry.task);
            }
            catch (Throwable t) {
                entry.future.completeExceptionally(t);
            }
            finally {
                entry.future.complete(null);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void executeTask(HierarchicalTestExecutorService.TestTask testTask) {
            boolean executed = this.tryExecuteTask(testTask);
            if (!executed) {
                ResourceLock resourceLock = testTask.getResourceLock();
                try (ResourceLock ignored = this.runBlocking(() -> false, () -> {
                    LOGGER.trace(() -> "blocking for resource lock: " + String.valueOf(resourceLock));
                    return resourceLock.acquire();
                });){
                    LOGGER.trace(() -> "acquired resource lock: " + String.valueOf(resourceLock));
                    this.doExecute(testTask);
                }
                catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                }
                finally {
                    LOGGER.trace(() -> "released resource lock: " + String.valueOf(resourceLock));
                }
            }
        }

        private boolean tryExecuteTask(HierarchicalTestExecutorService.TestTask testTask) {
            ResourceLock resourceLock = testTask.getResourceLock();
            if (resourceLock.tryAcquire()) {
                LOGGER.trace(() -> "acquired resource lock: " + String.valueOf(resourceLock));
                try {
                    ResourceLock resourceLock2 = resourceLock;
                    try {
                        this.doExecute(testTask);
                        boolean bl = true;
                        if (resourceLock2 != null) {
                            resourceLock2.close();
                        }
                        return bl;
                    }
                    catch (Throwable throwable) {
                        if (resourceLock2 != null) {
                            try {
                                resourceLock2.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                }
                finally {
                    LOGGER.trace(() -> "released resource lock: " + String.valueOf(resourceLock));
                }
            }
            LOGGER.trace(() -> "failed to acquire resource lock: " + String.valueOf(resourceLock));
            return false;
        }

        private void doExecute(HierarchicalTestExecutorService.TestTask testTask) {
            LOGGER.trace(() -> "executing: " + String.valueOf(testTask));
            this.stateStack.push(new State());
            try {
                testTask.execute();
            }
            finally {
                this.stateStack.pop();
                LOGGER.trace(() -> "finished executing: " + String.valueOf(testTask));
            }
        }

        private static CompletableFuture<?> toCombinedFuture(List<WorkQueue.Entry> entries) {
            if (entries.size() == 1) {
                return entries.get(0).future();
            }
            CompletableFuture[] futures = (CompletableFuture[])entries.stream().map(WorkQueue.Entry::future).toArray(CompletableFuture[]::new);
            return CompletableFuture.allOf(futures);
        }

        private int nextChildIndex() {
            return this.stateStack.element().nextChildIndex();
        }

        private void trackSubmittedChild(WorkQueue.Entry entry) {
            this.stateStack.element().trackSubmittedChild(entry);
        }

        private void tryToStealWorkFromSubmittedChildren() {
            State currentState = this.stateStack.element();
            List<WorkQueue.Entry> currentSubmittedChildren = currentState.submittedChildren;
            if (currentSubmittedChildren == null || currentSubmittedChildren.isEmpty()) {
                return;
            }
            currentSubmittedChildren.sort(WorkQueue.Entry.CHILD_COMPARATOR);
            Iterator<WorkQueue.Entry> iterator = currentSubmittedChildren.iterator();
            while (iterator.hasNext()) {
                WorkQueue.Entry entry = iterator.next();
                WorkStealResult result = this.tryToStealWork(entry, BlockingMode.NON_BLOCKING);
                if (!result.isExecuted()) continue;
                iterator.remove();
            }
            currentState.clearIfEmpty();
        }

        private static enum WorkStealResult {
            EXECUTED_BY_DIFFERENT_WORKER,
            RESOURCE_LOCK_UNAVAILABLE,
            EXECUTED_BY_THIS_WORKER;


            private boolean isExecuted() {
                return this != RESOURCE_LOCK_UNAVAILABLE;
            }
        }

        private static interface BlockingAction<T> {
            public T run() throws InterruptedException;
        }

        private static class State {
            private int nextChildIndex = 0;
            private @Nullable List<WorkQueue.Entry> submittedChildren;

            private State() {
            }

            private void trackSubmittedChild(WorkQueue.Entry entry) {
                if (this.submittedChildren == null) {
                    this.submittedChildren = new ArrayList<WorkQueue.Entry>();
                }
                this.submittedChildren.add(entry);
            }

            private void clearIfEmpty() {
                if (this.submittedChildren != null && this.submittedChildren.isEmpty()) {
                    this.submittedChildren = null;
                }
            }

            private int nextChildIndex() {
                return this.nextChildIndex++;
            }
        }
    }

    private static class WorkStealingFuture
    extends BlockingAwareFuture<Void> {
        private final WorkQueue.Entry entry;

        WorkStealingFuture(WorkQueue.Entry entry) {
            super(entry.future);
            this.entry = entry;
        }

        @Override
        protected @Nullable Void handle(Callable<@Nullable Void> callable) throws Exception {
            WorkerThread workerThread = WorkerThread.get();
            if (workerThread == null || this.entry.future.isDone()) {
                return callable.call();
            }
            workerThread.tryToStealWork(this.entry, BlockingMode.BLOCKING);
            if (this.entry.future.isDone()) {
                return callable.call();
            }
            workerThread.tryToStealWorkFromSubmittedChildren();
            if (this.entry.future.isDone()) {
                return callable.call();
            }
            LOGGER.trace(() -> "blocking for child task: " + String.valueOf(this.entry.task));
            return workerThread.runBlocking(this.entry.future::isDone, () -> {
                try {
                    return (Void)callable.call();
                }
                catch (Exception ex) {
                    throw ExceptionUtils.throwAsUncheckedException((Throwable)ex);
                }
            });
        }
    }

    static class WorkerLease
    implements AutoCloseable {
        private final BiFunction<Boolean, BooleanSupplier, WorkerLeaseManager.ReacquisitionToken> releaseAction;
        private @Nullable WorkerLeaseManager.ReacquisitionToken reacquisitionToken;

        WorkerLease(BiFunction<Boolean, BooleanSupplier, WorkerLeaseManager.ReacquisitionToken> releaseAction) {
            this.releaseAction = releaseAction;
        }

        @Override
        public void close() {
            this.release(true);
        }

        public void release(BooleanSupplier doneCondition) {
            this.release(true, doneCondition);
        }

        void release(boolean compensate) {
            this.release(compensate, () -> false);
        }

        void release(boolean compensate, BooleanSupplier doneCondition) {
            if (this.reacquisitionToken == null) {
                this.reacquisitionToken = this.releaseAction.apply(compensate, doneCondition);
            }
        }

        void reacquire() throws InterruptedException {
            Preconditions.notNull((Object)this.reacquisitionToken, (String)"Cannot reacquire an unreleased WorkerLease");
            this.reacquisitionToken.reacquire();
            this.reacquisitionToken = null;
        }
    }

    private record RunLeaseAwareWorker(WorkerLease workerLease, Runnable work, Runnable onWorkerFinished) implements Runnable
    {
        @Override
        public void run() {
            LOGGER.trace(() -> "starting worker");
            try {
                this.work.run();
            }
            finally {
                this.workerLease.release(false);
                LOGGER.trace(() -> "stopping worker");
            }
            this.onWorkerFinished.run();
        }
    }

    private static enum BlockingMode {
        NON_BLOCKING,
        BLOCKING;

    }
}

