/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.core;

import jakarta.json.stream.JsonGenerator;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.JsonpSerializer;
import org.opensearch.client.json.JsonpUtils;
import org.opensearch.client.json.PlainJsonSerializable;
import org.opensearch.client.opensearch._types.ErrorResponse;
import org.opensearch.client.opensearch._types.Refresh;
import org.opensearch.client.opensearch._types.RequestBase;
import org.opensearch.client.opensearch._types.Time;
import org.opensearch.client.opensearch._types.VersionType;
import org.opensearch.client.opensearch._types.WaitForActiveShards;
import org.opensearch.client.transport.Endpoint;
import org.opensearch.client.transport.endpoints.SimpleEndpoint;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: _global.create.Request

/**
 * Creates a new document in the index.
 * <p>
 * Returns a 409 response when a document with a same ID already exists in the
 * index.
 *
 */

public class CreateRequest<TDocument> extends RequestBase implements PlainJsonSerializable {
    private final String id;

    private final String index;

    @Nullable
    private final String pipeline;

    @Nullable
    private final Refresh refresh;

    @Nullable
    private final String routing;

    @Nullable
    private final Time timeout;

    @Nullable
    private final Long version;

    @Nullable
    private final VersionType versionType;

    @Nullable
    private final WaitForActiveShards waitForActiveShards;

    private final TDocument document;

    @Nullable
    private final JsonpSerializer<TDocument> tDocumentSerializer;

    // ---------------------------------------------------------------------------------------------

    private CreateRequest(Builder<TDocument> builder) {

        this.id = ApiTypeHelper.requireNonNull(builder.id, this, "id");
        this.index = ApiTypeHelper.requireNonNull(builder.index, this, "index");
        this.pipeline = builder.pipeline;
        this.refresh = builder.refresh;
        this.routing = builder.routing;
        this.timeout = builder.timeout;
        this.version = builder.version;
        this.versionType = builder.versionType;
        this.waitForActiveShards = builder.waitForActiveShards;
        this.document = ApiTypeHelper.requireNonNull(builder.document, this, "document");
        this.tDocumentSerializer = builder.tDocumentSerializer;

    }

    public static <TDocument> CreateRequest<TDocument> of(Function<Builder<TDocument>, ObjectBuilder<CreateRequest<TDocument>>> fn) {
        return fn.apply(new Builder<>()).build();
    }

    /**
     * Required - Document ID
     * <p>
     * API name: {@code id}
     */
    public final String id() {
        return this.id;
    }

    /**
     * Required - The name of the index
     * <p>
     * API name: {@code index}
     */
    public final String index() {
        return this.index;
    }

    /**
     * The pipeline id to preprocess incoming documents with
     * <p>
     * API name: {@code pipeline}
     */
    @Nullable
    public final String pipeline() {
        return this.pipeline;
    }

    /**
     * If <code>true</code> then refresh the affected shards to make this operation
     * visible to search, if <code>wait_for</code> then wait for a refresh to make
     * this operation visible to search, if <code>false</code> (the default) then do
     * nothing with refreshes.
     * <p>
     * API name: {@code refresh}
     */
    @Nullable
    public final Refresh refresh() {
        return this.refresh;
    }

    /**
     * Specific routing value
     * <p>
     * API name: {@code routing}
     */
    @Nullable
    public final String routing() {
        return this.routing;
    }

    /**
     * Explicit operation timeout
     * <p>
     * API name: {@code timeout}
     */
    @Nullable
    public final Time timeout() {
        return this.timeout;
    }

    /**
     * Explicit version number for concurrency control
     * <p>
     * API name: {@code version}
     */
    @Nullable
    public final Long version() {
        return this.version;
    }

    /**
     * Specific version type
     * <p>
     * API name: {@code version_type}
     */
    @Nullable
    public final VersionType versionType() {
        return this.versionType;
    }

    /**
     * Sets the number of shard copies that must be active before proceeding with
     * the index operation. Defaults to 1, meaning the primary shard only. Set to
     * <code>all</code> for all shard copies, otherwise set to any non-negative
     * value less than or equal to the total number of copies for the shard (number
     * of replicas + 1)
     * <p>
     * API name: {@code wait_for_active_shards}
     */
    @Nullable
    public final WaitForActiveShards waitForActiveShards() {
        return this.waitForActiveShards;
    }

    /**
     * Required - Request body.
     * <p>
     * API name: {@code _value_body}
     */
    public final TDocument document() {
        return this.document;
    }

    /**
     * Serialize this value to JSON.
     */
    public void serialize(JsonGenerator generator, JsonpMapper mapper) {
        JsonpUtils.serialize(this.document, generator, tDocumentSerializer, mapper);

    }

    public Builder<TDocument> toBuilder() {
        return new Builder<TDocument>().id(id)
            .index(index)
            .pipeline(pipeline)
            .refresh(refresh)
            .routing(routing)
            .timeout(timeout)
            .version(version)
            .versionType(versionType)
            .waitForActiveShards(waitForActiveShards)
            .document(document)
            .tDocumentSerializer(tDocumentSerializer);
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link CreateRequest}.
     */

    public static class Builder<TDocument> extends ObjectBuilderBase implements ObjectBuilder<CreateRequest<TDocument>> {
        private String id;

        private String index;

        @Nullable
        private String pipeline;

        @Nullable
        private Refresh refresh;

        @Nullable
        private String routing;

        @Nullable
        private Time timeout;

        @Nullable
        private Long version;

        @Nullable
        private VersionType versionType;

        @Nullable
        private WaitForActiveShards waitForActiveShards;

        private TDocument document;

        @Nullable
        private JsonpSerializer<TDocument> tDocumentSerializer;

        /**
         * Required - Document ID
         * <p>
         * API name: {@code id}
         */
        public final Builder<TDocument> id(String value) {
            this.id = value;
            return this;
        }

        /**
         * Required - The name of the index
         * <p>
         * API name: {@code index}
         */
        public final Builder<TDocument> index(String value) {
            this.index = value;
            return this;
        }

        /**
         * The pipeline id to preprocess incoming documents with
         * <p>
         * API name: {@code pipeline}
         */
        public final Builder<TDocument> pipeline(@Nullable String value) {
            this.pipeline = value;
            return this;
        }

        /**
         * If <code>true</code> then refresh the affected shards to make this operation
         * visible to search, if <code>wait_for</code> then wait for a refresh to make
         * this operation visible to search, if <code>false</code> (the default) then do
         * nothing with refreshes.
         * <p>
         * API name: {@code refresh}
         */
        public final Builder<TDocument> refresh(@Nullable Refresh value) {
            this.refresh = value;
            return this;
        }

        /**
         * Specific routing value
         * <p>
         * API name: {@code routing}
         */
        public final Builder<TDocument> routing(@Nullable String value) {
            this.routing = value;
            return this;
        }

        /**
         * Explicit operation timeout
         * <p>
         * API name: {@code timeout}
         */
        public final Builder<TDocument> timeout(@Nullable Time value) {
            this.timeout = value;
            return this;
        }

        /**
         * Explicit operation timeout
         * <p>
         * API name: {@code timeout}
         */
        public final Builder<TDocument> timeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.timeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * Explicit version number for concurrency control
         * <p>
         * API name: {@code version}
         */
        public final Builder<TDocument> version(@Nullable Long value) {
            this.version = value;
            return this;
        }

        /**
         * Specific version type
         * <p>
         * API name: {@code version_type}
         */
        public final Builder<TDocument> versionType(@Nullable VersionType value) {
            this.versionType = value;
            return this;
        }

        /**
         * Sets the number of shard copies that must be active before proceeding with
         * the index operation. Defaults to 1, meaning the primary shard only. Set to
         * <code>all</code> for all shard copies, otherwise set to any non-negative
         * value less than or equal to the total number of copies for the shard (number
         * of replicas + 1)
         * <p>
         * API name: {@code wait_for_active_shards}
         */
        public final Builder<TDocument> waitForActiveShards(@Nullable WaitForActiveShards value) {
            this.waitForActiveShards = value;
            return this;
        }

        /**
         * Sets the number of shard copies that must be active before proceeding with
         * the index operation. Defaults to 1, meaning the primary shard only. Set to
         * <code>all</code> for all shard copies, otherwise set to any non-negative
         * value less than or equal to the total number of copies for the shard (number
         * of replicas + 1)
         * <p>
         * API name: {@code wait_for_active_shards}
         */
        public final Builder<TDocument> waitForActiveShards(Function<WaitForActiveShards.Builder, ObjectBuilder<WaitForActiveShards>> fn) {
            return this.waitForActiveShards(fn.apply(new WaitForActiveShards.Builder()).build());
        }

        /**
         * Required - Request body.
         * <p>
         * API name: {@code _value_body}
         */
        public final Builder<TDocument> document(TDocument value) {
            this.document = value;
            return this;
        }

        /**
         * Serializer for TDocument. If not set, an attempt will be made to find a
         * serializer from the JSON context.
         */
        public final Builder<TDocument> tDocumentSerializer(@Nullable JsonpSerializer<TDocument> value) {
            this.tDocumentSerializer = value;
            return this;
        }

        /**
         * Builds a {@link CreateRequest}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public CreateRequest<TDocument> build() {
            _checkSingleUse();

            return new CreateRequest<TDocument>(this);
        }
    }

    public static <TDocument> JsonpDeserializer<CreateRequest<TDocument>> createCreateRequestDeserializer(
        JsonpDeserializer<TDocument> tDocumentDeserializer
    ) {

        JsonpDeserializer<TDocument> valueDeserializer = tDocumentDeserializer;

        return JsonpDeserializer.of(
            valueDeserializer.acceptedEvents(),
            (parser, mapper) -> new Builder<TDocument>().document(valueDeserializer.deserialize(parser, mapper)).build()
        );
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Endpoint "{@code create}".
     */
    public static final Endpoint<CreateRequest<?>, CreateResponse, ErrorResponse> _ENDPOINT = new SimpleEndpoint<>(

        // Request method
        request -> {
            return "PUT";

        },

        // Request path
        request -> {
            final int _index = 1 << 0;
            final int _id = 1 << 1;

            int propsSet = 0;

            propsSet |= _index;
            propsSet |= _id;

            if (propsSet == (_index | _id)) {
                StringBuilder buf = new StringBuilder();
                buf.append("/");
                SimpleEndpoint.pathEncode(request.index, buf);
                buf.append("/_create");
                buf.append("/");
                SimpleEndpoint.pathEncode(request.id, buf);
                return buf.toString();
            }
            throw SimpleEndpoint.noPathTemplateFound("path");

        },

        // Request parameters
        request -> {
            Map<String, String> params = new HashMap<>();
            if (request.pipeline != null) {
                params.put("pipeline", request.pipeline);
            }
            if (request.routing != null) {
                params.put("routing", request.routing);
            }
            if (request.versionType != null) {
                params.put("version_type", request.versionType.jsonValue());
            }
            if (request.refresh != null) {
                params.put("refresh", request.refresh.jsonValue());
            }
            if (request.waitForActiveShards != null) {
                params.put("wait_for_active_shards", request.waitForActiveShards._toJsonString());
            }
            if (request.version != null) {
                params.put("version", String.valueOf(request.version));
            }
            if (request.timeout != null) {
                params.put("timeout", request.timeout._toJsonString());
            }
            return params;

        },
        SimpleEndpoint.emptyMap(),
        true,
        CreateResponse._DESERIALIZER
    );
}
