/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.client.opensearch.snapshot;

import jakarta.json.stream.JsonGenerator;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;
import javax.annotation.Nullable;
import org.opensearch.client.json.JsonpDeserializable;
import org.opensearch.client.json.JsonpDeserializer;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.ObjectBuilderDeserializer;
import org.opensearch.client.json.ObjectDeserializer;
import org.opensearch.client.json.PlainJsonSerializable;
import org.opensearch.client.opensearch._types.ErrorResponse;
import org.opensearch.client.opensearch._types.RequestBase;
import org.opensearch.client.opensearch._types.Time;
import org.opensearch.client.transport.Endpoint;
import org.opensearch.client.transport.endpoints.SimpleEndpoint;
import org.opensearch.client.util.ApiTypeHelper;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.client.util.ObjectBuilderBase;

// typedef: snapshot.clone.Request

/**
 * Clones indices from one snapshot into another snapshot in the same
 * repository.
 */
@JsonpDeserializable
public class CloneSnapshotRequest extends RequestBase implements PlainJsonSerializable {
    private final String indices;

    @Deprecated
    @Nullable
    private final Time masterTimeout;

    @Nullable
    private final Time clusterManagerTimeout;

    private final String repository;

    private final String snapshot;

    private final String targetSnapshot;

    @Nullable
    private final Time timeout;

    // ---------------------------------------------------------------------------------------------

    private CloneSnapshotRequest(Builder builder) {

        this.indices = ApiTypeHelper.requireNonNull(builder.indices, this, "indices");
        this.masterTimeout = builder.masterTimeout;
        this.clusterManagerTimeout = builder.clusterManagerTimeout;
        this.repository = ApiTypeHelper.requireNonNull(builder.repository, this, "repository");
        this.snapshot = ApiTypeHelper.requireNonNull(builder.snapshot, this, "snapshot");
        this.targetSnapshot = ApiTypeHelper.requireNonNull(builder.targetSnapshot, this, "targetSnapshot");
        this.timeout = builder.timeout;

    }

    public static CloneSnapshotRequest of(Function<Builder, ObjectBuilder<CloneSnapshotRequest>> fn) {
        return fn.apply(new Builder()).build();
    }

    /**
     * Required - API name: {@code indices}
     */
    public final String indices() {
        return this.indices;
    }

    /**
     * Explicit operation timeout for connection to master node
     * <p>
     * API name: {@code master_timeout}
     */
    @Nullable
    public final Time masterTimeout() {
        return this.masterTimeout;
    }

    /**
     * Explicit operation timeout for connection to cluster-manager node
     * <p>
     * API name: {@code cluster_manager_timeout}
     */
    @Nullable
    public final Time clusterManagerTimeout() {
        return this.clusterManagerTimeout;
    }

    /**
     * Required - A repository name
     * <p>
     * API name: {@code repository}
     */
    public final String repository() {
        return this.repository;
    }

    /**
     * Required - The name of the snapshot to clone from
     * <p>
     * API name: {@code snapshot}
     */
    public final String snapshot() {
        return this.snapshot;
    }

    /**
     * Required - The name of the cloned snapshot to create
     * <p>
     * API name: {@code target_snapshot}
     */
    public final String targetSnapshot() {
        return this.targetSnapshot;
    }

    /**
     * API name: {@code timeout}
     */
    @Nullable
    public final Time timeout() {
        return this.timeout;
    }

    /**
     * Serialize this object to JSON.
     */
    public void serialize(JsonGenerator generator, JsonpMapper mapper) {
        generator.writeStartObject();
        serializeInternal(generator, mapper);
        generator.writeEnd();
    }

    protected void serializeInternal(JsonGenerator generator, JsonpMapper mapper) {

        generator.writeKey("indices");
        generator.write(this.indices);

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Builder for {@link CloneSnapshotRequest}.
     */

    public static class Builder extends ObjectBuilderBase implements ObjectBuilder<CloneSnapshotRequest> {
        private String indices;

        @Deprecated
        @Nullable
        private Time masterTimeout;

        @Nullable
        private Time clusterManagerTimeout;

        private String repository;

        private String snapshot;

        private String targetSnapshot;

        @Nullable
        private Time timeout;

        /**
         * Required - API name: {@code indices}
         */
        public final Builder indices(String value) {
            this.indices = value;
            return this;
        }

        /**
         * Explicit operation timeout for connection to master node
         * <p>
         * API name: {@code master_timeout}
         */
        @Deprecated
        public final Builder masterTimeout(@Nullable Time value) {
            this.masterTimeout = value;
            return this;
        }

        /**
         * Explicit operation timeout for connection to master node
         * <p>
         * API name: {@code master_timeout}
         */
        @Deprecated
        public final Builder masterTimeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.masterTimeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * Explicit operation timeout for connection to cluster-manager node
         * <p>
         * API name: {@code cluster_manager_timeout}
         */
        public final Builder clusterManagerTimeout(@Nullable Time value) {
            this.clusterManagerTimeout = value;
            return this;
        }

        /**
         * Explicit operation timeout for connection to cluster-manager node
         * <p>
         * API name: {@code cluster_manager_timeout}
         */
        public final Builder clusterManagerTimeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.clusterManagerTimeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * Required - A repository name
         * <p>
         * API name: {@code repository}
         */
        public final Builder repository(String value) {
            this.repository = value;
            return this;
        }

        /**
         * Required - The name of the snapshot to clone from
         * <p>
         * API name: {@code snapshot}
         */
        public final Builder snapshot(String value) {
            this.snapshot = value;
            return this;
        }

        /**
         * Required - The name of the cloned snapshot to create
         * <p>
         * API name: {@code target_snapshot}
         */
        public final Builder targetSnapshot(String value) {
            this.targetSnapshot = value;
            return this;
        }

        /**
         * API name: {@code timeout}
         */
        public final Builder timeout(@Nullable Time value) {
            this.timeout = value;
            return this;
        }

        /**
         * API name: {@code timeout}
         */
        public final Builder timeout(Function<Time.Builder, ObjectBuilder<Time>> fn) {
            return this.timeout(fn.apply(new Time.Builder()).build());
        }

        /**
         * Builds a {@link CloneSnapshotRequest}.
         *
         * @throws NullPointerException
         *             if some of the required fields are null.
         */
        public CloneSnapshotRequest build() {
            _checkSingleUse();

            return new CloneSnapshotRequest(this);
        }
    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Json deserializer for {@link CloneSnapshotRequest}
     */
    public static final JsonpDeserializer<CloneSnapshotRequest> _DESERIALIZER = ObjectBuilderDeserializer.lazy(
        Builder::new,
        CloneSnapshotRequest::setupCloneSnapshotRequestDeserializer
    );

    protected static void setupCloneSnapshotRequestDeserializer(ObjectDeserializer<CloneSnapshotRequest.Builder> op) {

        op.add(Builder::indices, JsonpDeserializer.stringDeserializer(), "indices");

    }

    // ---------------------------------------------------------------------------------------------

    /**
     * Endpoint "{@code snapshot.clone}".
     */
    public static final Endpoint<CloneSnapshotRequest, CloneSnapshotResponse, ErrorResponse> _ENDPOINT = new SimpleEndpoint<>(

        // Request method
        request -> {
            return "PUT";

        },

        // Request path
        request -> {
            final int _targetSnapshot = 1 << 0;
            final int _repository = 1 << 1;
            final int _snapshot = 1 << 2;

            int propsSet = 0;

            propsSet |= _targetSnapshot;
            propsSet |= _repository;
            propsSet |= _snapshot;

            if (propsSet == (_repository | _snapshot | _targetSnapshot)) {
                StringBuilder buf = new StringBuilder();
                buf.append("/_snapshot");
                buf.append("/");
                SimpleEndpoint.pathEncode(request.repository, buf);
                buf.append("/");
                SimpleEndpoint.pathEncode(request.snapshot, buf);
                buf.append("/_clone");
                buf.append("/");
                SimpleEndpoint.pathEncode(request.targetSnapshot, buf);
                return buf.toString();
            }
            throw SimpleEndpoint.noPathTemplateFound("path");

        },

        // Request parameters
        request -> {
            Map<String, String> params = new HashMap<>();
            if (request.masterTimeout != null) {
                params.put("master_timeout", request.masterTimeout._toJsonString());
            }
            if (request.clusterManagerTimeout != null) {
                params.put("cluster_manager_timeout", request.clusterManagerTimeout._toJsonString());
            }
            if (request.timeout != null) {
                params.put("timeout", request.timeout._toJsonString());
            }
            return params;

        },
        SimpleEndpoint.emptyMap(),
        true,
        CloneSnapshotResponse._DESERIALIZER
    );
}
