/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: FileRegistryComponent.java 18616 2009-09-14 19:58:01Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.registry.impl.file;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.jonas.camel.registry.api.AbstractRegistryComponent;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class FileRegistryComponent extends AbstractRegistryComponent {

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(FileRegistryComponent.class);

    /**
     * The registry datas.
     */
    private Map<String, String> registry;

    /**
     * Set the registry map from the file.
     * 
     * @param file the file that owns the datas.
     * @throws Exception If anything fails.
     */
    public void setFile(final String file) throws Exception {
        Map<String, String> tmpRegistry = new HashMap<String, String>();

        try {
            if (this.registry == null) {
                this.registry = new HashMap<String, String>();
            }

            FileRegistryComponent.logger.debug("Setting file: " + file);

            DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
            Document doc;

            ClassLoader cl = this.getClass().getClassLoader();
            if (cl.getResource(file) != null) {
                FileRegistryComponent.logger.debug("Loading the registy XML file from classpath");
                InputStream resource = cl.getResourceAsStream(file);
                try {
                    doc = docBuilder.parse(resource);
                } finally {
                    resource.close();
                    resource = null;
                }
            } else {
                FileRegistryComponent.logger.debug("Registy XML not in classpath, looking up as a file");

                File f = new File(file);
                if (f.isFile()) {
                    FileRegistryComponent.logger.debug("Loading the registy XML file from file");
                    doc = docBuilder.parse(f);
                } else {
                    FileRegistryComponent.logger.debug("Registy XML not a file, assuming it's a URL");
                    doc = docBuilder.parse(file);
                }
            }
            doc.getDocumentElement().normalize();

            FileRegistryComponent.logger.debug("Reading the registy XML file");
            NodeList listOfEntries = doc.getElementsByTagName("entry");
            for (int i = 0; i < listOfEntries.getLength(); i++) {
                String name = null;
                String value = null;
                NodeList entry = listOfEntries.item(i).getChildNodes();
                for (int j = 0; j < entry.getLength(); j++) {
                    Node entryDetail = entry.item(j);
                    if ("logicalName".equals(entryDetail.getNodeName())) {
                        name = this.getValueAndRemoveAllWhitespace(entryDetail);
                    } else if ("technicalName".equals(entryDetail.getNodeName())) {
                        value = this.getValueAndRemoveAllWhitespace(entryDetail);
                    }
                }
                if (this.registry.containsKey(name)) {
                    throw new IllegalStateException("The entry name " + name + " is already set.");
                }
                tmpRegistry.put(name, value);
            }

            if (this.registry == null) {
                this.registry = new HashMap<String, String>();
            }
            this.registry.putAll(tmpRegistry);

            FileRegistryComponent.logger.debug("Registy XML loaded from: " + file);
        } catch (Exception e) {
            this.registry = null;
            throw e;
        }
    }

    /**
     * Add values to the registry component.
     * 
     * @param input the inputStream that contains what to add.
     */
    @Override
    public void addValues(final InputStream input) throws Exception {
        if (this.registry == null) {
            this.registry = new HashMap<String, String>();
        }
        Map<String, String> tmpRegistry = new HashMap<String, String>();

        try {
            DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
            Document doc;

            doc = docBuilder.parse(input);

            doc.getDocumentElement().normalize();

            NodeList listOfEntries = doc.getElementsByTagName("entry");
            for (int i = 0; i < listOfEntries.getLength(); i++) {
                String name = null;
                String value = null;
                NodeList entry = listOfEntries.item(i).getChildNodes();
                for (int j = 0; j < entry.getLength(); j++) {
                    Node entryDetail = entry.item(j);
                    if ("logicalName".equals(entryDetail.getNodeName())) {
                        name = this.getValueAndRemoveAllWhitespace(entryDetail);
                    } else if ("technicalName".equals(entryDetail.getNodeName())) {
                        value = this.getValueAndRemoveAllWhitespace(entryDetail);
                    }
                }
                if (this.registry.containsKey(name)) {
                    throw new IllegalStateException("The entry name " + name + " is already set.");
                }
                tmpRegistry.put(name, value);
            }
        } finally {
            try {
                input.close();
            } catch (IOException e) {
                FileRegistryComponent.logger.error("Error while closing the inputStream", e);
            }
        }

        this.registry.putAll(tmpRegistry);
    }

    /**
     * Remove values to the registry component.
     * 
     * @param input the inputStream that contains what to add.
     */
    @Override
    public void removeValues(final InputStream input) {
        try {
            if (this.registry == null) {
                this.registry = new HashMap<String, String>();
            }

            DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
            Document doc;

            doc = docBuilder.parse(input);

            doc.getDocumentElement().normalize();

            NodeList listOfEntries = doc.getElementsByTagName("entry");
            for (int i = 0; i < listOfEntries.getLength(); i++) {
                String name = null;
                NodeList entry = listOfEntries.item(i).getChildNodes();
                for (int j = 0; j < entry.getLength(); j++) {
                    Node entryDetail = entry.item(j);
                    if ("logicalName".equals(entryDetail.getNodeName())) {
                        name = this.getValueAndRemoveAllWhitespace(entryDetail);
                    }
                }
                if (this.registry.containsKey(name)) {
                    this.registry.remove(name);
                } else {
                    FileRegistryComponent.logger.warn("The entry {0} does not exist in the registry.");
                }

            }

        } catch (Exception e) {
            throw new IllegalArgumentException("Error while unregistering the entries.", e);
        } finally {
            try {
                input.close();
            } catch (IOException e) {
                FileRegistryComponent.logger.error("Error while closing the inputStream", e);
            }
        }
    }

    private String getValueAndRemoveAllWhitespace(final Node node) {
        StringBuilder result = new StringBuilder();
        NodeList children = node.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            result.append(children.item(i).getNodeValue());
        }
        String value = result.toString();
        value = value.replace('\r', '\n');
        result.setLength(0);
        for (String line : value.split("\n")) {
            result.append(line.trim());
        }
        return result.toString();
    }

    /**
     * Look up a definition from the registry.
     * 
     * @param name the name of the definition to lookup.
     * @return the technical definition.
     * @throws Exception If anything fails.
     */
    @Override
    protected String lookup(final String name) throws Exception {
        if (this.registry == null) {
            throw new IllegalStateException("FileRegistryComponent doesn't have a valid file specified");
        }

        return this.registry.get(name);
    }
}
