/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: FileRegistryTest.java 18616 2009-09-14 19:58:01Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.registry.impl.test.file.mock;

import java.io.InputStream;
import java.net.URL;

import org.junit.Assert;
import org.junit.Test;

/**
 * Tests the file registry component. No OSGi environment here.
 * 
 * @author Guillaume Renault
 */
public class FileRegistryTest {

    /**
     * Test the fileRegistry component with a file path.
     */
    @Test
    public void fileRegistryAsFilePathTest() throws Exception {
        FileRegistryComponentMock fileRegistryComponent = new FileRegistryComponentMock();

        URL url = Thread.currentThread().getContextClassLoader().getResource("registry1.xml");

        fileRegistryComponent.setFile(url.getPath());

        Assert.assertEquals("Check that the registered entry is the same as expected", "test-jms:queue:sampleQueue",
            fileRegistryComponent.getEntry("queue"));
    }

    /**
     * Test the fileRegistry component with an {@link InputStream}.
     */
    @Test
    public void fileRegistryAsInputStreamTest() throws Exception {
        FileRegistryComponentMock fileRegistryComponent = new FileRegistryComponentMock();

        InputStream inputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream("registry1.xml");

        fileRegistryComponent.addValues(inputStream);

        Assert.assertEquals("Check that the registered entry is the same as expected", "test-jms:queue:sampleQueue",
            fileRegistryComponent.getEntry("queue"));
    }

    /**
     * Test the fileRegistry component. Add twice the same entries. An exception
     * must be thrown.
     */
    @Test
    public void addTwiceTheSameEntriesTest() throws Exception {
        FileRegistryComponentMock fileRegistryComponent = new FileRegistryComponentMock();

        InputStream inputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream("registry1.xml");

        fileRegistryComponent.addValues(inputStream);

        Assert.assertEquals("Check that the registered entry is the same as expected", "test-jms:queue:sampleQueue",
            fileRegistryComponent.getEntry("queue"));

        inputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream("registry1.xml");
        try {
            fileRegistryComponent.addValues(inputStream);
            Assert.fail("An exception must be thrown, as the same entries cannot be set up twice.");
        } catch (IllegalStateException e) {
            Assert
                .assertEquals("Assert that the entry already exists.", "The entry name queue is already set.", e.getMessage());
        }

    }

    /**
     * Test to remove entries from the fileRegistry component with an
     * {@link InputStream}.
     */
    @Test
    public void RemoveEntriesFromFileRegistryAsInputStreamTest() throws Exception {
        FileRegistryComponentMock fileRegistryComponent = new FileRegistryComponentMock();

        // add the first file.
        InputStream inputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream("registry1.xml");

        fileRegistryComponent.addValues(inputStream);

        Assert.assertEquals("Check that the registered entry is the same as expected", "test-jms:queue:sampleQueue",
            fileRegistryComponent.getEntry("queue"));

        Assert.assertEquals("Check that the registered entry is the same as expected", "test-jms:queue:sampleQueue2",
            fileRegistryComponent.getEntry("queue2"));

        Assert.assertEquals("Check that the registered entry is the same as expected",
            "cxf://url?parameter-separated-with-tab", fileRegistryComponent.getEntry("multiline"));

        Assert.assertEquals("Check that the registered entry is the same as expected",
            "file://C:/Documents and Settings/My User", fileRegistryComponent.getEntry("multiline-with-spaces"));

        // Add the second file.
        inputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream("registry2.xml");

        fileRegistryComponent.addValues(inputStream);

        Assert.assertEquals("Check that the registered entry is the same as expected", "test-jms:queue:sampleQueue3",
            fileRegistryComponent.getEntry("queue3"));

        // Remove the first file.
        inputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream("registry1.xml");

        fileRegistryComponent.removeValues(inputStream);

        // If the entry is removed, the map returns null. Both the entries of
        // the first file must be unavailable.
        Assert.assertNull("If the entry is removed, the map returns null", fileRegistryComponent.getEntry("queue"));
        Assert.assertNull("If the entry is removed, the map returns null", fileRegistryComponent.getEntry("queue2"));

        // The entry from the second file must still be available.
        Assert.assertEquals("Check that the registered entry is the same as expected", "test-jms:queue:sampleQueue3",
            fileRegistryComponent.getEntry("queue3"));
    }
}
