/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ExampleCXF.java 18287 2009-08-17 19:35:02Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.example.cxf.route;

import java.io.InputStream;

import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.cxf.CxfComponent;
import org.apache.cxf.frontend.ClientProxyFactoryBean;
import org.apache.cxf.message.MessageContentsList;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.service.http.HttpService;
import org.ow2.jonas.camel.example.cxf.webservice.api.ISayHello;
import org.ow2.jonas.camel.service.api.ICamelService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * A simple example on how to use the camel service on JOnAS 5.
 * 
 * @author Guillaume Renault
 */
public class ExampleCXF {

    /**
     * The logger.
     */
    private Log logger = LogFactory.getLog(this.getClass());

    /**
     * The camel service. Injected by the container.
     */
    private ICamelService camelService = null;

    /**
     * The camel context name of the example.
     */
    private String camelContextName = null;

    /**
     * The OSGi platform's bundle context.
     */
    private BundleContext bundleContext;

    /**
     * Constructor: saves the BundleContext.
     */
    public ExampleCXF(final BundleContext bundleContext) {
        this.bundleContext = bundleContext;
    }

    /**
     * Start the route.
     */
    public void start() throws Throwable {
        final ClassLoader bundleCL = this.getClass().getClassLoader();
        final ClassLoader oldCL = Thread.currentThread().getContextClassLoader();

        try {
            // Route definition
            RouteBuilder builder = new RouteBuilder() {
                @Override
                public void configure() throws Exception {
                    this.from("registry:cxfToFile").process(new Processor() {
                        public void process(final Exchange exchange) throws Exception {
                            MessageContentsList msgList = exchange.getIn().getBody(MessageContentsList.class);
                            String name = (String) msgList.get(0);
                            ExampleCXF.this.logger.info("Received CXF message {0}", name);

                            MessageContentsList response = new MessageContentsList();
                            response.add("hello, " + name);
                            exchange.getOut().setBody(response);
                        }
                    });
                }
            };

            // Start a new Camel context for the route
            this.camelContextName = this.camelService.startNewContext();
            this.logger.info("Camel context {0} started", camelContextName);

            // Add entries to the registry
            InputStream input = bundleCL.getResourceAsStream("cxf-registry.xml");
            this.camelService.addRegistry(input, this.camelContextName);

            // Add the CXF component
            this.camelService.addComponent("cxf", new CxfComponent(), this.camelContextName);

            try {
                // Change the TCCL since CXF will be loading classes
                // dynamically and cannot use the bundle CL for this.
                //
                // This is needed for both creating the Camel-CXF route
                // and using the ClientProxyFactoryBean-based client.
                Thread.currentThread().setContextClassLoader(bundleCL);

                // Add the route in the camel context.
                this.camelService.addRoutes(builder, this.camelContextName);

                this.test();
            } finally {
                Thread.currentThread().setContextClassLoader(oldCL);
            }
        } catch (Throwable t) {
            this.logger.error("Cannot start ExampleCXF", t);
            throw t;
        }
    }

    /**
     * Test the route.
     */
    public void test() {
        ServiceReference httpService = this.bundleContext.getServiceReference(HttpService.class.getName());
        if (httpService == null) {
            throw new NullPointerException("HttpService is not registered");
        }
        final String port = (String) httpService.getProperty("org.osgi.service.http.port");

        ClientProxyFactoryBean factory = new ClientProxyFactoryBean();
        factory.setServiceClass(ISayHello.class);
        factory.setAddress("http://localhost:" + port + "/services/SayHello");
        ISayHello client = (ISayHello) factory.create();

        String response = client.hello("guillaume");
        this.logger.info("Got CXF response {0}", response);
    }

    /**
     * Stop the route.
     */
    public void stop() throws Exception {
        try {
            this.camelService.stop(this.camelContextName);
        } finally {
            this.camelContextName = null;
            this.camelService = null;
        }
    }

}
