/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.boot.ssl.pem;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.security.AlgorithmParameters;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.HashMap;
import java.util.HexFormat;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.crypto.Cipher;
import javax.crypto.EncryptedPrivateKeyInfo;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import org.springframework.util.Assert;

final class PemPrivateKeyParser {
    private static final String PKCS1_RSA_HEADER = "-+BEGIN\\s+RSA\\s+PRIVATE\\s+KEY[^-]*-+(?:\\s|\\r|\\n)+";
    private static final String PKCS1_RSA_FOOTER = "-+END\\s+RSA\\s+PRIVATE\\s+KEY[^-]*-+";
    private static final String PKCS8_HEADER = "-+BEGIN\\s+PRIVATE\\s+KEY[^-]*-+(?:\\s|\\r|\\n)+";
    private static final String PKCS8_FOOTER = "-+END\\s+PRIVATE\\s+KEY[^-]*-+";
    private static final String PKCS8_ENCRYPTED_HEADER = "-+BEGIN\\s+ENCRYPTED\\s+PRIVATE\\s+KEY[^-]*-+(?:\\s|\\r|\\n)+";
    private static final String PKCS8_ENCRYPTED_FOOTER = "-+END\\s+ENCRYPTED\\s+PRIVATE\\s+KEY[^-]*-+";
    private static final String SEC1_EC_HEADER = "-+BEGIN\\s+EC\\s+PRIVATE\\s+KEY[^-]*-+(?:\\s|\\r|\\n)+";
    private static final String SEC1_EC_FOOTER = "-+END\\s+EC\\s+PRIVATE\\s+KEY[^-]*-+";
    private static final String BASE64_TEXT = "([a-z0-9+/=\\r\\n]+)";
    public static final int BASE64_TEXT_GROUP = 1;
    private static final EncodedOid RSA_ALGORITHM = EncodedOid.OID_1_2_840_113549_1_1_1;
    private static final EncodedOid ELLIPTIC_CURVE_ALGORITHM = EncodedOid.OID_1_2_840_10045_2_1;
    private static final EncodedOid ELLIPTIC_CURVE_384_BIT = EncodedOid.OID_1_3_132_0_34;
    private static final Map<EncodedOid, String> ALGORITHMS;
    private static final List<PemParser> PEM_PARSERS;

    private PemPrivateKeyParser() {
    }

    private static PKCS8EncodedKeySpec createKeySpecForPkcs1Rsa(byte[] bytes, String password) {
        return PemPrivateKeyParser.createKeySpecForAlgorithm(bytes, RSA_ALGORITHM, null);
    }

    private static PKCS8EncodedKeySpec createKeySpecForSec1Ec(byte[] bytes, String password) {
        DerElement ecPrivateKey = DerElement.of(bytes);
        Assert.state((boolean)ecPrivateKey.isType(DerElement.ValueType.ENCODED, DerElement.TagType.SEQUENCE), (String)"Key spec should be an ASN.1 encoded sequence");
        DerElement version = DerElement.of(ecPrivateKey.getContents());
        Assert.state((version != null && version.isType(DerElement.ValueType.PRIMITIVE, DerElement.TagType.INTEGER) ? 1 : 0) != 0, (String)"Key spec should start with version");
        Assert.state((version.getContents().remaining() == 1 && version.getContents().get() == 1 ? 1 : 0) != 0, (String)"Key spec version must be 1");
        DerElement privateKey = DerElement.of(ecPrivateKey.getContents());
        Assert.state((privateKey != null && privateKey.isType(DerElement.ValueType.PRIMITIVE, DerElement.TagType.OCTET_STRING) ? 1 : 0) != 0, (String)"Key spec should contain private key");
        DerElement parameters = DerElement.of(ecPrivateKey.getContents());
        return PemPrivateKeyParser.createKeySpecForAlgorithm(bytes, ELLIPTIC_CURVE_ALGORITHM, PemPrivateKeyParser.getEcParameters(parameters));
    }

    private static EncodedOid getEcParameters(DerElement parameters) {
        if (parameters == null) {
            return ELLIPTIC_CURVE_384_BIT;
        }
        Assert.state((boolean)parameters.isType(DerElement.ValueType.ENCODED), (String)"Key spec should contain encoded parameters");
        DerElement contents = DerElement.of(parameters.getContents());
        Assert.state((contents != null && contents.isType(DerElement.ValueType.PRIMITIVE, DerElement.TagType.OBJECT_IDENTIFIER) ? 1 : 0) != 0, (String)"Key spec parameters should contain object identifier");
        return EncodedOid.of(contents);
    }

    private static PKCS8EncodedKeySpec createKeySpecForAlgorithm(byte[] bytes, EncodedOid algorithm, EncodedOid parameters) {
        try {
            DerEncoder encoder = new DerEncoder();
            encoder.integer(0);
            DerEncoder algorithmIdentifier = new DerEncoder();
            algorithmIdentifier.objectIdentifier(algorithm);
            algorithmIdentifier.objectIdentifier(parameters);
            encoder.sequence(algorithmIdentifier.toByteArray());
            encoder.octetString(bytes);
            return new PKCS8EncodedKeySpec(encoder.toSequence());
        }
        catch (IOException ex) {
            throw new IllegalStateException(ex);
        }
    }

    private static PKCS8EncodedKeySpec createKeySpecForPkcs8(byte[] bytes, String password) {
        DerElement ecPrivateKey = DerElement.of(bytes);
        Assert.state((boolean)ecPrivateKey.isType(DerElement.ValueType.ENCODED, DerElement.TagType.SEQUENCE), (String)"Key spec should be an ASN.1 encoded sequence");
        DerElement version = DerElement.of(ecPrivateKey.getContents());
        Assert.state((version != null && version.isType(DerElement.ValueType.PRIMITIVE, DerElement.TagType.INTEGER) ? 1 : 0) != 0, (String)"Key spec should start with version");
        DerElement sequence = DerElement.of(ecPrivateKey.getContents());
        Assert.state((sequence != null && sequence.isType(DerElement.ValueType.ENCODED, DerElement.TagType.SEQUENCE) ? 1 : 0) != 0, (String)"Key spec should contain private key");
        DerElement algorithmId = DerElement.of(sequence.getContents());
        Assert.state((algorithmId != null && algorithmId.isType(DerElement.ValueType.PRIMITIVE, DerElement.TagType.OBJECT_IDENTIFIER) ? 1 : 0) != 0, (String)"Key spec container object identifier");
        String algorithmName = ALGORITHMS.get(EncodedOid.of(algorithmId));
        return algorithmName != null ? new PKCS8EncodedKeySpec(bytes, algorithmName) : new PKCS8EncodedKeySpec(bytes);
    }

    private static PKCS8EncodedKeySpec createKeySpecForPkcs8Encrypted(byte[] bytes, String password) {
        return Pkcs8PrivateKeyDecryptor.decrypt(bytes, password);
    }

    static PrivateKey parse(String text) {
        return PemPrivateKeyParser.parse(text, null);
    }

    static PrivateKey parse(String text, String password) {
        if (text == null) {
            return null;
        }
        try {
            for (PemParser pemParser : PEM_PARSERS) {
                PrivateKey privateKey = pemParser.parse(text, password);
                if (privateKey == null) continue;
                return privateKey;
            }
        }
        catch (Exception ex) {
            throw new IllegalStateException("Error loading private key file: " + ex.getMessage(), ex);
        }
        throw new IllegalStateException("Missing private key or unrecognized format");
    }

    static {
        HashMap<EncodedOid, String> algorithms = new HashMap<EncodedOid, String>();
        algorithms.put(EncodedOid.OID_1_2_840_113549_1_1_1, "RSA");
        algorithms.put(EncodedOid.OID_1_2_840_113549_1_1_10, "RSA");
        algorithms.put(EncodedOid.OID_1_2_840_10040_4_1, "DSA");
        algorithms.put(EncodedOid.OID_1_3_101_110, "XDH");
        algorithms.put(EncodedOid.OID_1_3_101_111, "XDH");
        algorithms.put(EncodedOid.OID_1_3_101_112, "EdDSA");
        algorithms.put(EncodedOid.OID_1_3_101_113, "EdDSA");
        algorithms.put(EncodedOid.OID_1_2_840_10045_2_1, "EC");
        ALGORITHMS = Collections.unmodifiableMap(algorithms);
        ArrayList<PemParser> parsers = new ArrayList<PemParser>();
        parsers.add(new PemParser(PKCS1_RSA_HEADER, PKCS1_RSA_FOOTER, PemPrivateKeyParser::createKeySpecForPkcs1Rsa, "RSA"));
        parsers.add(new PemParser(SEC1_EC_HEADER, SEC1_EC_FOOTER, PemPrivateKeyParser::createKeySpecForSec1Ec, "EC"));
        parsers.add(new PemParser(PKCS8_HEADER, PKCS8_FOOTER, PemPrivateKeyParser::createKeySpecForPkcs8, "RSA", "RSASSA-PSS", "EC", "DSA", "EdDSA", "XDH"));
        parsers.add(new PemParser(PKCS8_ENCRYPTED_HEADER, PKCS8_ENCRYPTED_FOOTER, PemPrivateKeyParser::createKeySpecForPkcs8Encrypted, "RSA", "RSASSA-PSS", "EC", "DSA", "EdDSA", "XDH"));
        PEM_PARSERS = Collections.unmodifiableList(parsers);
    }

    static final class EncodedOid {
        static final EncodedOid OID_1_2_840_10040_4_1 = EncodedOid.of("2a8648ce380401");
        static final EncodedOid OID_1_2_840_113549_1_1_1 = EncodedOid.of("2A864886F70D010101");
        static final EncodedOid OID_1_2_840_113549_1_1_10 = EncodedOid.of("2a864886f70d01010a");
        static final EncodedOid OID_1_3_101_110 = EncodedOid.of("2b656e");
        static final EncodedOid OID_1_3_101_111 = EncodedOid.of("2b656f");
        static final EncodedOid OID_1_3_101_112 = EncodedOid.of("2b6570");
        static final EncodedOid OID_1_3_101_113 = EncodedOid.of("2b6571");
        static final EncodedOid OID_1_2_840_10045_2_1 = EncodedOid.of("2a8648ce3d0201");
        static final EncodedOid OID_1_3_132_0_34 = EncodedOid.of("2b81040022");
        private final byte[] value;

        private EncodedOid(byte[] value) {
            this.value = value;
        }

        byte[] toByteArray() {
            return (byte[])this.value.clone();
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null || this.getClass() != obj.getClass()) {
                return false;
            }
            return Arrays.equals(this.value, ((EncodedOid)obj).value);
        }

        public int hashCode() {
            return Arrays.hashCode(this.value);
        }

        static EncodedOid of(String hexString) {
            return EncodedOid.of(HexFormat.of().parseHex(hexString));
        }

        static EncodedOid of(DerElement derElement) {
            return EncodedOid.of(derElement.getContents());
        }

        static EncodedOid of(ByteBuffer byteBuffer) {
            return EncodedOid.of(byteBuffer.array(), byteBuffer.arrayOffset() + byteBuffer.position(), byteBuffer.remaining());
        }

        static EncodedOid of(byte[] bytes) {
            return EncodedOid.of(bytes, 0, bytes.length);
        }

        static EncodedOid of(byte[] bytes, int off, int len) {
            byte[] value = new byte[len];
            System.arraycopy(bytes, off, value, 0, len);
            return new EncodedOid(value);
        }
    }

    static final class DerElement {
        private final ValueType valueType;
        private final long tagType;
        private final ByteBuffer contents;

        private DerElement(ByteBuffer bytes) {
            byte b = bytes.get();
            this.valueType = (b & 0x20) == 0 ? ValueType.PRIMITIVE : ValueType.ENCODED;
            this.tagType = this.decodeTagType(b, bytes);
            int length = this.decodeLength(bytes);
            bytes.limit(bytes.position() + length);
            this.contents = bytes.slice();
            bytes.limit(bytes.capacity());
            bytes.position(bytes.position() + length);
        }

        private long decodeTagType(byte b, ByteBuffer bytes) {
            long tagType = b & 0x1F;
            if (tagType != 31L) {
                return tagType;
            }
            tagType = 0L;
            b = bytes.get();
            while ((b & 0x80) != 0) {
                tagType <<= 7;
                tagType |= (long)(b & 0x7F);
                b = bytes.get();
            }
            return tagType;
        }

        private int decodeLength(ByteBuffer bytes) {
            byte b = bytes.get();
            if ((b & 0x80) == 0) {
                return b & 0x7F;
            }
            int numberOfLengthBytes = b & 0x7F;
            Assert.state((numberOfLengthBytes != 0 ? 1 : 0) != 0, (String)"Infinite length encoding is not supported");
            Assert.state((numberOfLengthBytes != 127 ? 1 : 0) != 0, (String)"Reserved length encoding is not supported");
            Assert.state((numberOfLengthBytes <= 4 ? 1 : 0) != 0, (String)"Length overflow");
            int length = 0;
            for (int i = 0; i < numberOfLengthBytes; ++i) {
                length <<= 8;
                length |= bytes.get() & 0xFF;
            }
            return length;
        }

        boolean isType(ValueType valueType) {
            return this.valueType == valueType;
        }

        boolean isType(ValueType valueType, TagType tagType) {
            return this.valueType == valueType && this.tagType == (long)tagType.getNumber();
        }

        ByteBuffer getContents() {
            return this.contents;
        }

        static DerElement of(byte[] bytes) {
            return DerElement.of(ByteBuffer.wrap(bytes));
        }

        static DerElement of(ByteBuffer bytes) {
            return bytes.remaining() > 0 ? new DerElement(bytes) : null;
        }

        static enum ValueType {
            PRIMITIVE,
            ENCODED;

        }

        static enum TagType {
            INTEGER(2),
            OCTET_STRING(4),
            OBJECT_IDENTIFIER(6),
            SEQUENCE(16);

            private final int number;

            private TagType(int number) {
                this.number = number;
            }

            int getNumber() {
                return this.number;
            }
        }
    }

    static class DerEncoder {
        private final ByteArrayOutputStream stream = new ByteArrayOutputStream();

        DerEncoder() {
        }

        void objectIdentifier(EncodedOid encodedOid) throws IOException {
            int code = encodedOid != null ? 6 : 5;
            this.codeLengthBytes(code, encodedOid != null ? encodedOid.toByteArray() : null);
        }

        void integer(int ... encodedInteger) throws IOException {
            this.codeLengthBytes(2, DerEncoder.bytes(encodedInteger));
        }

        void octetString(byte[] bytes) throws IOException {
            this.codeLengthBytes(4, bytes);
        }

        void sequence(byte[] bytes) throws IOException {
            this.codeLengthBytes(48, bytes);
        }

        void codeLengthBytes(int code, byte[] bytes) throws IOException {
            int length;
            this.stream.write(code);
            int n = length = bytes != null ? bytes.length : 0;
            if (length <= 127) {
                this.stream.write(length & 0xFF);
            } else {
                ByteArrayOutputStream lengthStream = new ByteArrayOutputStream();
                while (length != 0) {
                    lengthStream.write(length & 0xFF);
                    length >>= 8;
                }
                byte[] lengthBytes = lengthStream.toByteArray();
                this.stream.write(0x80 | lengthBytes.length);
                for (int i = lengthBytes.length - 1; i >= 0; --i) {
                    this.stream.write(lengthBytes[i]);
                }
            }
            if (bytes != null) {
                this.stream.write(bytes);
            }
        }

        private static byte[] bytes(int ... elements) {
            if (elements == null) {
                return null;
            }
            byte[] result = new byte[elements.length];
            for (int i = 0; i < elements.length; ++i) {
                result[i] = (byte)elements[i];
            }
            return result;
        }

        byte[] toSequence() throws IOException {
            DerEncoder sequenceEncoder = new DerEncoder();
            sequenceEncoder.sequence(this.toByteArray());
            return sequenceEncoder.toByteArray();
        }

        byte[] toByteArray() {
            return this.stream.toByteArray();
        }
    }

    static class Pkcs8PrivateKeyDecryptor {
        public static final String PBES2_ALGORITHM = "PBES2";

        Pkcs8PrivateKeyDecryptor() {
        }

        static PKCS8EncodedKeySpec decrypt(byte[] bytes, String password) {
            Assert.state((password != null ? 1 : 0) != 0, (String)"Password is required for an encrypted private key");
            try {
                EncryptedPrivateKeyInfo keyInfo = new EncryptedPrivateKeyInfo(bytes);
                AlgorithmParameters algorithmParameters = keyInfo.getAlgParameters();
                String encryptionAlgorithm = Pkcs8PrivateKeyDecryptor.getEncryptionAlgorithm(algorithmParameters, keyInfo.getAlgName());
                SecretKeyFactory keyFactory = SecretKeyFactory.getInstance(encryptionAlgorithm);
                SecretKey key = keyFactory.generateSecret(new PBEKeySpec(password.toCharArray()));
                Cipher cipher = Cipher.getInstance(encryptionAlgorithm);
                cipher.init(2, (Key)key, algorithmParameters);
                return keyInfo.getKeySpec(cipher);
            }
            catch (IOException | GeneralSecurityException ex) {
                throw new IllegalArgumentException("Error decrypting private key", ex);
            }
        }

        private static String getEncryptionAlgorithm(AlgorithmParameters algParameters, String algName) {
            if (algParameters != null && PBES2_ALGORITHM.equals(algName)) {
                return algParameters.toString();
            }
            return algName;
        }
    }

    private static class PemParser {
        private final Pattern pattern;
        private final BiFunction<byte[], String, PKCS8EncodedKeySpec> keySpecFactory;
        private final String[] algorithms;

        PemParser(String header, String footer, BiFunction<byte[], String, PKCS8EncodedKeySpec> keySpecFactory, String ... algorithms) {
            this.pattern = Pattern.compile(header + PemPrivateKeyParser.BASE64_TEXT + footer, 2);
            this.keySpecFactory = keySpecFactory;
            this.algorithms = algorithms;
        }

        PrivateKey parse(String text, String password) {
            Matcher matcher = this.pattern.matcher(text);
            return !matcher.find() ? null : this.parse(PemParser.decodeBase64(matcher.group(1)), password);
        }

        private static byte[] decodeBase64(String content) {
            byte[] contentBytes = content.replaceAll("\r", "").replaceAll("\n", "").getBytes();
            return Base64.getDecoder().decode(contentBytes);
        }

        private PrivateKey parse(byte[] bytes, String password) {
            PKCS8EncodedKeySpec keySpec = this.keySpecFactory.apply(bytes, password);
            if (keySpec.getAlgorithm() != null) {
                try {
                    KeyFactory keyFactory = KeyFactory.getInstance(keySpec.getAlgorithm());
                    return keyFactory.generatePrivate(keySpec);
                }
                catch (NoSuchAlgorithmException | InvalidKeySpecException generalSecurityException) {
                    // empty catch block
                }
            }
            for (String algorithm : this.algorithms) {
                try {
                    KeyFactory keyFactory = KeyFactory.getInstance(algorithm);
                    return keyFactory.generatePrivate(keySpec);
                }
                catch (NoSuchAlgorithmException | InvalidKeySpecException generalSecurityException) {
                }
            }
            return null;
        }
    }
}

