/*
 * Decompiled with CFR 0.152.
 */
package com.github.mizosoft.methanol.internal;

import com.github.mizosoft.methanol.internal.Validate;
import com.github.mizosoft.methanol.internal.text.HttpCharMatchers;
import java.io.Closeable;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.nio.ByteBuffer;
import java.time.Clock;
import java.time.Duration;
import java.time.ZoneOffset;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import org.checkerframework.checker.nullness.qual.Nullable;

public class Utils {
    private static final System.Logger logger = System.getLogger(Utils.class.getName());
    private static final Clock SYSTEM_MILLIS_UTC = Clock.tickMillis(ZoneOffset.UTC);

    private Utils() {
    }

    public static boolean isValidToken(String token) {
        return !token.isEmpty() && HttpCharMatchers.TOKEN_MATCHER.allMatch(token);
    }

    public static void validateToken(String token) {
        Validate.requireArgument(Utils.isValidToken(token), "illegal token: '%s'", token);
    }

    public static void validateHeaderName(String name) {
        Validate.requireArgument(Utils.isValidToken(name), "illegal header name: '%s'", name);
    }

    public static void validateHeaderValue(String value) {
        Validate.requireArgument(HttpCharMatchers.FIELD_VALUE_MATCHER.allMatch(value), "illegal header value: '%s'", value);
    }

    public static void validateHeader(String name, String value) {
        Utils.validateHeaderName(name);
        Utils.validateHeaderValue(value);
    }

    public static void requirePositiveDuration(Duration duration) {
        Validate.requireArgument(!duration.isNegative() && !duration.isZero(), "non-positive duration: %s", duration);
    }

    public static void requireNonNegativeDuration(Duration duration) {
        Validate.requireArgument(!duration.isNegative(), "negative duration: %s", duration);
    }

    public static int copyRemaining(ByteBuffer src, ByteBuffer dst) {
        int toCopy = Math.min(src.remaining(), dst.remaining());
        int srcLimit = src.limit();
        src.limit(src.position() + toCopy);
        dst.put(src);
        src.limit(srcLimit);
        return toCopy;
    }

    public static ByteBuffer copy(ByteBuffer buffer) {
        return ByteBuffer.allocate(buffer.remaining()).put(buffer).flip();
    }

    public static Clock systemMillisUtc() {
        return SYSTEM_MILLIS_UTC;
    }

    private static RuntimeException rethrowAsyncThrowable(Throwable throwable) throws IOException, InterruptedException {
        Throwable clonedThrowable = Utils.tryCloneThrowable(throwable);
        if (clonedThrowable instanceof RuntimeException) {
            throw (RuntimeException)clonedThrowable;
        }
        if (clonedThrowable instanceof Error) {
            throw (Error)clonedThrowable;
        }
        if (clonedThrowable instanceof IOException) {
            throw (IOException)clonedThrowable;
        }
        if (clonedThrowable instanceof InterruptedException) {
            throw (InterruptedException)clonedThrowable;
        }
        throw new IOException(throwable);
    }

    private static @Nullable Throwable tryCloneThrowable(Throwable t) {
        Throwable throwableToClone = Utils.getDeepCompletionCause(t);
        if (!(throwableToClone instanceof RuntimeException || throwableToClone instanceof Error || throwableToClone instanceof IOException || throwableToClone instanceof InterruptedException)) {
            return null;
        }
        try {
            for (Constructor<?> constructor : throwableToClone.getClass().getConstructors()) {
                Class<?>[] paramTypes = constructor.getParameterTypes();
                if (paramTypes.length == 2 && paramTypes[0] == String.class && paramTypes[1] == Throwable.class) {
                    return (Throwable)constructor.newInstance(t.getMessage(), t);
                }
                if (paramTypes.length == 1 && paramTypes[0] == String.class) {
                    return ((Throwable)constructor.newInstance(t.getMessage())).initCause(t);
                }
                if (paramTypes.length == 1 && paramTypes[0] == Throwable.class) {
                    return (Throwable)constructor.newInstance(t);
                }
                if (paramTypes.length != 0) continue;
                return ((Throwable)constructor.newInstance(new Object[0])).initCause(t);
            }
        }
        catch (ReflectiveOperationException reflectiveOperationException) {
            // empty catch block
        }
        return null;
    }

    public static Throwable getDeepCompletionCause(Throwable t) {
        Throwable deeperCause;
        Throwable cause = t;
        while ((cause instanceof CompletionException || cause instanceof ExecutionException) && (deeperCause = cause.getCause()) != null) {
            cause = deeperCause;
        }
        return cause;
    }

    public static <T> T block(Future<T> future) throws IOException, InterruptedException {
        try {
            return future.get();
        }
        catch (ExecutionException e) {
            throw Utils.rethrowAsyncThrowable(e.getCause());
        }
    }

    public static <T> T blockOnIO(Future<T> future) throws IOException {
        try {
            return future.get();
        }
        catch (ExecutionException e) {
            try {
                throw Utils.rethrowAsyncThrowable(e.getCause());
            }
            catch (InterruptedException t) {
                throw new IOException("interrupted while waiting for I/O", t);
            }
        }
        catch (InterruptedException e) {
            throw new IOException("interrupted while waiting for I/O", e);
        }
    }

    public static void closeQuietly(@Nullable Closeable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
            }
            catch (IOException e) {
                logger.log(System.Logger.Level.WARNING, "exception while closing: " + closeable, (Throwable)e);
            }
        }
    }

    public static String escapeAndQuoteValueIfNeeded(String value) {
        return Utils.isValidToken(value) ? value : Utils.escapeAndQuote(value);
    }

    private static String escapeAndQuote(String value) {
        StringBuilder escaped = new StringBuilder();
        escaped.append('\"');
        for (int i = 0; i < value.length(); ++i) {
            char c = value.charAt(i);
            if (c == '\"' || c == '\\') {
                escaped.append('\\');
            }
            escaped.append(c);
        }
        escaped.append('\"');
        return escaped.toString();
    }
}

