package com.primexpy.openapi.utils;

import cn.hutool.http.HttpResponse;
import cn.hutool.http.HttpStatus;
import cn.hutool.http.HttpUtil;
import cn.hutool.json.JSONUtil;
import com.primexpy.openapi.code.RSACoder;
import com.primexpy.openapi.common.OpenApiConstant;
import com.primexpy.openapi.pojo.OpenApiBody;
import com.primexpy.openapi.pojo.OpenApiKeyPair;
import com.primexpy.openapi.res.CommonRes;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;

import java.util.concurrent.FutureTask;

/**
 * 自动加解密客户端
 *
 * @author primexpy
 */
@Slf4j
public class OpenApiClient {

    private String sysPriKey;

    private String thirdPubKey;

    private String app;

    private OpenApiClient(String sysPriKey, String thirdPubKey, String app) {
        this.sysPriKey = sysPriKey;
        this.thirdPubKey = thirdPubKey;
        this.app = app;
    }

    /**
     * 创建实例
     *
     * @param sysPriKey   使用者私钥
     * @param thirdPubKey 对方公钥
     * @param app         我方应用标识
     * @return OpenApiClient实例
     */
    public static OpenApiClient instance(String sysPriKey, String thirdPubKey, String app) {
        return new OpenApiClient(sysPriKey, thirdPubKey, app);
    }

    /**
     * 发送请求的核心方法
     *
     * @param url       URL
     * @param req       请求体DTO
     * @param beanClass 响应体class
     * @param <S>       响应体VO
     * @return 响应体
     * @throws Exception 可能会发生异常，请考虑它
     * @deprecated 为兼容老版本而存在，后续版本可能会删除它，推荐使用新方法OpenApiClient#request(String, Object, Class, String)
     */
    @Deprecated
    public <S> S request(String url, Object req, Class<S> beanClass) throws Exception {
        return this.req(url, req, beanClass, "");
    }

    /**
     * 发送请求
     *
     * @param url       URL
     * @param req       请求体DTO
     * @param beanClass 响应体class
     * @param <S>       响应体VO
     * @param salt      盐值
     * @return 响应体
     * @throws Exception 可能会发生异常，请考虑它
     */
    public <S> S request(String url, Object req, Class<S> beanClass, String salt) throws Exception {
        Validate.notEmpty(salt, "OpenApiClient validate error: salt can't be empty");
        return this.req(url, req, beanClass, salt);
    }

    /**
     * 生成密钥对
     *
     * @return 密钥对
     * @throws Exception 可能会发生异常，请考虑它
     */
    public static OpenApiKeyPair generatePrivateAndPublicKey() throws Exception {
        return RSACoder.generatePrivateAndPublicKey();
    }

    /**
     * 发送请求的核心方法
     *
     * @param url       URL
     * @param req       请求体DTO
     * @param beanClass 响应体class
     * @param <S>       响应体VO
     * @param salt      盐值
     * @return 响应体
     * @throws Exception 可能会发生异常，请考虑它
     */
    private <S> S req(String url, Object req, Class<S> beanClass, String salt) throws Exception {
        FutureTask<S> task = new FutureTask(() -> reqTask(url, req, beanClass, salt));
        new Thread(task).start();
        return task.get();
    }

    private <S> S reqTask(String url, Object req, Class<S> beanClass, String salt) throws Exception {
        String reqStr = JSONUtil.toJsonStr(req);
        log.info("OpenApiClient url: {} req: {}", url, reqStr);
        OpenApiBody openApiBody = OpenApiUtil.encrypt(reqStr, this.thirdPubKey);
        String reqSign = OpenApiUtil.sign(openApiBody, this.sysPriKey, salt);
        String reqBody = JSONUtil.toJsonStr(openApiBody);
        String ts = String.valueOf(System.currentTimeMillis());
        log.info("OpenApiClient >>>>>>>>>>>>>>>>>>> url: {} app:{} ts: {} reqSign: {} reqBody: {}", url, app, ts, reqSign, reqBody);
        HttpResponse executeRes = HttpUtil.createPost(url)
                .header(OpenApiConstant.CONTENT_TYPE, OpenApiConstant.CONTENT_TYPE_JSON)
                .header(OpenApiConstant.HEADER_SIGN, reqSign)
                .header(OpenApiConstant.HEADER_APP, app)
                .header(OpenApiConstant.HEADER_TS, ts)
                .body(reqBody)
                .execute();
        int status = executeRes.getStatus();
        String resBody = executeRes.body();
        String resSign = executeRes.header(OpenApiConstant.HEADER_SIGN);
        String resApp = executeRes.header(OpenApiConstant.HEADER_APP);
        log.info("OpenApiClient <<<<<<<<<<<<<<<<<< httpStatus: {} resSign: {} resApp: {} resBody: {} ", status, resSign, resApp, resBody);
        if (status != HttpStatus.HTTP_OK) {
            log.info("OpenApiClient error res: {}", resBody);
            CommonRes commonRes = JSONUtil.toBean(resBody, CommonRes.class);
            throw new RuntimeException("CODE:" + commonRes.getCode() + "|MSG:" + commonRes.getMsg());
        }
        log.info("OpenApiClient ok res: {}", resBody);
        OpenApiBody resBodyBean = JSONUtil.toBean(resBody, OpenApiBody.class);
        if (StringUtils.isNotEmpty(salt)) {
            boolean verify = OpenApiUtil.verify(resBodyBean, this.thirdPubKey, resSign, salt);
            log.info("OpenApiClient verify: {}", verify);
            Validate.isTrue(verify, "OpenApiClient verify sign error");
        }
        String decrypt = OpenApiUtil.decrypt(resBodyBean, this.sysPriKey);
        log.info("OpenApiClient decrypt: {}", decrypt);
        return JSONUtil.toBean(decrypt, beanClass);
    }


}
