package com.ttk.agg.openapi.sdk;

import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.parser.Feature;
import com.ttk.agg.openapi.sdk.dto.CallResult;
import com.ttk.agg.openapi.sdk.dto.ExceptionResult;
import com.ttk.agg.openapi.sdk.dto.InvoiceInspectionParamBody;
import com.ttk.agg.openapi.sdk.enums.InspectReturnMappingEnum;
import com.ttk.agg.openapi.sdk.inspect.BaseInvoice;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.ttk.agg.openapi.sdk.dto.OpenApiBusinessException;
import org.apache.logging.log4j.ThreadContext;

/**
 * @author lvaolin
 * 2020/3/4 下午4:57
 */

public class AggOpenAPI {


    /**
     * api 网关主机地址
     */
    private String apiHost;

    /**
     * 应用key
     */
    private String appKey;

    /**
     * 应用密钥
     */
    private String appSecret;

    /**
     * 应用级访问令牌
     */
    private volatile String token;

    /**
     * token有效期，毫秒
     */
    private volatile Long expiresIn;

    /**
     * 刷新token用
     */
    private volatile String refreshToken;

    /**
     * token获取时间
     */
    private Long lastTokenTime;

    private String apiVersion = "v1";

    /**
     * 代理服务器
     */
    private String proxyHost;

    /**
     * 代理服务器端口
     */
    private Integer proxyPort;

    /**
     * 代理服务器 认证用户名
     */
    private String proxyUser;

    /**
     * 代理服务器 认证用户密码
     */
    private String proxyPassword;


    /**
     * Socket超时时间，单位为毫秒。
     */
    private int socketTimeout = -1;

    /**
     * 连接超时时间，单位为毫秒。
     */
    private int connectTimeout = -1;

    /**
     * 从连接池获取连接的超时时间，单位为毫秒。
     */
    private int connectionRequestTimeout = -1;

    public AggOpenAPI(String apiHost, String appKey, String appSecret) {
        if (StringUtil.isNullOrEmpty(apiHost)) {
            throw new OpenApiBusinessException("", "参数apiHost不正确");
        }
        if (StringUtil.isNullOrEmpty(appKey)) {
            throw new OpenApiBusinessException("", "参数appKey不正确");
        }
        if (StringUtil.isNullOrEmpty(appSecret)) {
            throw new OpenApiBusinessException("", "参数appSecret不正确");
        }
        if (apiHost.endsWith("/")) {
            apiHost = apiHost.substring(0, apiHost.length() - 1);
        }
        this.apiHost = apiHost;
        this.appKey = appKey;
        this.appSecret = appSecret;
    }

    public AggOpenAPI(String apiHost, String appKey, String appSecret, String webHost) {
        this(apiHost, appKey, appSecret);
    }

    /**
     * 通用接口调用
     * @param path 接口地址
     * @param jsonParameter 接口参数，json 格式字符串
     * @return 返回值，json 格式字符串
     */
    public String restStr(String path, String jsonParameter) {
        if (path == null) {
            throw new OpenApiBusinessException("", "path不能为空");
        }
        if (jsonParameter == null) {
            throw new OpenApiBusinessException("", "jsonParameter不能为空");
        }

        try {
            if(path.toLowerCase().startsWith("/v")){
                setApiVersion(path.substring(1, 3));
            }
            if(StringUtil.isNullOrEmpty(proxyHost)) {
                return HttpUtil.postRestfulRequest(this, path.trim(), jsonParameter, null, socketTimeout, connectTimeout, connectionRequestTimeout);
            } else {
                return HttpUtil.postRestfulRequest(this, path.trim(), jsonParameter, null, socketTimeout, proxyHost, proxyPort, proxyUser, proxyPassword, connectTimeout, connectionRequestTimeout);
            }
        } catch (Exception e) {
            throw new OpenApiBusinessException("", e.getMessage());
        }
    }

    public String restStr(String path, String jsonParameter, int timeout) {
        if (path == null) {
            throw new OpenApiBusinessException("", "path不能为空");
        }
        if (jsonParameter == null) {
            throw new OpenApiBusinessException("", "jsonParameter不能为空");
        }

        if(socketTimeout > -1){
            timeout = socketTimeout;
        }
        try {
            if(path.toLowerCase().startsWith("/v")){
                setApiVersion(path.substring(1, 3));
            }
            if(StringUtil.isNullOrEmpty(proxyHost)) {
                return HttpUtil.postRestfulRequest(this, path.trim(), jsonParameter, null, timeout, connectTimeout, connectionRequestTimeout);
            } else {
                return HttpUtil.postRestfulRequest(this, path.trim(), jsonParameter, null, timeout, proxyHost, proxyPort, proxyUser, proxyPassword, connectTimeout, connectionRequestTimeout);
            }
        } catch (Exception e) {
            throw new OpenApiBusinessException("", e.getMessage());
        }
    }

    /**
     * 通用接口调用
     * @param path 接口地址
     * @param jsonParameter 接口参数，json 格式字符串
     * @return JSONObject json结果
     */
    public JSONObject rest(String path, String jsonParameter) {
        String result = restStr(path, jsonParameter);
        JSONObject resultObj;
        try {
            resultObj = JSONObject.parseObject(result, Feature.IgnoreNotMatch);
        } catch (Exception ex) {
            throw new OpenApiBusinessException("", "返回值不是正确的 json 格式，请联系管理员");
        }
        return resultObj;
    }

    public JSONObject rest(String path, String jsonParameter, int timeout) {
        String result = restStr(path, jsonParameter, timeout);
        JSONObject resultObj;
        try {
            resultObj = JSONObject.parseObject(result, Feature.IgnoreNotMatch);
        } catch (Exception ex) {
            throw new OpenApiBusinessException("", "返回值不是正确的 json 格式，请联系管理员");
        }
        return resultObj;
    }

    /**
     * 对发票进行检验的函数。
     *
     * @param invoiceInspectionParamBody 包含发票检验所需参数的实体对象，该对象将被转换为JSON字符串作为请求体发送。
     * @return 返回一个JSONObject对象，包含服务器响应的结果。
     */
    public CallResult<BaseInvoice> check(InvoiceInspectionParamBody invoiceInspectionParamBody) {
        // 调用rest方法，向指定URL发送POST请求，传入参数invoiceInspectionParamBody的JSON字符串形式
        String path = "/v2/invoice/customizedCheck";
        try {
            JSONObject jsonObject = rest(path, JSONObject.toJSONString(invoiceInspectionParamBody));
            if (!jsonObject.getBoolean("success")) {
                JSONObject errorJson = jsonObject.getJSONObject("exceptionResult");
                errorJson.put("requestId", jsonObject.getString("requestId"));
                return CallResult.fail(parse(errorJson.toJSONString(), apiHost+path));
            }
            JSONObject jsonData = jsonObject.getJSONObject("data");
            Class<BaseInvoice> clazz = InspectReturnMappingEnum.of(jsonData.getString("invoiceType")).getClazz();
            return CallResult.success(JSON.parseObject(jsonData.toString() ,clazz));
        } catch (Exception e) {
            ExceptionResult result = new ExceptionResult(UUID.randomUUID().toString(), apiHost+path, "CallException", e.getMessage());
            return CallResult.fail(result);
        }
    }
    private static ExceptionResult parse(String result, String url) {
        try {
            return (ExceptionResult)JSON.parseObject(result, ExceptionResult.class);
        } catch (Exception var3) {
            return new ExceptionResult(UUID.randomUUID().toString(), url, "CallException", "无法连接服务器，请稍后重试。");
        }
    }
    /**
     * 获取应用级访问token
     * @return string token
     */
    protected String getAccessToken() throws NoSuchAlgorithmException, KeyManagementException {
        if (StringUtil.isNullOrEmpty(token)) {
            synchronized (this) {
                if (StringUtil.isNullOrEmpty(token)) {
                    //获取新token
                    newToken();
                }
            }
        } else {
            //验证有效期，如果快到有效期则刷新token，如果超了有效期则重新获取token
            Long currentTime = System.currentTimeMillis();
            //token持有时间
            Long holdingTime = currentTime - lastTokenTime;
            //到期前24~2小时内 均可刷新token
            if (holdingTime >= (expiresIn - 3600 * 24 * 1000) && holdingTime <= (expiresIn - 3600 * 2 * 1000)) {
                synchronized (this) {
                    //刷新token
                    refreshToken();
                }
            } else if (holdingTime > (expiresIn - 3600 * 2 * 1000)) {
                synchronized (this) {
                    //获取新token
                    newToken();
                }

            }
        }
        return token;
    }

    /**
     * 获取新token
     */
    private void newToken() throws NoSuchAlgorithmException, KeyManagementException {
        //1、body数据准备
        JSONObject requestBody = new JSONObject();
        requestBody.put("grant_type", "client_credentials");
        requestBody.put("client_appkey", appKey);
        requestBody.put("client_secret", HttpUtil.MD5(appSecret));
        //2、发送请求
        String url = apiHost;
        if(apiHost.toLowerCase().indexOf("/v") > -1){
            url = url.substring(0, url.indexOf("/v"));
        }
        if("v1".equalsIgnoreCase(apiVersion)) {
            url += "/" + apiVersion + "/AGG/oauth2/login";
        } else {  // v2 版本的api地址
            url += "/" + apiVersion + "/public/oauth2/login";
        }
        String responseStr = HttpUtil.post(url, requestBody.toJSONString(), null, socketTimeout, connectTimeout, connectionRequestTimeout);
        JSONObject jsonObject;
        try {
            jsonObject = JSONObject.parseObject(responseStr, Feature.IgnoreNotMatch);
            //3、解析返回结果
            if (jsonObject == null) {
                throw new OpenApiBusinessException("", "获取token失败，请确认网址" + apiHost + " 是否能正常访问！");
            }
        } catch (Exception ex) {
            if (ex instanceof OpenApiBusinessException) {
                throw ex;
            }
            throw new OpenApiBusinessException("", ex.getMessage());
        }

        //4、解析logId 全局追踪id 日志流水id
        if("v1".equalsIgnoreCase(apiVersion)) {
            JSONObject jsonObjectResult = jsonObject.getJSONObject("result");
            if (jsonObjectResult != null) {
                String logId = jsonObjectResult.getString("req_id");
                if (logId != null) {
                    ThreadContext.put(StringUtil.TRACE_LOG_ID, logId);
                }
            }
        } else {
            String logId = jsonObject.getString("reqId");
            if (logId != null) {
                ThreadContext.put(StringUtil.TRACE_LOG_ID, logId);
            }
        }

        //5、解析返回token
        JSONObject jsonObjectBody;
        if("v1".equalsIgnoreCase(apiVersion)) {
            jsonObjectBody = jsonObject.getJSONObject("value");
            if (jsonObjectBody.get("error_msg") != null) {
                throw new OpenApiBusinessException("", jsonObjectBody.get("error_msg").toString());
            }
        } else {
            if (!jsonObject.getBoolean("success")) {
                throw new OpenApiBusinessException(jsonObject.get("code").toString(), jsonObject.get("message").toString());
            }
            jsonObjectBody = jsonObject.getJSONObject("data");
        }
        token = jsonObjectBody.getString("access_token");
        expiresIn = jsonObjectBody.getLong("expires_in");
        lastTokenTime = System.currentTimeMillis();
        refreshToken = jsonObjectBody.getString("refresh_token");

        if(StringUtil.isNullOrEmpty(token)) {
            throw new OpenApiBusinessException("", "access_token不能为空");
        }
    }

    /**
     * 刷新token
     */
    private void refreshToken() throws NoSuchAlgorithmException, KeyManagementException {
        //refresh_token
        //1、body数据准备
        JSONObject requestBody = new JSONObject();
        requestBody.put("grant_type", "refresh_token");
        requestBody.put("refresh_token", refreshToken);
        //2、发送请求
        String url = apiHost;
        if(apiHost.toLowerCase().indexOf("/v") > -1){
            url = url.substring(0, url.indexOf("/v"));
        }
        url += "/" + apiVersion + "/AGG/oauth2/login";
        String responseStr = HttpUtil.post(url, requestBody.toJSONString(), null, socketTimeout, connectTimeout, connectionRequestTimeout);
        JSONObject jsonObject;
        try {
            jsonObject = JSONObject.parseObject(responseStr, Feature.IgnoreNotMatch);
            //3、解析返回结果
            if (jsonObject == null) {
                throw new OpenApiBusinessException("", "刷新token失败，请确认网址" + apiHost + " 是否能正常访问！");
            }
        } catch (Exception ex) {
            if (ex instanceof OpenApiBusinessException) {
                throw ex;
            }
            throw new OpenApiBusinessException("", ex.getMessage());
        }

        //4、解析logId 全局追踪id 日志流水id
        if("v1".equalsIgnoreCase(apiVersion)) {
            JSONObject jsonObjectResult = jsonObject.getJSONObject("result");
            if (jsonObjectResult != null) {
                String logId = jsonObjectResult.getString("req_id");
                if (logId != null) {
                    ThreadContext.put(StringUtil.TRACE_LOG_ID, logId);
                }
            }
        } else {
            String logId = jsonObject.getString("reqId");
            if (logId != null) {
                ThreadContext.put(StringUtil.TRACE_LOG_ID, logId);
            }
        }

        //5、解析返回token
        JSONObject jsonObjectBody;
        if("v1".equalsIgnoreCase(apiVersion)) {
            jsonObjectBody = jsonObject.getJSONObject("value");
            if (jsonObjectBody.get("info_msg") != null) {
                throw new OpenApiBusinessException("", jsonObjectBody.get("info_msg").toString());
            }
        } else {
            if (!jsonObject.getBoolean("success")) {
                throw new OpenApiBusinessException(jsonObject.get("code").toString(), jsonObject.get("message").toString());
            }
            jsonObjectBody = jsonObject.getJSONObject("data");
        }
        token = jsonObjectBody.getString("access_token");
        expiresIn = jsonObjectBody.getLong("expires_in");
        lastTokenTime = System.currentTimeMillis();
        refreshToken = jsonObjectBody.getString("refresh_token");

        if(StringUtil.isNullOrEmpty(token)) {
            throw new OpenApiBusinessException("", "access_token不能为空");
        }
    }


    /**
     * 获取发票下载任务的状态
     * <p>参数传入 json 字符串
     * <p>参数格式：nsrsbh 企业纳税人识别号，必填
     * <p>kpyfs 开票月份列表，必填，[202104, 202103]，格式yyyyMM
     * <p>jxxbzs 进销项标识列表，必填，jx 进项，xx 销项
     * <p>fplxs 发票类型代码列表，必填
     * <p>addJob 是否发起下载任务，默认不发起
     * <p>第一次调用接口传入 true 发起任务，之后定时调用接口传入 false 获取状态，直到接口返回处理成功或者失败，返回成功调用采集发票接口
     *
     * @param jsonParameter json参数
     * @return 任务状态列表
     * <p>格式 {"jxxbz": "jx", "kpyf": 202104, "fplx": "01", "status": "processed", "msg": ""}
     * <p>任务处理成功的情况 status: processed
     * <p>任务处理失败的情况 status: failed, msg 为失败原因
     * <p>其他情况 toProcess：已发起申请，待处理；processing：处理中；disabled：不支持
     * @throws OpenApiBusinessException 业务异常
     */
    public JSONArray getFpxzStatus(String jsonParameter) throws OpenApiBusinessException {
        if (StringUtil.isNullOrEmpty(jsonParameter)) {
            throw new OpenApiBusinessException("", "参数不能为空");
        }
        JSONObject jsonObj;
        try {
            jsonObj = JSONObject.parseObject(jsonParameter, Feature.IgnoreNotMatch);
        } catch (Exception ex) {
            String msg = "获取发票下载状态失败，接口参数转 json 异常：" + ex.getMessage();
            throw new OpenApiBusinessException("", msg);
        }
        List<String> invalidList = new ArrayList<>();
        String nsrsbh = jsonObj.getString("nsrsbh");
        if (StringUtil.isNullOrEmpty(nsrsbh)) {
            invalidList.add("纳税人识别号");
        }
        JSONArray kpyfs = jsonObj.getJSONArray("kpyfs");
        if (kpyfs == null || kpyfs.isEmpty()) {
            invalidList.add("开票月份列表");
        }
        JSONArray jxxbzs = jsonObj.getJSONArray("jxxbzs");
        if (jxxbzs == null || jxxbzs.isEmpty()) {
            invalidList.add("进销项标识列表");
        }
        JSONArray fplxs = jsonObj.getJSONArray("fplxs");
        if (fplxs == null || fplxs.isEmpty()) {
            invalidList.add("发票类型代码列表");
        }
        if (!invalidList.isEmpty()) {
            throw new OpenApiBusinessException("", StringUtil.join(invalidList, "、") + "不能为空");
        }

        boolean addJob = false;
        if (jsonObj.containsKey("addJob")) {
            addJob = jsonObj.getBooleanValue("addJob");
        }
        jsonObj.put("addJob", addJob);
        jsonObj.remove("kpyfs");

        JSONArray result = new JSONArray();
        for (Object kpyf : kpyfs) {
            jsonObj.put("kpyf", kpyf);
            JSONObject fpxzObj = rest("/FP/getFpxzStatus", jsonObj.toJSONString());
            JSONObject fpxzResultObj = fpxzObj.getJSONObject("result");
            Boolean success = fpxzResultObj.getBoolean("success");
            if (success != null && success) {
                result.addAll(fpxzObj.getJSONArray("value"));
            } else {
                String msg = fpxzObj.getJSONObject("error").getString("message");
                Date date = new Date();
                for (Object jxxbz : jxxbzs) {
                    for (Object fplx : fplxs) {
                        JSONObject error = new JSONObject();
                        error.put("jxxbz", jxxbz);
                        error.put("fplx", fplx);
                        error.put("kpyf", kpyf);
                        error.put("status", "failed");
                        error.put("msg", msg);
                        error.put("time", date);
                        result.add(error);
                    }
                }
            }
        }
        return result;
    }

    /**
     * 采集发票
     * <p>参数传入 json 字符串
     * <p>参数格式：nsrsbh 企业纳税人识别号，必填
     * <p>kpyfs 开票月份列表，必填，[202104, 202103]，格式yyyyMM
     * <p>jxxbzs 进销项标识列表，必填，jx 进项，xx 销项
     * <p>fplxs 发票类型代码列表，必填
     *
     * @param jsonParameter json参数
     * @return 发票结果
     * <p>以税款所属期+进销项标识+发票类型（例如202104_jx_01）为 key 的 map
     * <p>value 格式 {"result": true/false, "list": [], "msg": ""}
     * <p>采集成功 result: true, list 为对应的发票列表
     * <p>采集失败 result: false, msg 为失败原因
     * @throws OpenApiBusinessException 业务异常
     */
    public JSONObject cj(String jsonParameter) throws OpenApiBusinessException {
        if (StringUtil.isNullOrEmpty(jsonParameter)) {
            throw new OpenApiBusinessException("", "参数不能为空");
        }
        JSONObject jsonObj;
        try {
            jsonObj = JSONObject.parseObject(jsonParameter, Feature.IgnoreNotMatch);
        } catch (Exception ex) {
            String msg = "采集发票失败，接口参数转 json 异常：" + ex.getMessage();
            throw new OpenApiBusinessException("", msg);
        }
        List<String> invalidList = new ArrayList<>();
        String nsrsbh = jsonObj.getString("nsrsbh");
        if (StringUtil.isNullOrEmpty(nsrsbh)) {
            invalidList.add("纳税人识别号");
        }
        JSONArray kpyfs = jsonObj.getJSONArray("kpyfs");
        if (kpyfs == null || kpyfs.isEmpty()) {
            invalidList.add("开票月份列表");
        }
        JSONArray jxxbzs = jsonObj.getJSONArray("jxxbzs");
        if (jxxbzs == null || jxxbzs.isEmpty()) {
            invalidList.add("进销项标识列表");
        }
        JSONArray fplxs = jsonObj.getJSONArray("fplxs");
        if (fplxs == null || fplxs.isEmpty()) {
            invalidList.add("发票类型代码列表");
        }
        if (!invalidList.isEmpty()) {
            throw new OpenApiBusinessException("", StringUtil.join(invalidList, "、") + "不能为空");
        }
        JSONObject result = new JSONObject();
        jsonObj.remove("kpyfs");
        jsonObj.remove("jxxbzs");
        jsonObj.remove("fplxs");
        for (Object kpyf : kpyfs) {
            jsonObj.put("kpyf", kpyf);
            for (Object jxxbz : jxxbzs) {
                jsonObj.put("jxxbz", jxxbz);
                for (Object fplx : fplxs) {
                    jsonObj.put("fplx", fplx);
                    String key = "" + kpyf + "_" + jxxbz + "_" + fplx;
                    JSONObject cjObj = rest("/FP/cj", jsonObj.toJSONString());
                    JSONObject cjResultObj = cjObj.getJSONObject("result");
                    Boolean success = cjResultObj.getBoolean("success");
                    JSONObject item = new JSONObject();
                    if (success != null && success) {
                        JSONArray list = cjObj.getJSONObject("value").getJSONArray("list");
                        item.put("result", true);
                        item.put("list", list);
                    } else {
                        String msg = cjObj.getJSONObject("error").getString("message");
                        item.put("result", false);
                        item.put("msg", msg);
                    }
                    result.put(key, item);
                }
            }
        }
        return result;
    }

    /**
     * 获取更新勾选状态任务的状态
     * <p>参数传入 json 字符串
     * <p>参数格式：nsrsbh 企业纳税人识别号，必填
     * <p>skssqs 税款所属期列表，必填，[202104, 202103]，格式yyyyMM
     *
     * @param jsonParameter json参数
     * @return 任务状态列表
     * <p>格式 {"skssq": 202104, "status": "processed", "msg": ""}
     * <p>任务处理成功的情况 status: processed
     * <p>任务处理失败的情况 status: failed, msg 为失败原因
     * <p>其他情况 toProcess：已发起申请，待处理；processing：处理中；disabled：不支持
     * @throws OpenApiBusinessException 业务异常
     */
    public JSONArray getGxgxztStatus(String jsonParameter) throws OpenApiBusinessException {
        if (StringUtil.isNullOrEmpty(jsonParameter)) {
            throw new OpenApiBusinessException("", "参数不能为空");
        }
        JSONObject jsonObj;
        try {
            jsonObj = JSONObject.parseObject(jsonParameter, Feature.IgnoreNotMatch);
        } catch (Exception ex) {
            String msg = "获取更新发票勾选状态失败，接口参数转 json 异常：" + ex.getMessage();
            throw new OpenApiBusinessException("", msg);
        }
        List<String> invalidList = new ArrayList<>();
        String nsrsbh = jsonObj.getString("nsrsbh");
        if (StringUtil.isNullOrEmpty(nsrsbh)) {
            invalidList.add("纳税人识别号");
        }
        JSONArray skssqs = jsonObj.getJSONArray("skssqs");
        if (skssqs == null || skssqs.isEmpty()) {
            invalidList.add("税款所属期列表");
        }
        if (!invalidList.isEmpty()) {
            throw new OpenApiBusinessException("", StringUtil.join(invalidList, "、") + "不能为空");
        }

        boolean addJob = false;
        if (jsonObj.containsKey("addJob")) {
            addJob = jsonObj.getBooleanValue("addJob");
        }
        jsonObj.put("addJob", addJob);
        jsonObj.remove("skssqs");

        JSONArray result = new JSONArray();
        for (Object skssq : skssqs) {
            jsonObj.put("skssq", skssq);
            JSONObject gxgxztObj = rest("/FP/getGxgxztStatus", jsonObj.toJSONString());
            JSONObject gxgxztResultObj = gxgxztObj.getJSONObject("result");
            Boolean success = gxgxztResultObj.getBoolean("success");
            if (success != null && success) {
                JSONObject item = gxgxztObj.getJSONObject("value");
                result.add(item);
            } else {
                String msg = gxgxztObj.getJSONObject("error").getString("message");
                Date date = new Date();
                JSONObject error = new JSONObject();
                error.put("status", "failed");
                error.put("msg", msg);
                error.put("time", date);
                error.put("skssq", skssq);
                result.add(error);
            }
        }
        return result;
    }

    /**
     * 采集已勾选发票，支持多税款所属期
     * <p>参数传入 json 字符串
     * <p>参数格式：nsrsbh 企业纳税人识别号，必填
     * <p>skssqs 税款所属期列表，必填，[202104, 202103]，格式yyyyMM
     * <p>fplxs 发票类型代码列表，必填，["01", "03", "14", "17"]
     *
     * @param jsonParameter json参数
     * @return 发票结果
     * <p>以税款所属期+发票类型（例如202104_01）为 key 的 map
     * <p>value 格式 {"result": true/false, "list": [], "msg": ""}
     * <p>采集成功 result: true, list 为对应的发票列表
     * <p>采集失败 result: false, msg 为失败原因
     * @throws OpenApiBusinessException 业务异常
     */
    public JSONObject cjYgx(String jsonParameter) throws OpenApiBusinessException {
        if (StringUtil.isNullOrEmpty(jsonParameter)) {
            throw new OpenApiBusinessException("", "参数不能为空");
        }
        JSONObject jsonObj;
        try {
            jsonObj = JSONObject.parseObject(jsonParameter, Feature.IgnoreNotMatch);
        } catch (Exception ex) {
            String msg = "采集发票失败，接口参数转 json 异常：" + ex.getMessage();
            throw new OpenApiBusinessException("", msg);
        }
        List<String> invalidList = new ArrayList<>();
        String nsrsbh = jsonObj.getString("nsrsbh");
        if (StringUtil.isNullOrEmpty(nsrsbh)) {
            invalidList.add("纳税人识别号");
        }
        JSONArray skssqs = jsonObj.getJSONArray("skssqs");
        if (skssqs == null || skssqs.isEmpty()) {
            invalidList.add("税款所属期列表");
        }
        JSONArray fplxs = jsonObj.getJSONArray("fplxs");
        if (fplxs == null || fplxs.isEmpty()) {
            invalidList.add("发票类型代码列表");
        }
        if (!invalidList.isEmpty()) {
            throw new OpenApiBusinessException("", StringUtil.join(invalidList, "、") + "不能为空");
        }
        jsonObj.remove("skssqs");
        jsonObj.remove("fplxs");

        JSONObject result = new JSONObject();
        for (Object skssq : skssqs) {
            jsonObj.put("skssq", skssq);
            for (Object fplx : fplxs) {
                jsonObj.put("fplx", fplx);
                String key = "" + skssq + "_" + fplx;
                JSONObject item = new JSONObject();
                JSONObject cjObj = rest("/FP/cjYgx", jsonObj.toJSONString());
                JSONObject cjResultObj = cjObj.getJSONObject("result");
                Boolean success = cjResultObj.getBoolean("success");
                if (success != null && success) {
                    JSONArray list = cjObj.getJSONObject("value").getJSONArray("list");
                    item.put("result", true);
                    item.put("list", list);
                } else {
                    String msg = cjObj.getJSONObject("error").getString("message");
                    item.put("result", false);
                    item.put("msg", msg);
                }
                result.put(key, item);
            }
        }
        return result;
    }

    public String getApiHost() {
        return apiHost;
    }

    public void setApiHost(String apiHost) {
        this.apiHost = apiHost;
    }

    public String getAppKey() {
        return appKey;
    }

    public void setAppKey(String appKey) {
        this.appKey = appKey;
    }

    public String getAppSecret() {
        return appSecret;
    }

    public void setAppSecret(String appSecret) {
        this.appSecret = appSecret;
    }

    public String getToken() {
        return token;
    }

    public void setToken(String token) {
        this.token = token;
    }

    public String getApiVersion(){
        return apiVersion;
    }

    public void setApiVersion(String apiVersion) {
        this.apiVersion = apiVersion;
    }

    public String getProxyHost() {
        return proxyHost;
    }

    public void setProxyHost(String proxyHost) {
        this.proxyHost = proxyHost;
    }

    public Integer getProxyPort() {
        return proxyPort;
    }

    public void setProxyPort(Integer proxyPort) {
        this.proxyPort = proxyPort;
    }

    public String getProxyUser() {
        return proxyUser;
    }

    public void setProxyUser(String proxyUser) {
        this.proxyUser = proxyUser;
    }

    public String getProxyPassword() {
        return proxyPassword;
    }

    public void setProxyPassword(String proxyPassword) {
        this.proxyPassword = proxyPassword;
    }

    public int getSocketTimeout() {
        return socketTimeout;
    }

    public void setSocketTimeout(int socketTimeout) {
        this.socketTimeout = socketTimeout;
    }

    public int getConnectTimeout() {
        return connectTimeout;
    }

    public void setConnectTimeout(int connectTimeout) {
        this.connectTimeout= connectTimeout;
    }
    public int getConnectionRequestTimeout() {
        return connectionRequestTimeout;
    }

    public void setConnectionRequestTimeout(int connectionRequestTimeout) {
        this.connectionRequestTimeout = connectionRequestTimeout;
    }
}
