package com.ttk.agg.openapi.sdk;

import java.io.UnsupportedEncodingException;
import java.security.KeyManagementException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.ResourceBundle;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;

import com.ttk.agg.openapi.sdk.dto.OpenApiBusinessException;
import org.apache.logging.log4j.ThreadContext;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;

public class HttpUtil {

    private static String sdkVersion = null;

    /**
     * 读取sdk版本号
     */
    static {
        try {
            ResourceBundle resource = ResourceBundle.getBundle("sdk");
            if (resource != null) {
                sdkVersion = resource.getString("sdkVersion");
            }
        } catch (Exception e) {
            sdkVersion = "";
        }
    }

    public static String post(String url, String requestBody) throws NoSuchAlgorithmException, KeyManagementException {
        return post(url, requestBody, null);
    }

    public static String post(String url, String requestBody, Map<String, String> headerMap) throws NoSuchAlgorithmException, KeyManagementException {
        if (StringUtil.isNullOrEmpty(url)) {
            throw new OpenApiBusinessException("", "url不能为空");
        }
        if (StringUtil.isNullOrEmpty(requestBody)) {
            throw new OpenApiBusinessException("", "requestBody不能为null");
        }
        // 创建SSLContext并设置TrustManager
        SSLContext sslContext = SSLContext.getInstance("TLS");
        sslContext.init(null, new TrustManager[]{new NoopTrustManager()}, null);
        SSLConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(sslContext);
        CloseableHttpClient httpClient = HttpClients.custom()
                .setSSLSocketFactory(sslsf)
                .build();
        HttpPost httpPost = new HttpPost(url);

        // 定制header参数
        httpPost.setHeader("Accept", "application/json");
        httpPost.setHeader("Content-Type", "application/json");
        if (sdkVersion == null) {
            sdkVersion = "";
        }
        httpPost.setHeader("sdkVersion", sdkVersion);
        // 添加通用参数
        if (headerMap != null) {
            Iterator<Map.Entry<String, String>> iterator = headerMap.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<String, String> entry = iterator.next();
                httpPost.setHeader(entry.getKey(), entry.getValue());
            }
        }

        // 链路追踪logId
        Object logId = ThreadContext.get(StringUtil.TRACE_LOG_ID);
        if(logId != null) {
            httpPost.setHeader(StringUtil.TRACE_LOG_ID, logId.toString());
        }

        String charSet = "UTF-8";
        // 请求体body
        StringEntity entity = new StringEntity(requestBody, charSet);
        httpPost.setEntity(entity);
        CloseableHttpResponse httpResponse = null;
        try {
            // 开始网络请求
            httpResponse = httpClient.execute(httpPost);
            // 响应状态码
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode == 200) {
                HttpEntity httpEntity = httpResponse.getEntity();
                String responseStr = EntityUtils.toString(httpEntity, charSet);
                return responseStr;
            } else {
                throw new OpenApiBusinessException("", "rest请求失败，(" + url + ")状态码：" + statusCode);
            }
        } catch (Exception e) {
            throw new OpenApiBusinessException("", "rest请求过程中有异常发生：" + e.getMessage());
        } finally {
            try {
                if (httpResponse != null) {
                    httpResponse.close();
                }
                if (httpClient != null) {
                    httpClient.close();
                }
            } catch (Exception e) {
                throw new OpenApiBusinessException("", "http关闭时有异常发生：" + e.getMessage());
            }
        }

    }

    public static String post(String url, String requestBody, Map<String, String> headerMap
            , String proxyHost, Integer proxyPort
            , String proxyUser, String proxyPassword) throws NoSuchAlgorithmException, KeyManagementException {
        if (StringUtil.isNullOrEmpty(url)) {
            throw new OpenApiBusinessException("", "url不能为空");
        }
        if (StringUtil.isNullOrEmpty(requestBody)) {
            throw new OpenApiBusinessException("", "requestBody不能为null");
        }
        // 创建SSLContext并设置TrustManager
        SSLContext sslContext = SSLContext.getInstance("TLS");
        sslContext.init(null, new TrustManager[]{new NoopTrustManager()}, null);
        SSLConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(sslContext);
        CloseableHttpClient httpClient = HttpClients.custom()
                .setSSLSocketFactory(sslsf)
                .build();
        // 设置代理服务器 带认证
        if(!StringUtil.isNullOrEmpty(proxyUser)) {
            CredentialsProvider credsProvider = new BasicCredentialsProvider();
            credsProvider.setCredentials(AuthScope.ANY, new UsernamePasswordCredentials(proxyUser, proxyPassword));
            httpClient = HttpClients.custom()
                    .setSSLSocketFactory(sslsf).setDefaultCredentialsProvider(credsProvider).
                    build();
        }
        HttpPost httpPost = new HttpPost(url);

        // 设置代理服务器
        if(!StringUtil.isNullOrEmpty(proxyHost)){
            RequestConfig requestConfig = httpPost.getConfig();
            if(requestConfig == null) {
                HttpHost proxy = new HttpHost(proxyHost, proxyPort);
                requestConfig = RequestConfig.custom().setProxy(proxy).build();
                httpPost.setConfig(requestConfig);
            }
        }

        // 定制header参数
        httpPost.setHeader("Accept", "application/json");
        httpPost.setHeader("Content-Type", "application/json");
        if (sdkVersion == null) {
            sdkVersion = "";
        }
        httpPost.setHeader("sdkVersion", sdkVersion);
        // 添加通用参数
        if (headerMap != null) {
            Iterator<Map.Entry<String, String>> iterator = headerMap.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<String, String> entry = iterator.next();
                httpPost.setHeader(entry.getKey(), entry.getValue());
            }
        }

        // 链路追踪logId
        Object logId = ThreadContext.get(StringUtil.TRACE_LOG_ID);
        if(logId != null) {
            httpPost.setHeader(StringUtil.TRACE_LOG_ID, logId.toString());
        }

        String charSet = "UTF-8";
        // 请求体body
        StringEntity entity = new StringEntity(requestBody, charSet);
        httpPost.setEntity(entity);
        CloseableHttpResponse httpResponse = null;
        try {
            // 开始网络请求
            httpResponse = httpClient.execute(httpPost);
            // 响应状态码
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode == 200) {
                HttpEntity httpEntity = httpResponse.getEntity();
                String responseStr = EntityUtils.toString(httpEntity, charSet);
                return responseStr;
            } else {
                HttpEntity httpEntity = httpResponse.getEntity();
                String responseStr = EntityUtils.toString(httpEntity, charSet);
                throw new OpenApiBusinessException("", "rest请求失败，(" + url + ")状态码：" + statusCode + " responseStr=" + responseStr);
            }
        } catch (Exception e) {
            throw new OpenApiBusinessException("", "rest请求过程中有异常发生：" + e.getMessage());
        } finally {
            try {
                if (httpResponse != null) {
                    httpResponse.close();
                }
                if (httpClient != null) {
                    httpClient.close();
                }
            } catch (Exception e) {
                throw new OpenApiBusinessException("", "http关闭时有异常发生：" + e.getMessage());
            }
        }

    }

    public static String post(String url, String requestBody, Map<String, String> headerMap, int timeout, int connectTimeout, int connectionRequestTimeout) throws NoSuchAlgorithmException, KeyManagementException {
        if (StringUtil.isNullOrEmpty(url)) {
            throw new OpenApiBusinessException("", "url不能为空");
        }
        if (StringUtil.isNullOrEmpty(requestBody)) {
            throw new OpenApiBusinessException("", "requestBody不能为null");
        }
        // 创建SSLContext并设置TrustManager
        SSLContext sslContext = SSLContext.getInstance("TLS");
        sslContext.init(null, new TrustManager[]{new NoopTrustManager()}, null);
        SSLConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(sslContext);
        CloseableHttpClient httpClient = HttpClients.custom()
                .setSSLSocketFactory(sslsf)
                .build();
        HttpPost httpPost = new HttpPost(url);
        RequestConfig.Builder customReqConf = RequestConfig.custom();
        if (timeout > -1) {
            customReqConf.setSocketTimeout(timeout);
            httpPost.setHeader("socketTimeout", String.valueOf(timeout));
        }
        if (connectTimeout > -1) {
            customReqConf.setConnectTimeout(connectTimeout);
            httpPost.setHeader("connectTimeout", String.valueOf(connectTimeout));
        }
        if (connectionRequestTimeout > -1) {
            customReqConf.setConnectionRequestTimeout(connectionRequestTimeout);
            httpPost.setHeader("connectionRequestTimeout", String.valueOf(connectionRequestTimeout));
        }
        RequestConfig requestConfig = customReqConf.build();
        httpPost.setConfig(requestConfig);
        // 定制header参数
        httpPost.setHeader("Accept", "application/json");
        httpPost.setHeader("Content-Type", "application/json");
        if (sdkVersion == null) {
            sdkVersion = "";
        }
        httpPost.setHeader("sdkVersion", sdkVersion);
        // 添加通用参数
        if (headerMap != null) {
            Iterator<Map.Entry<String, String>> iterator = headerMap.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<String, String> entry = iterator.next();
                httpPost.setHeader(entry.getKey(), entry.getValue());
            }
        }

        // 链路追踪logId
        Object logId = ThreadContext.get(StringUtil.TRACE_LOG_ID);
        if(logId != null) {
            httpPost.setHeader(StringUtil.TRACE_LOG_ID, logId.toString());
        }

        String charSet = "UTF-8";
        // 请求体body
        StringEntity entity = new StringEntity(requestBody, charSet);
        httpPost.setEntity(entity);
        CloseableHttpResponse httpResponse = null;
        try {
            // 开始网络请求
            httpResponse = httpClient.execute(httpPost);
            // 响应状态码
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode == 200) {
                HttpEntity httpEntity = httpResponse.getEntity();
                String responseStr = EntityUtils.toString(httpEntity, charSet);
                return responseStr;
            } else {
                throw new OpenApiBusinessException("", "rest请求失败，(" + url + ")状态码：" + statusCode);
            }
        } catch (Exception e) {
            throw new OpenApiBusinessException("", "rest请求过程中有异常发生：" + e.getMessage());
        } finally {
            try {
                if (httpResponse != null) {
                    httpResponse.close();
                }
                if (httpClient != null) {
                    httpClient.close();
                }
            } catch (Exception e) {
                throw new OpenApiBusinessException("", "http关闭时有异常发生：" + e.getMessage());
            }
        }

    }

    public static String post(String url, String requestBody, Map<String, String> headerMap, int timeout
            , String proxyHost, Integer proxyPort
            , String proxyUser, String proxyPassword, int connectTimeout, int connectionRequestTimeout) throws NoSuchAlgorithmException, KeyManagementException {
        if (StringUtil.isNullOrEmpty(url)) {
            throw new OpenApiBusinessException("", "url不能为空");
        }
        if (StringUtil.isNullOrEmpty(requestBody)) {
            throw new OpenApiBusinessException("", "requestBody不能为null");
        }
        // 创建SSLContext并设置TrustManager
        SSLContext sslContext = SSLContext.getInstance("TLS");
        sslContext.init(null, new TrustManager[]{new NoopTrustManager()}, null);
        SSLConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(sslContext);
        CloseableHttpClient httpClient = HttpClients.custom()
                .setSSLSocketFactory(sslsf)
                .build();
        // 设置代理服务器 带认证
        if(!StringUtil.isNullOrEmpty(proxyUser)) {
            CredentialsProvider cdtProvider = new BasicCredentialsProvider();
            cdtProvider.setCredentials(new AuthScope(proxyHost, proxyPort), new UsernamePasswordCredentials(proxyUser, proxyPassword));
            httpClient = HttpClients.custom()
                    .setSSLSocketFactory(sslsf).setDefaultCredentialsProvider(cdtProvider).
                    build();
        }
        HttpPost httpPost = new HttpPost(url);
        RequestConfig.Builder customReqConf = RequestConfig.custom();
        if (timeout > -1) {
            customReqConf.setSocketTimeout(timeout);
            httpPost.setHeader("socketTimeout", String.valueOf(timeout));
        }
        if (connectTimeout > -1) {
            customReqConf.setConnectTimeout(connectTimeout);
            httpPost.setHeader("connectTimeout", String.valueOf(connectTimeout));
        }
        if (connectionRequestTimeout > -1) {
            customReqConf.setConnectionRequestTimeout(connectionRequestTimeout);
            httpPost.setHeader("connectionRequestTimeout", String.valueOf(connectionRequestTimeout));
        }
        // 设置代理服务器
        if(!StringUtil.isNullOrEmpty(proxyHost)){
                HttpHost proxy = new HttpHost(proxyHost, proxyPort);
                customReqConf.setProxy(proxy);
        }
        RequestConfig requestConfig = customReqConf.build();
        httpPost.setConfig(requestConfig);
        // 定制header参数
        httpPost.setHeader("Accept", "application/json");
        httpPost.setHeader("Content-Type", "application/json");
        if (sdkVersion == null) {
            sdkVersion = "";
        }
        httpPost.setHeader("sdkVersion", sdkVersion);
        // 添加通用参数
        if (headerMap != null) {
            Iterator<Map.Entry<String, String>> iterator = headerMap.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<String, String> entry = iterator.next();
                httpPost.setHeader(entry.getKey(), entry.getValue());
            }
        }

        // 链路追踪logId
        Object logId = ThreadContext.get(StringUtil.TRACE_LOG_ID);
        if(logId != null) {
            httpPost.setHeader(StringUtil.TRACE_LOG_ID, logId.toString());
        }

        String charSet = "UTF-8";
        // 请求体body
        StringEntity entity = new StringEntity(requestBody, charSet);
        httpPost.setEntity(entity);
        CloseableHttpResponse httpResponse = null;
        try {
            // 开始网络请求
            httpResponse = httpClient.execute(httpPost);
            // 响应状态码
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode == 200) {
                HttpEntity httpEntity = httpResponse.getEntity();
                String responseStr = EntityUtils.toString(httpEntity, charSet);
                return responseStr;
            } else {
                throw new OpenApiBusinessException("", "rest请求失败，(" + url + ")状态码：" + statusCode);
            }
        } catch (Exception e) {
            throw new OpenApiBusinessException("", "rest请求过程中有异常发生：" + e.getMessage());
        } finally {
            try {
                if (httpResponse != null) {
                    httpResponse.close();
                }
                if (httpClient != null) {
                    httpClient.close();
                }
            } catch (Exception e) {
                throw new OpenApiBusinessException("", "http关闭时有异常发生：" + e.getMessage());
            }
        }

    }

    public static String postRestfulRequest(AggOpenAPI aggOpenAPI, String url, String requestBody) throws NoSuchAlgorithmException, KeyManagementException {
        return postRestfulRequest(aggOpenAPI, url, requestBody, null);
    }

    public static String postRestfulRequest(AggOpenAPI aggOpenAPI, String url, String requestBody
            , String proxyHost, Integer proxyPort
            , String proxyUser, String proxyPassword) throws NoSuchAlgorithmException, KeyManagementException {
        return postRestfulRequest(aggOpenAPI, url, requestBody, null, proxyHost, proxyPort, proxyUser, proxyPassword);
    }

    public static String postRestfulRequest(AggOpenAPI aggOpenAPI, String url, String requestBody, Map<String, String> headerMap) throws NoSuchAlgorithmException, KeyManagementException {
        String appKey = aggOpenAPI.getAppKey();
        String appSecret = aggOpenAPI.getAppSecret();
        String access_token = aggOpenAPI.getAccessToken();
        if (StringUtil.isNullOrEmpty(url)) {
            throw new OpenApiBusinessException("", "url不能为空");
        }
        if (StringUtil.isNullOrEmpty(access_token)) {
            throw new OpenApiBusinessException("", "access_token不能为空");
        }
        if (StringUtil.isNullOrEmpty(appSecret)) {
            throw new OpenApiBusinessException("", "appSecret不能为空");
        }

        String apiHost = aggOpenAPI.getApiHost();
        url = apiHost + url;
        if (url.endsWith("/")) {
            url = url.substring(0, url.length() - 1);
        }
        // 签名准备
        long timestamp = System.currentTimeMillis();
        // 签名前
        String before = "POST" + "_" + MD5(requestBody) + "_" + timestamp + "_" + access_token + "_" + appSecret;
        // 签名后
        String sign = "API-SV1:" + appKey + ":" + Base64.encodeBase64String(MD5(before).getBytes());

        // header里额外参数
        if (headerMap == null) {
            headerMap = new HashMap<>();
        }
        headerMap.put("access_token", access_token);
        headerMap.put("req_date", String.valueOf(timestamp));
        headerMap.put("req_sign", sign);

        // 发送请求
        String responseStr = HttpUtil.post(url, requestBody, headerMap);

        // 得到结果
        return responseStr;
    }

    public static String postRestfulRequest(AggOpenAPI aggOpenAPI, String url, String requestBody, Map<String, String> headerMap
            , String proxyHost, Integer proxyPort
            , String proxyUser, String proxyPassword) throws NoSuchAlgorithmException, KeyManagementException {
        String appKey = aggOpenAPI.getAppKey();
        String appSecret = aggOpenAPI.getAppSecret();
        String access_token = aggOpenAPI.getAccessToken();
        if (StringUtil.isNullOrEmpty(url)) {
            throw new OpenApiBusinessException("", "url不能为空");
        }
        if (StringUtil.isNullOrEmpty(access_token)) {
            throw new OpenApiBusinessException("", "access_token不能为空");
        }
        if (StringUtil.isNullOrEmpty(appSecret)) {
            throw new OpenApiBusinessException("", "appSecret不能为空");
        }

        String apiHost = aggOpenAPI.getApiHost();
        url = apiHost + url;
        if (url.endsWith("/")) {
            url = url.substring(0, url.length() - 1);
        }
        // 签名准备
        long timestamp = System.currentTimeMillis();
        // 签名前
        String before = "POST" + "_" + MD5(requestBody) + "_" + timestamp + "_" + access_token + "_" + appSecret;
        // 签名后
        String sign = "API-SV1:" + appKey + ":" + Base64.encodeBase64String(MD5(before).getBytes());

        // header里额外参数
        if (headerMap == null) {
            headerMap = new HashMap<>();
        }
        headerMap.put("access_token", access_token);
        headerMap.put("req_date", String.valueOf(timestamp));
        headerMap.put("req_sign", sign);

        // 发送请求
        String responseStr = HttpUtil.post(url, requestBody, headerMap, proxyHost, proxyPort, proxyUser, proxyPassword);

        // 得到结果
        return responseStr;
    }

    public static String postRestfulRequest(AggOpenAPI aggOpenAPI, String url, String requestBody, Map<String, String> headerMap, int timeout, int connectTimeout, int connectionRequestTimeout) throws NoSuchAlgorithmException, KeyManagementException {
        String appKey = aggOpenAPI.getAppKey();
        String appSecret = aggOpenAPI.getAppSecret();
        String access_token = aggOpenAPI.getAccessToken();
        if (StringUtil.isNullOrEmpty(url)) {
            throw new OpenApiBusinessException("", "url不能为空");
        }
        if (StringUtil.isNullOrEmpty(access_token)) {
            throw new OpenApiBusinessException("", "access_token不能为空");
        }
        if (StringUtil.isNullOrEmpty(appSecret)) {
            throw new OpenApiBusinessException("", "appSecret不能为空");
        }

        String apiHost = aggOpenAPI.getApiHost();
        url = apiHost + url;
        if (url.endsWith("/")) {
            url = url.substring(0, url.length() - 1);
        }
        // 签名准备
        long timestamp = System.currentTimeMillis();
        // 签名前
        String before = "POST" + "_" + MD5(requestBody) + "_" + timestamp + "_" + access_token + "_" + appSecret;
        // 签名后
        String sign = "API-SV1:" + appKey + ":" + Base64.encodeBase64String(MD5(before).getBytes());

        // header里额外参数
        if (headerMap == null) {
            headerMap = new HashMap<>();
        }
        headerMap.put("access_token", access_token);
        headerMap.put("req_date", String.valueOf(timestamp));
        headerMap.put("req_sign", sign);

        // 发送请求
        String responseStr = HttpUtil.post(url, requestBody, headerMap, timeout, connectTimeout, connectionRequestTimeout);

        // 得到结果
        return responseStr;
    }

    public static String postRestfulRequest(AggOpenAPI aggOpenAPI, String url, String requestBody, Map<String, String> headerMap, int timeout
            , String proxyHost, Integer proxyPort
            , String proxyUser, String proxyPassword, int connectTimeout, int connectionRequestTimeout) throws NoSuchAlgorithmException, KeyManagementException {
        String appKey = aggOpenAPI.getAppKey();
        String appSecret = aggOpenAPI.getAppSecret();
        String access_token = aggOpenAPI.getAccessToken();
        if (StringUtil.isNullOrEmpty(url)) {
            throw new OpenApiBusinessException("", "url不能为空");
        }
        if (StringUtil.isNullOrEmpty(access_token)) {
            throw new OpenApiBusinessException("", "access_token不能为空");
        }
        if (StringUtil.isNullOrEmpty(appSecret)) {
            throw new OpenApiBusinessException("", "appSecret不能为空");
        }

        String apiHost = aggOpenAPI.getApiHost();
        url = apiHost + url;
        if (url.endsWith("/")) {
            url = url.substring(0, url.length() - 1);
        }
        // 签名准备
        long timestamp = System.currentTimeMillis();
        // 签名前
        String before = "POST" + "_" + MD5(requestBody) + "_" + timestamp + "_" + access_token + "_" + appSecret;
        // 签名后
        String sign = "API-SV1:" + appKey + ":" + Base64.encodeBase64String(MD5(before).getBytes());

        // header里额外参数
        if (headerMap == null) {
            headerMap = new HashMap<>();
        }
        headerMap.put("access_token", access_token);
        headerMap.put("req_date", String.valueOf(timestamp));
        headerMap.put("req_sign", sign);

        // 发送请求
        String responseStr = HttpUtil.post(url, requestBody, headerMap, timeout, proxyHost, proxyPort, proxyUser, proxyPassword, connectTimeout, connectionRequestTimeout);

        // 得到结果
        return responseStr;
    }

    /**
     * 计算MD5值
     * 
     * @param sourceStr sourceStr
     * @return sourceStr
     */
    public static String MD5(String sourceStr) {
        String result = "";
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            md.update(sourceStr.getBytes("UTF-8"));
            byte b[] = md.digest();
            int i;
            StringBuffer buf = new StringBuffer("");
            for (int offset = 0; offset < b.length; offset++) {
                i = b[offset];
                if (i < 0) {
                    i += 256;
                }
                if (i < 16) {
                    buf.append("0");
                }

                buf.append(Integer.toHexString(i));
            }
            result = buf.toString();
        } catch (NoSuchAlgorithmException e) {
            throw new OpenApiBusinessException("", e.getMessage());
        } catch (UnsupportedEncodingException e) {
            throw new OpenApiBusinessException("", e.getMessage());
        }
        return result;
    }

}
