/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.util;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.WritableByteChannel;
import java.nio.file.AccessDeniedException;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.util.Random;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import org.apache.flink.core.fs.FSDataInputStream;
import org.apache.flink.core.fs.FSDataOutputStream;
import org.apache.flink.core.fs.FileStatus;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.util.IOUtils;
import org.apache.flink.util.OperatingSystem;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.function.ThrowingConsumer;

public final class FileUtils {
    private static final Object WINDOWS_DELETE_LOCK = new Object();
    private static final char[] ALPHABET = new char[]{'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', 'a', 'b', 'c', 'd', 'e', 'f'};
    private static final int RANDOM_FILE_NAME_LENGTH = 12;

    public static void writeCompletely(WritableByteChannel channel, ByteBuffer src) throws IOException {
        while (src.hasRemaining()) {
            channel.write(src);
        }
    }

    public static String getRandomFilename(String prefix) {
        Random rnd = new Random();
        StringBuilder stringBuilder = new StringBuilder(prefix);
        for (int i = 0; i < 12; ++i) {
            stringBuilder.append(ALPHABET[rnd.nextInt(ALPHABET.length)]);
        }
        return stringBuilder.toString();
    }

    public static String readFile(File file, String charsetName) throws IOException {
        byte[] bytes = Files.readAllBytes(file.toPath());
        return new String(bytes, charsetName);
    }

    public static String readFileUtf8(File file) throws IOException {
        return FileUtils.readFile(file, "UTF-8");
    }

    public static void writeFile(File file, String contents, String encoding) throws IOException {
        byte[] bytes = contents.getBytes(encoding);
        Files.write(file.toPath(), bytes, StandardOpenOption.WRITE);
    }

    public static void writeFileUtf8(File file, String contents) throws IOException {
        FileUtils.writeFile(file, contents, "UTF-8");
    }

    public static void deleteFileOrDirectory(File file) throws IOException {
        Preconditions.checkNotNull(file, "file");
        FileUtils.guardIfWindows(FileUtils::deleteFileOrDirectoryInternal, file);
    }

    public static void deleteDirectory(File directory) throws IOException {
        Preconditions.checkNotNull(directory, "directory");
        FileUtils.guardIfWindows(FileUtils::deleteDirectoryInternal, directory);
    }

    public static void deleteDirectoryQuietly(File directory) {
        if (directory == null) {
            return;
        }
        try {
            FileUtils.deleteDirectory(directory);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    public static void cleanDirectory(File directory) throws IOException {
        Preconditions.checkNotNull(directory, "directory");
        FileUtils.guardIfWindows(FileUtils::cleanDirectoryInternal, directory);
    }

    private static void deleteFileOrDirectoryInternal(File file) throws IOException {
        if (file.isDirectory()) {
            FileUtils.deleteDirectoryInternal(file);
        } else {
            Files.deleteIfExists(file.toPath());
        }
    }

    private static void deleteDirectoryInternal(File directory) throws IOException {
        if (directory.isDirectory()) {
            try {
                FileUtils.cleanDirectoryInternal(directory);
            }
            catch (FileNotFoundException ignored) {
                return;
            }
            Files.deleteIfExists(directory.toPath());
        } else if (directory.exists()) {
            throw new IOException(directory + " is not a directory");
        }
    }

    private static void cleanDirectoryInternal(File directory) throws IOException {
        if (directory.isDirectory()) {
            File[] files = directory.listFiles();
            if (files == null) {
                if (directory.exists()) {
                    throw new IOException("Failed to list contents of " + directory);
                }
                throw new FileNotFoundException(directory.toString());
            }
            for (File file : files) {
                if (file == null) continue;
                FileUtils.deleteFileOrDirectory(file);
            }
        } else {
            if (directory.exists()) {
                throw new IOException(directory + " is not a directory but a regular file");
            }
            throw new FileNotFoundException(directory.toString());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void guardIfWindows(ThrowingConsumer<File, IOException> toRun, File file) throws IOException {
        if (!OperatingSystem.isWindows()) {
            toRun.accept(file);
        } else {
            Object object = WINDOWS_DELETE_LOCK;
            synchronized (object) {
                for (int attempt = 1; attempt <= 10; ++attempt) {
                    try {
                        toRun.accept(file);
                        break;
                    }
                    catch (AccessDeniedException accessDeniedException) {
                        try {
                            Thread.sleep(1L);
                            continue;
                        }
                        catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                            throw new IOException("operation interrupted");
                        }
                    }
                }
            }
        }
    }

    public static boolean deletePathIfEmpty(FileSystem fileSystem, Path path) throws IOException {
        FileStatus[] fileStatuses;
        try {
            fileStatuses = fileSystem.listStatus(path);
        }
        catch (FileNotFoundException e) {
            return true;
        }
        catch (Exception e) {
            return false;
        }
        if (fileStatuses == null) {
            return true;
        }
        if (fileStatuses.length == 0) {
            return fileSystem.delete(path, false);
        }
        return false;
    }

    public static void copy(Path sourcePath, Path targetPath, boolean executable) throws IOException {
        FileSystem sFS = FileSystem.getUnguardedFileSystem(sourcePath.toUri());
        FileSystem tFS = FileSystem.getUnguardedFileSystem(targetPath.toUri());
        if (!tFS.exists(targetPath)) {
            if (sFS.getFileStatus(sourcePath).isDir()) {
                FileUtils.internalCopyDirectory(sourcePath, targetPath, executable, sFS, tFS);
            } else {
                FileUtils.internalCopyFile(sourcePath, targetPath, executable, sFS, tFS);
            }
        }
    }

    private static void internalCopyDirectory(Path sourcePath, Path targetPath, boolean executable, FileSystem sFS, FileSystem tFS) throws IOException {
        FileStatus[] contents;
        tFS.mkdirs(targetPath);
        for (FileStatus content : contents = sFS.listStatus(sourcePath)) {
            String distPath = content.getPath().toString();
            if (content.isDir() && distPath.endsWith("/")) {
                distPath = distPath.substring(0, distPath.length() - 1);
            }
            String localPath = targetPath + distPath.substring(distPath.lastIndexOf("/"));
            FileUtils.copy(content.getPath(), new Path(localPath), executable);
        }
    }

    private static void internalCopyFile(Path sourcePath, Path targetPath, boolean executable, FileSystem sFS, FileSystem tFS) throws IOException {
        try (FSDataOutputStream lfsOutput = tFS.create(targetPath, FileSystem.WriteMode.NO_OVERWRITE);
             FSDataInputStream fsInput = sFS.open(sourcePath);){
            IOUtils.copyBytes(fsInput, lfsOutput);
            new File(targetPath.toString()).setExecutable(executable);
        }
    }

    public static Path compressDirectory(Path directory, Path target) throws IOException {
        FileSystem sourceFs = directory.getFileSystem();
        FileSystem targetFs = target.getFileSystem();
        try (ZipOutputStream out = new ZipOutputStream(targetFs.create(target, FileSystem.WriteMode.NO_OVERWRITE));){
            FileUtils.addToZip(directory, sourceFs, directory.getParent(), out);
        }
        return target;
    }

    private static void addToZip(Path fileOrDirectory, FileSystem fs, Path rootDir, ZipOutputStream out) throws IOException {
        String relativePath = fileOrDirectory.getPath().replace(rootDir.getPath() + '/', "");
        if (fs.getFileStatus(fileOrDirectory).isDir()) {
            out.putNextEntry(new ZipEntry(relativePath + '/'));
            for (FileStatus containedFile : fs.listStatus(fileOrDirectory)) {
                FileUtils.addToZip(containedFile.getPath(), fs, rootDir, out);
            }
        } else {
            ZipEntry entry = new ZipEntry(relativePath);
            out.putNextEntry(entry);
            try (FSDataInputStream in = fs.open(fileOrDirectory);){
                IOUtils.copyBytes(in, out, false);
            }
            out.closeEntry();
        }
    }

    public static Path expandDirectory(Path file, Path targetDirectory) throws IOException {
        FileSystem sourceFs = file.getFileSystem();
        FileSystem targetFs = targetDirectory.getFileSystem();
        Path rootDir = null;
        try (ZipInputStream zis = new ZipInputStream(sourceFs.open(file));){
            ZipEntry entry;
            while ((entry = zis.getNextEntry()) != null) {
                Path relativePath = new Path(entry.getName());
                if (rootDir == null) {
                    rootDir = relativePath;
                }
                Path newFile = new Path(targetDirectory, relativePath);
                if (entry.isDirectory()) {
                    targetFs.mkdirs(newFile);
                } else {
                    try (FSDataOutputStream fileStream = targetFs.create(newFile, FileSystem.WriteMode.NO_OVERWRITE);){
                        IOUtils.copyBytes(zis, fileStream, false);
                    }
                }
                zis.closeEntry();
            }
        }
        return new Path(targetDirectory, rootDir);
    }

    private FileUtils() {
    }
}

