/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.net;

import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.net.BindException;
import java.net.ConnectException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.NetworkInterface;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketAddress;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.List;
import java.util.concurrent.TimeUnit;
import junit.framework.AssertionFailedError;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.net.NetUtils;
import org.apache.hadoop.net.SocketInputWrapper;
import org.apache.hadoop.security.NetUtilsTestResolver;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

public class TestNetUtils {
    private static final Log LOG = LogFactory.getLog(TestNetUtils.class);
    private static final int DEST_PORT = 4040;
    private static final String DEST_PORT_NAME = Integer.toString(4040);
    private static final int LOCAL_PORT = 8080;
    private static final String LOCAL_PORT_NAME = Integer.toString(8080);
    static final long TIME_FUDGE_MILLIS = 200L;
    static NetUtilsTestResolver resolver;
    static Configuration config;

    @Test
    public void testAvoidLoopbackTcpSockets() throws Exception {
        Configuration conf = new Configuration();
        Socket socket = NetUtils.getDefaultSocketFactory((Configuration)conf).createSocket();
        socket.bind(new InetSocketAddress("127.0.0.1", 0));
        System.err.println("local address: " + socket.getLocalAddress());
        System.err.println("local port: " + socket.getLocalPort());
        try {
            NetUtils.connect((Socket)socket, (SocketAddress)new InetSocketAddress(socket.getLocalAddress(), socket.getLocalPort()), (int)20000);
            socket.close();
            Assert.fail((String)"Should not have connected");
        }
        catch (ConnectException ce) {
            System.err.println("Got exception: " + ce);
            Assert.assertTrue((boolean)ce.getMessage().contains("resulted in a loopback"));
        }
        catch (SocketException se) {
            Assert.assertTrue((boolean)se.getMessage().contains("Invalid argument"));
        }
    }

    @Test
    public void testSocketReadTimeoutWithChannel() throws Exception {
        this.doSocketReadTimeoutTest(true);
    }

    @Test
    public void testSocketReadTimeoutWithoutChannel() throws Exception {
        this.doSocketReadTimeoutTest(false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doSocketReadTimeoutTest(boolean withChannel) throws IOException {
        SocketInputWrapper stm;
        Socket s;
        ServerSocket ss;
        block5: {
            ss = new ServerSocket(0);
            if (withChannel) {
                s = NetUtils.getDefaultSocketFactory((Configuration)new Configuration()).createSocket();
                Assume.assumeNotNull((Object[])new Object[]{s.getChannel()});
            } else {
                s = new Socket();
                Assert.assertNull((Object)s.getChannel());
            }
            stm = null;
            try {
                NetUtils.connect((Socket)s, (SocketAddress)ss.getLocalSocketAddress(), (int)1000);
                stm = NetUtils.getInputStream((Socket)s, (long)1000L);
                this.assertReadTimeout(stm, 1000);
                stm.setTimeout(1L);
                this.assertReadTimeout(stm, 1);
                s.setSoTimeout(1000);
                if (withChannel) {
                    this.assertReadTimeout(stm, 1);
                    break block5;
                }
                this.assertReadTimeout(stm, 1000);
            }
            catch (Throwable throwable) {
                IOUtils.closeStream(stm);
                IOUtils.closeSocket((Socket)s);
                ss.close();
                throw throwable;
            }
        }
        IOUtils.closeStream((Closeable)stm);
        IOUtils.closeSocket((Socket)s);
        ss.close();
    }

    private void assertReadTimeout(SocketInputWrapper stm, int timeoutMillis) throws IOException {
        long st = System.nanoTime();
        try {
            stm.read();
            Assert.fail((String)"Didn't time out");
        }
        catch (SocketTimeoutException ste) {
            this.assertTimeSince(st, timeoutMillis);
        }
    }

    private void assertTimeSince(long startNanos, int expectedMillis) {
        long durationNano = System.nanoTime() - startNanos;
        long millis = TimeUnit.MILLISECONDS.convert(durationNano, TimeUnit.NANOSECONDS);
        Assert.assertTrue((String)("Expected " + expectedMillis + "ms, but took " + millis), (Math.abs(millis - (long)expectedMillis) < 200L ? 1 : 0) != 0);
    }

    @Test
    public void testGetLocalInetAddress() throws Exception {
        Assert.assertNotNull((Object)NetUtils.getLocalInetAddress((String)"127.0.0.1"));
        Assert.assertNull((Object)NetUtils.getLocalInetAddress((String)"invalid-address-for-test"));
        Assert.assertNull((Object)NetUtils.getLocalInetAddress(null));
    }

    @Test(expected=UnknownHostException.class)
    public void testVerifyHostnamesException() throws UnknownHostException {
        String[] names = new String[]{"valid.host.com", "1.com", "invalid host here"};
        NetUtils.verifyHostnames((String[])names);
    }

    @Test
    public void testVerifyHostnamesNoException() {
        String[] names = new String[]{"valid.host.com", "1.com"};
        try {
            NetUtils.verifyHostnames((String[])names);
        }
        catch (UnknownHostException e) {
            Assert.fail((String)"NetUtils.verifyHostnames threw unexpected UnknownHostException");
        }
    }

    @Test
    public void testIsLocalAddress() throws Exception {
        Assert.assertTrue((boolean)NetUtils.isLocalAddress((InetAddress)InetAddress.getLocalHost()));
        Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
        if (interfaces != null) {
            while (interfaces.hasMoreElements()) {
                NetworkInterface i = interfaces.nextElement();
                Enumeration<InetAddress> addrs = i.getInetAddresses();
                if (addrs == null) continue;
                while (addrs.hasMoreElements()) {
                    InetAddress addr = addrs.nextElement();
                    Assert.assertTrue((boolean)NetUtils.isLocalAddress((InetAddress)addr));
                }
            }
        }
        Assert.assertFalse((boolean)NetUtils.isLocalAddress((InetAddress)InetAddress.getByName("8.8.8.8")));
    }

    @Test
    public void testWrapConnectException() throws Throwable {
        ConnectException e = new ConnectException("failed");
        IOException wrapped = this.verifyExceptionClass(e, ConnectException.class);
        this.assertInException(wrapped, "failed");
        this.assertWikified(wrapped);
        this.assertInException(wrapped, "localhost");
        this.assertRemoteDetailsIncluded(wrapped);
        this.assertInException(wrapped, "/ConnectionRefused");
    }

    @Test
    public void testWrapBindException() throws Throwable {
        BindException e = new BindException("failed");
        IOException wrapped = this.verifyExceptionClass(e, BindException.class);
        this.assertInException(wrapped, "failed");
        this.assertLocalDetailsIncluded(wrapped);
        this.assertNotInException(wrapped, DEST_PORT_NAME);
        this.assertInException(wrapped, "/BindException");
    }

    @Test
    public void testWrapUnknownHostException() throws Throwable {
        UnknownHostException e = new UnknownHostException("failed");
        IOException wrapped = this.verifyExceptionClass(e, UnknownHostException.class);
        this.assertInException(wrapped, "failed");
        this.assertWikified(wrapped);
        this.assertInException(wrapped, "localhost");
        this.assertRemoteDetailsIncluded(wrapped);
        this.assertInException(wrapped, "/UnknownHost");
    }

    @Test
    public void testWrapEOFException() throws Throwable {
        EOFException e = new EOFException("eof");
        IOException wrapped = this.verifyExceptionClass(e, EOFException.class);
        this.assertInException(wrapped, "eof");
        this.assertWikified(wrapped);
        this.assertInException(wrapped, "localhost");
        this.assertRemoteDetailsIncluded(wrapped);
        this.assertInException(wrapped, "/EOFException");
    }

    @Test
    public void testGetConnectAddress() throws IOException {
        NetUtils.addStaticResolution((String)"host", (String)"127.0.0.1");
        InetSocketAddress addr = NetUtils.createSocketAddrForHost((String)"host", (int)1);
        InetSocketAddress connectAddr = NetUtils.getConnectAddress((InetSocketAddress)addr);
        Assert.assertEquals((Object)addr.getHostName(), (Object)connectAddr.getHostName());
        addr = new InetSocketAddress(1);
        connectAddr = NetUtils.getConnectAddress((InetSocketAddress)addr);
        Assert.assertEquals((Object)InetAddress.getLocalHost().getHostName(), (Object)connectAddr.getHostName());
    }

    @Test
    public void testCreateSocketAddress() throws Throwable {
        InetSocketAddress addr = NetUtils.createSocketAddr((String)"127.0.0.1:12345", (int)1000, (String)"myconfig");
        Assert.assertEquals((Object)"127.0.0.1", (Object)addr.getAddress().getHostAddress());
        Assert.assertEquals((long)12345L, (long)addr.getPort());
        addr = NetUtils.createSocketAddr((String)"127.0.0.1", (int)1000, (String)"myconfig");
        Assert.assertEquals((Object)"127.0.0.1", (Object)addr.getAddress().getHostAddress());
        Assert.assertEquals((long)1000L, (long)addr.getPort());
        try {
            addr = NetUtils.createSocketAddr((String)"127.0.0.1:blahblah", (int)1000, (String)"myconfig");
            Assert.fail((String)"Should have failed to parse bad port");
        }
        catch (IllegalArgumentException iae) {
            this.assertInException(iae, "myconfig");
        }
    }

    private void assertRemoteDetailsIncluded(IOException wrapped) throws Throwable {
        this.assertInException(wrapped, "desthost");
        this.assertInException(wrapped, DEST_PORT_NAME);
    }

    private void assertLocalDetailsIncluded(IOException wrapped) throws Throwable {
        this.assertInException(wrapped, "localhost");
        this.assertInException(wrapped, LOCAL_PORT_NAME);
    }

    private void assertWikified(Exception e) throws Throwable {
        this.assertInException(e, "http://wiki.apache.org/hadoop/");
    }

    private void assertInException(Exception e, String text) throws Throwable {
        String message = this.extractExceptionMessage(e);
        if (!message.contains(text)) {
            throw new AssertionFailedError("Wrong text in message \"" + message + "\"" + " expected \"" + text + "\"").initCause((Throwable)e);
        }
    }

    private String extractExceptionMessage(Exception e) throws Throwable {
        Assert.assertNotNull((String)"Null Exception", (Object)e);
        String message = e.getMessage();
        if (message == null) {
            throw new AssertionFailedError("Empty text in exception " + e).initCause((Throwable)e);
        }
        return message;
    }

    private void assertNotInException(Exception e, String text) throws Throwable {
        String message = this.extractExceptionMessage(e);
        if (message.contains(text)) {
            throw new AssertionFailedError("Wrong text in message \"" + message + "\"" + " did not expect \"" + text + "\"").initCause((Throwable)e);
        }
    }

    private IOException verifyExceptionClass(IOException e, Class expectedClass) throws Throwable {
        Assert.assertNotNull((String)"Null Exception", (Object)e);
        IOException wrapped = NetUtils.wrapException((String)"desthost", (int)4040, (String)"localhost", (int)8080, (IOException)e);
        LOG.info((Object)wrapped.toString(), (Throwable)wrapped);
        if (!wrapped.getClass().equals(expectedClass)) {
            throw new AssertionFailedError("Wrong exception class; expected " + expectedClass + " got " + wrapped.getClass() + ": " + wrapped).initCause((Throwable)wrapped);
        }
        return wrapped;
    }

    @BeforeClass
    public static void setupResolver() {
        resolver = NetUtilsTestResolver.install();
    }

    @Before
    public void resetResolver() {
        resolver.reset();
        config = new Configuration();
    }

    private void verifyGetByExactNameSearch(String host, String ... searches) {
        Assert.assertNull((Object)resolver.getByExactName(host));
        this.assertBetterArrayEquals(searches, resolver.getHostSearches());
    }

    @Test
    public void testResolverGetByExactNameUnqualified() {
        this.verifyGetByExactNameSearch("unknown", "unknown.");
    }

    @Test
    public void testResolverGetByExactNameUnqualifiedWithDomain() {
        this.verifyGetByExactNameSearch("unknown.domain", "unknown.domain.");
    }

    @Test
    public void testResolverGetByExactNameQualified() {
        this.verifyGetByExactNameSearch("unknown.", "unknown.");
    }

    @Test
    public void testResolverGetByExactNameQualifiedWithDomain() {
        this.verifyGetByExactNameSearch("unknown.domain.", "unknown.domain.");
    }

    private void verifyGetByNameWithSearch(String host, String ... searches) {
        Assert.assertNull((Object)resolver.getByNameWithSearch(host));
        this.assertBetterArrayEquals(searches, resolver.getHostSearches());
    }

    @Test
    public void testResolverGetByNameWithSearchUnqualified() {
        String host = "unknown";
        this.verifyGetByNameWithSearch(host, host + ".a.b.", host + ".b.", host + ".c.");
    }

    @Test
    public void testResolverGetByNameWithSearchUnqualifiedWithDomain() {
        String host = "unknown.domain";
        this.verifyGetByNameWithSearch(host, host + ".a.b.", host + ".b.", host + ".c.");
    }

    @Test
    public void testResolverGetByNameWithSearchQualified() {
        String host = "unknown.";
        this.verifyGetByNameWithSearch(host, host);
    }

    @Test
    public void testResolverGetByNameWithSearchQualifiedWithDomain() {
        String host = "unknown.domain.";
        this.verifyGetByNameWithSearch(host, host);
    }

    private void verifyGetByName(String host, String ... searches) {
        InetAddress addr = null;
        try {
            addr = resolver.getByName(host);
        }
        catch (UnknownHostException unknownHostException) {
            // empty catch block
        }
        Assert.assertNull((Object)addr);
        this.assertBetterArrayEquals(searches, resolver.getHostSearches());
    }

    @Test
    public void testResolverGetByNameQualified() {
        String host = "unknown.";
        this.verifyGetByName(host, host);
    }

    @Test
    public void testResolverGetByNameQualifiedWithDomain() {
        this.verifyGetByName("unknown.domain.", "unknown.domain.");
    }

    @Test
    public void testResolverGetByNameUnqualified() {
        String host = "unknown";
        this.verifyGetByName(host, host + ".a.b.", host + ".b.", host + ".c.", host + ".");
    }

    @Test
    public void testResolverGetByNameUnqualifiedWithDomain() {
        String host = "unknown.domain";
        this.verifyGetByName(host, host + ".", host + ".a.b.", host + ".b.", host + ".c.");
    }

    private InetAddress verifyResolve(String host, String ... searches) {
        InetAddress addr = null;
        try {
            addr = resolver.getByName(host);
        }
        catch (UnknownHostException unknownHostException) {
            // empty catch block
        }
        Assert.assertNotNull((Object)addr);
        this.assertBetterArrayEquals(searches, resolver.getHostSearches());
        return addr;
    }

    private void verifyInetAddress(InetAddress addr, String host, String ip) {
        Assert.assertNotNull((Object)addr);
        Assert.assertEquals((Object)host, (Object)addr.getHostName());
        Assert.assertEquals((Object)ip, (Object)addr.getHostAddress());
    }

    @Test
    public void testResolverUnqualified() {
        String host = "host";
        InetAddress addr = this.verifyResolve(host, host + ".a.b.");
        this.verifyInetAddress(addr, "host.a.b", "1.1.1.1");
    }

    @Test
    public void testResolverUnqualifiedWithDomain() {
        String host = "host.a";
        InetAddress addr = this.verifyResolve(host, host + ".", host + ".a.b.", host + ".b.");
        this.verifyInetAddress(addr, "host.a.b", "1.1.1.1");
    }

    @Test
    public void testResolverUnqualifedFull() {
        String host = "host.a.b";
        InetAddress addr = this.verifyResolve(host, host + ".");
        this.verifyInetAddress(addr, host, "1.1.1.1");
    }

    @Test
    public void testResolverQualifed() {
        String host = "host.a.b.";
        InetAddress addr = this.verifyResolve(host, host);
        this.verifyInetAddress(addr, host, "1.1.1.1");
    }

    @Test
    public void testResolverLoopback() {
        String host = "Localhost";
        InetAddress addr = this.verifyResolve(host, new String[0]);
        this.verifyInetAddress(addr, "Localhost", "127.0.0.1");
    }

    @Test
    public void testResolverIP() {
        String host = "1.1.1.1";
        InetAddress addr = this.verifyResolve(host, new String[0]);
        this.verifyInetAddress(addr, host, host);
    }

    @Test
    public void testCanonicalUriWithPort() {
        URI uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host:123"), (int)456);
        Assert.assertEquals((Object)"scheme://host.a.b:123", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host:123/"), (int)456);
        Assert.assertEquals((Object)"scheme://host.a.b:123/", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host:123/path"), (int)456);
        Assert.assertEquals((Object)"scheme://host.a.b:123/path", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host:123/path?q#frag"), (int)456);
        Assert.assertEquals((Object)"scheme://host.a.b:123/path?q#frag", (Object)uri.toString());
    }

    @Test
    public void testCanonicalUriWithDefaultPort() {
        URI uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host"), (int)123);
        Assert.assertEquals((Object)"scheme://host.a.b:123", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host/"), (int)123);
        Assert.assertEquals((Object)"scheme://host.a.b:123/", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host/path"), (int)123);
        Assert.assertEquals((Object)"scheme://host.a.b:123/path", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host/path?q#frag"), (int)123);
        Assert.assertEquals((Object)"scheme://host.a.b:123/path?q#frag", (Object)uri.toString());
    }

    @Test
    public void testCanonicalUriWithPath() {
        URI uri = NetUtils.getCanonicalUri((URI)URI.create("path"), (int)2);
        Assert.assertEquals((Object)"path", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("/path"), (int)2);
        Assert.assertEquals((Object)"/path", (Object)uri.toString());
    }

    @Test
    public void testCanonicalUriWithNoAuthority() {
        URI uri = NetUtils.getCanonicalUri((URI)URI.create("scheme:/"), (int)2);
        Assert.assertEquals((Object)"scheme:/", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme:/path"), (int)2);
        Assert.assertEquals((Object)"scheme:/path", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme:///"), (int)2);
        Assert.assertEquals((Object)"scheme:///", (Object)uri.toString());
        uri = NetUtils.getCanonicalUri((URI)URI.create("scheme:///path"), (int)2);
        Assert.assertEquals((Object)"scheme:///path", (Object)uri.toString());
    }

    @Test
    public void testCanonicalUriWithNoHost() {
        URI uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://:123/path"), (int)2);
        Assert.assertEquals((Object)"scheme://:123/path", (Object)uri.toString());
    }

    @Test
    public void testCanonicalUriWithNoPortNoDefaultPort() {
        URI uri = NetUtils.getCanonicalUri((URI)URI.create("scheme://host/path"), (int)-1);
        Assert.assertEquals((Object)"scheme://host.a.b/path", (Object)uri.toString());
    }

    @Test
    public void testNormalizeHostName() {
        List<String> hosts = Arrays.asList("127.0.0.1", "localhost", "1.kanyezone.appspot.com", "UnknownHost123");
        List normalizedHosts = NetUtils.normalizeHostNames(hosts);
        Assert.assertEquals(normalizedHosts.get(0), (Object)hosts.get(0));
        Assert.assertFalse((boolean)((String)normalizedHosts.get(1)).equals(hosts.get(1)));
        Assert.assertEquals(normalizedHosts.get(1), (Object)hosts.get(0));
        Assert.assertFalse((boolean)((String)normalizedHosts.get(2)).equals(hosts.get(2)));
        Assert.assertEquals(normalizedHosts.get(3), (Object)hosts.get(3));
    }

    @Test
    public void testGetHostNameOfIP() {
        Assert.assertNull((Object)NetUtils.getHostNameOfIP(null));
        Assert.assertNull((Object)NetUtils.getHostNameOfIP((String)""));
        Assert.assertNull((Object)NetUtils.getHostNameOfIP((String)"crazytown"));
        Assert.assertNull((Object)NetUtils.getHostNameOfIP((String)"127.0.0.1:"));
        Assert.assertNull((Object)NetUtils.getHostNameOfIP((String)"127.0.0.1:-1"));
        Assert.assertNull((Object)NetUtils.getHostNameOfIP((String)"127.0.0.1:A"));
        Assert.assertNotNull((Object)NetUtils.getHostNameOfIP((String)"127.0.0.1"));
        Assert.assertNotNull((Object)NetUtils.getHostNameOfIP((String)"127.0.0.1:1"));
    }

    @Test
    public void testTrimCreateSocketAddress() {
        Configuration conf = new Configuration();
        NetUtils.addStaticResolution((String)"host", (String)"127.0.0.1");
        String defaultAddr = "host:1  ";
        InetSocketAddress addr = NetUtils.createSocketAddr((String)"host:1  ");
        conf.setSocketAddr("myAddress", addr);
        Assert.assertEquals((Object)"host:1  ".trim(), (Object)NetUtils.getHostPortString((InetSocketAddress)addr));
    }

    private <T> void assertBetterArrayEquals(T[] expect, T[] got) {
        String expectStr = StringUtils.join((Object[])expect, (String)", ");
        String gotStr = StringUtils.join((Object[])got, (String)", ");
        Assert.assertEquals((Object)expectStr, (Object)gotStr);
    }
}

