/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.rest;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.nio.charset.Charset;
import java.security.AccessControlException;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import org.apache.lucene.util.BytesRef;
import org.apache.solr.cloud.ZkSolrResourceLoader;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.cloud.SolrZkClient;
import org.apache.solr.common.util.NamedList;
import org.apache.solr.core.SolrResourceLoader;
import org.noggit.JSONParser;
import org.noggit.JSONUtil;
import org.noggit.ObjectBuilder;
import org.restlet.data.Status;
import org.restlet.resource.ResourceException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class ManagedResourceStorage {
    public static final String STORAGE_IO_CLASS_INIT_ARG = "storageIO";
    public static final String STORAGE_DIR_INIT_ARG = "storageDir";
    public static final Logger log = LoggerFactory.getLogger(ManagedResourceStorage.class);
    public static final Charset UTF_8 = Charset.forName("UTF-8");
    protected StorageIO storageIO;
    protected SolrResourceLoader loader;

    public static StorageIO newStorageIO(String collection, SolrResourceLoader resourceLoader, NamedList<String> initArgs) {
        StorageIO storageIO = null;
        SolrZkClient zkClient = null;
        String zkConfigName = null;
        if (resourceLoader instanceof ZkSolrResourceLoader) {
            zkClient = ((ZkSolrResourceLoader)resourceLoader).getZkController().getZkClient();
            try {
                zkConfigName = ((ZkSolrResourceLoader)resourceLoader).getZkController().getZkStateReader().readConfigName(collection);
            }
            catch (Exception e) {
                log.error("Failed to get config name for collection {} due to: {}", (Object)collection, (Object)e.toString());
            }
            if (zkConfigName == null) {
                throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Could not find config name for collection:" + collection);
            }
        }
        if (initArgs.get(STORAGE_IO_CLASS_INIT_ARG) != null) {
            storageIO = resourceLoader.newInstance((String)initArgs.get(STORAGE_IO_CLASS_INIT_ARG), StorageIO.class);
        } else if (zkClient != null) {
            String znodeBase = "/configs/" + zkConfigName;
            log.info("Setting up ZooKeeper-based storage for the RestManager with znodeBase: " + znodeBase);
            storageIO = new ZooKeeperStorageIO(zkClient, znodeBase);
        } else {
            storageIO = new FileStorageIO();
        }
        if (storageIO instanceof FileStorageIO && initArgs.get(STORAGE_DIR_INIT_ARG) == null) {
            File configDir = new File(resourceLoader.getConfigDir());
            boolean hasAccess = false;
            try {
                hasAccess = configDir.isDirectory() && configDir.canWrite();
            }
            catch (AccessControlException ace) {
                // empty catch block
            }
            if (hasAccess) {
                initArgs.add(STORAGE_DIR_INIT_ARG, (Object)configDir.getAbsolutePath());
            } else {
                log.warn("Cannot write to config directory " + configDir.getAbsolutePath() + "; switching to use InMemory storage instead.");
                storageIO = new InMemoryStorageIO();
            }
        }
        storageIO.configure(resourceLoader, initArgs);
        return storageIO;
    }

    protected ManagedResourceStorage(StorageIO storageIO, SolrResourceLoader loader) {
        this.storageIO = storageIO;
        this.loader = loader;
    }

    public SolrResourceLoader getResourceLoader() {
        return this.loader;
    }

    public StorageIO getStorageIO() {
        return this.storageIO;
    }

    public abstract String getStoredResourceId(String var1);

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Object load(String resourceId) throws IOException {
        String storedResourceId = this.getStoredResourceId(resourceId);
        log.info("Reading {} using {}", (Object)storedResourceId, (Object)this.storageIO.getInfo());
        InputStream inputStream = this.storageIO.openInputStream(storedResourceId);
        if (inputStream == null) {
            return null;
        }
        Object parsed = null;
        InputStreamReader reader = null;
        try {
            reader = new InputStreamReader(inputStream, UTF_8);
            parsed = this.parseText(reader, resourceId);
        }
        finally {
            if (reader != null) {
                try {
                    reader.close();
                }
                catch (Exception ignore) {}
            }
        }
        String objectType = parsed != null ? parsed.getClass().getSimpleName() : "null";
        log.info(String.format(Locale.ROOT, "Loaded %s at path %s using %s", objectType, storedResourceId, this.storageIO.getInfo()));
        return parsed;
    }

    protected Object parseText(Reader reader, String resourceId) throws IOException {
        return null;
    }

    public abstract void store(String var1, Object var2) throws IOException;

    public boolean delete(String resourceId) throws IOException {
        return this.storageIO.delete(this.getStoredResourceId(resourceId));
    }

    public static class JsonStorage
    extends ManagedResourceStorage {
        public JsonStorage(StorageIO storageIO, SolrResourceLoader loader) {
            super(storageIO, loader);
        }

        @Override
        public String getStoredResourceId(String resourceId) {
            return resourceId.replace('/', '_') + ".json";
        }

        @Override
        protected Object parseText(Reader reader, String resourceId) throws IOException {
            return ObjectBuilder.getVal((JSONParser)new JSONParser(reader));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void store(String resourceId, Object toStore) throws IOException {
            String json = JSONUtil.toJSON((Object)toStore);
            String storedResourceId = this.getStoredResourceId(resourceId);
            OutputStreamWriter writer = null;
            try {
                writer = new OutputStreamWriter(this.storageIO.openOutputStream(storedResourceId), UTF_8);
                writer.write(json);
                writer.flush();
            }
            finally {
                if (writer != null) {
                    try {
                        writer.close();
                    }
                    catch (Exception ignore) {}
                }
            }
            log.info("Saved JSON object to path {} using {}", (Object)storedResourceId, (Object)this.storageIO.getInfo());
        }
    }

    public static class InMemoryStorageIO
    implements StorageIO {
        Map<String, BytesRef> storage = new HashMap<String, BytesRef>();

        @Override
        public void configure(SolrResourceLoader loader, NamedList<String> initArgs) throws SolrException {
        }

        @Override
        public boolean exists(String storedResourceId) throws IOException {
            return this.storage.containsKey(storedResourceId);
        }

        @Override
        public InputStream openInputStream(String storedResourceId) throws IOException {
            BytesRef storedVal = this.storage.get(storedResourceId);
            if (storedVal == null) {
                throw new FileNotFoundException(storedResourceId);
            }
            return new ByteArrayInputStream(storedVal.bytes, storedVal.offset, storedVal.length);
        }

        @Override
        public OutputStream openOutputStream(final String storedResourceId) throws IOException {
            ByteArrayOutputStream boas = new ByteArrayOutputStream(){

                @Override
                public void close() {
                    InMemoryStorageIO.this.storage.put(storedResourceId, new BytesRef(this.toByteArray()));
                }
            };
            return boas;
        }

        @Override
        public boolean delete(String storedResourceId) throws IOException {
            return this.storage.remove(storedResourceId) != null;
        }

        @Override
        public String getInfo() {
            return "InMemoryStorage";
        }
    }

    public static class ZooKeeperStorageIO
    implements StorageIO {
        protected SolrZkClient zkClient;
        protected String znodeBase;
        protected boolean retryOnConnLoss = true;

        public ZooKeeperStorageIO(SolrZkClient zkClient, String znodeBase) {
            this.zkClient = zkClient;
            this.znodeBase = znodeBase;
        }

        @Override
        public void configure(SolrResourceLoader loader, NamedList<String> initArgs) throws SolrException {
            try {
                if (!this.zkClient.exists(this.znodeBase, this.retryOnConnLoss).booleanValue()) {
                    this.zkClient.makePath(this.znodeBase, this.retryOnConnLoss);
                }
            }
            catch (Exception exc) {
                String errMsg = String.format(Locale.ROOT, "Failed to verify znode at %s due to: %s", this.znodeBase, exc.toString());
                log.error(errMsg, (Throwable)exc);
                throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, errMsg, (Throwable)exc);
            }
            log.info("Configured ZooKeeperStorageIO with znodeBase: " + this.znodeBase);
        }

        @Override
        public boolean exists(String storedResourceId) throws IOException {
            String znodePath = this.getZnodeForResource(storedResourceId);
            try {
                return this.zkClient.exists(znodePath, this.retryOnConnLoss);
            }
            catch (Exception e) {
                if (e instanceof IOException) {
                    throw (IOException)e;
                }
                throw new IOException("Failed to read data at " + znodePath, e);
            }
        }

        @Override
        public InputStream openInputStream(String storedResourceId) throws IOException {
            String znodePath = this.getZnodeForResource(storedResourceId);
            byte[] znodeData = null;
            try {
                if (this.zkClient.exists(znodePath, this.retryOnConnLoss).booleanValue()) {
                    znodeData = this.zkClient.getData(znodePath, null, null, this.retryOnConnLoss);
                }
            }
            catch (Exception e) {
                if (e instanceof IOException) {
                    throw (IOException)e;
                }
                throw new IOException("Failed to read data at " + znodePath, e);
            }
            if (znodeData != null) {
                log.info("Read {} bytes from znode {}", (Object)znodeData.length, (Object)znodePath);
            } else {
                znodeData = new byte[]{};
                log.info("No data found for znode {}", (Object)znodePath);
            }
            return new ByteArrayInputStream(znodeData);
        }

        @Override
        public OutputStream openOutputStream(String storedResourceId) throws IOException {
            final String znodePath = this.getZnodeForResource(storedResourceId);
            final boolean retryOnConnLoss = this.retryOnConnLoss;
            ByteArrayOutputStream baos = new ByteArrayOutputStream(){

                @Override
                public void close() {
                    byte[] znodeData = this.toByteArray();
                    try {
                        if (ZooKeeperStorageIO.this.zkClient.exists(znodePath, retryOnConnLoss).booleanValue()) {
                            ZooKeeperStorageIO.this.zkClient.setData(znodePath, znodeData, retryOnConnLoss);
                            log.info("Wrote {} bytes to existing znode {}", (Object)znodeData.length, (Object)znodePath);
                        } else {
                            ZooKeeperStorageIO.this.zkClient.makePath(znodePath, znodeData, retryOnConnLoss);
                            log.info("Wrote {} bytes to new znode {}", (Object)znodeData.length, (Object)znodePath);
                        }
                    }
                    catch (Exception e) {
                        if (e instanceof RuntimeException) {
                            throw (RuntimeException)e;
                        }
                        throw new ResourceException(Status.SERVER_ERROR_INTERNAL, "Failed to save data to ZooKeeper znode: " + znodePath + " due to: " + e, (Throwable)e);
                    }
                }
            };
            return baos;
        }

        protected String getZnodeForResource(String storedResourceId) {
            return String.format(Locale.ROOT, "%s/%s", this.znodeBase, storedResourceId);
        }

        @Override
        public boolean delete(String storedResourceId) throws IOException {
            boolean wasDeleted = false;
            String znodePath = this.getZnodeForResource(storedResourceId);
            try {
                if (this.zkClient.exists(znodePath, this.retryOnConnLoss).booleanValue()) {
                    log.info("Attempting to delete znode {}", (Object)znodePath);
                    this.zkClient.delete(znodePath, -1, this.retryOnConnLoss);
                    wasDeleted = this.zkClient.exists(znodePath, this.retryOnConnLoss);
                    if (wasDeleted) {
                        log.info("Deleted znode {}", (Object)znodePath);
                    } else {
                        log.warn("Failed to delete znode {}", (Object)znodePath);
                    }
                } else {
                    log.warn("Znode {} does not exist; delete operation ignored.", (Object)znodePath);
                }
            }
            catch (Exception e) {
                if (e instanceof IOException) {
                    throw (IOException)e;
                }
                throw new IOException("Failed to read data at " + znodePath, e);
            }
            return wasDeleted;
        }

        @Override
        public String getInfo() {
            return "ZooKeeperStorageIO:path=" + this.znodeBase;
        }
    }

    public static class FileStorageIO
    implements StorageIO {
        private String storageDir;

        @Override
        public void configure(SolrResourceLoader loader, NamedList<String> initArgs) throws SolrException {
            String storageDirArg = (String)initArgs.get(ManagedResourceStorage.STORAGE_DIR_INIT_ARG);
            if (storageDirArg == null || storageDirArg.trim().length() == 0) {
                throw new IllegalArgumentException("Required configuration parameter 'storageDir' not provided!");
            }
            File dir = new File(storageDirArg);
            if (!dir.isDirectory()) {
                dir.mkdirs();
            }
            this.storageDir = dir.getAbsolutePath();
            log.info("File-based storage initialized to use dir: " + this.storageDir);
        }

        @Override
        public boolean exists(String storedResourceId) throws IOException {
            return new File(this.storageDir, storedResourceId).exists();
        }

        @Override
        public InputStream openInputStream(String storedResourceId) throws IOException {
            return new FileInputStream(this.storageDir + "/" + storedResourceId);
        }

        @Override
        public OutputStream openOutputStream(String storedResourceId) throws IOException {
            return new FileOutputStream(this.storageDir + "/" + storedResourceId);
        }

        @Override
        public boolean delete(String storedResourceId) throws IOException {
            File storedFile = new File(this.storageDir, storedResourceId);
            return storedFile.isFile() ? storedFile.delete() : false;
        }

        @Override
        public String getInfo() {
            return "file:dir=" + this.storageDir;
        }
    }

    public static interface StorageIO {
        public String getInfo();

        public void configure(SolrResourceLoader var1, NamedList<String> var2) throws SolrException;

        public boolean exists(String var1) throws IOException;

        public InputStream openInputStream(String var1) throws IOException;

        public OutputStream openOutputStream(String var1) throws IOException;

        public boolean delete(String var1) throws IOException;
    }
}

