/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.network.client;

import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.util.concurrent.GenericFutureListener;
import java.io.Closeable;
import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import org.apache.spark.network.buffer.NioManagedBuffer;
import org.apache.spark.network.client.ChunkReceivedCallback;
import org.apache.spark.network.client.RpcResponseCallback;
import org.apache.spark.network.client.StreamCallback;
import org.apache.spark.network.client.TransportResponseHandler;
import org.apache.spark.network.protocol.ChunkFetchRequest;
import org.apache.spark.network.protocol.OneWayMessage;
import org.apache.spark.network.protocol.RpcRequest;
import org.apache.spark.network.protocol.StreamChunkId;
import org.apache.spark.network.protocol.StreamRequest;
import org.apache.spark.network.util.NettyUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spark-project.guava.annotations.VisibleForTesting;
import org.spark-project.guava.base.Objects;
import org.spark-project.guava.base.Preconditions;
import org.spark-project.guava.base.Throwables;
import org.spark-project.guava.util.concurrent.SettableFuture;

public class TransportClient
implements Closeable {
    private final Logger logger = LoggerFactory.getLogger(TransportClient.class);
    private final Channel channel;
    private final TransportResponseHandler handler;
    @Nullable
    private String clientId;
    private volatile boolean timedOut;

    public TransportClient(Channel channel, TransportResponseHandler handler) {
        this.channel = Preconditions.checkNotNull(channel);
        this.handler = Preconditions.checkNotNull(handler);
        this.timedOut = false;
    }

    public Channel getChannel() {
        return this.channel;
    }

    public boolean isActive() {
        return !this.timedOut && (this.channel.isOpen() || this.channel.isActive());
    }

    public SocketAddress getSocketAddress() {
        return this.channel.remoteAddress();
    }

    public String getClientId() {
        return this.clientId;
    }

    public void setClientId(String id) {
        Preconditions.checkState(this.clientId == null, "Client ID has already been set.");
        this.clientId = id;
    }

    public void fetchChunk(long streamId, final int chunkIndex, final ChunkReceivedCallback callback) {
        final String serverAddr = NettyUtils.getRemoteAddress(this.channel);
        final long startTime = System.currentTimeMillis();
        this.logger.debug("Sending fetch chunk request {} to {}", (Object)chunkIndex, (Object)serverAddr);
        final StreamChunkId streamChunkId = new StreamChunkId(streamId, chunkIndex);
        this.handler.addFetchRequest(streamChunkId, callback);
        this.channel.writeAndFlush((Object)new ChunkFetchRequest(streamChunkId)).addListener((GenericFutureListener)new ChannelFutureListener(){

            public void operationComplete(ChannelFuture future) throws Exception {
                if (future.isSuccess()) {
                    long timeTaken = System.currentTimeMillis() - startTime;
                    TransportClient.this.logger.trace("Sending request {} to {} took {} ms", new Object[]{streamChunkId, serverAddr, timeTaken});
                } else {
                    String errorMsg = String.format("Failed to send request %s to %s: %s", streamChunkId, serverAddr, future.cause());
                    TransportClient.this.logger.error(errorMsg, future.cause());
                    TransportClient.this.handler.removeFetchRequest(streamChunkId);
                    TransportClient.this.channel.close();
                    try {
                        callback.onFailure(chunkIndex, new IOException(errorMsg, future.cause()));
                    }
                    catch (Exception e) {
                        TransportClient.this.logger.error("Uncaught exception in RPC response callback handler!", (Throwable)e);
                    }
                }
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stream(final String streamId, final StreamCallback callback) {
        final String serverAddr = NettyUtils.getRemoteAddress(this.channel);
        final long startTime = System.currentTimeMillis();
        this.logger.debug("Sending stream request for {} to {}", (Object)streamId, (Object)serverAddr);
        TransportClient transportClient = this;
        synchronized (transportClient) {
            this.handler.addStreamCallback(callback);
            this.channel.writeAndFlush((Object)new StreamRequest(streamId)).addListener((GenericFutureListener)new ChannelFutureListener(){

                public void operationComplete(ChannelFuture future) throws Exception {
                    if (future.isSuccess()) {
                        long timeTaken = System.currentTimeMillis() - startTime;
                        TransportClient.this.logger.trace("Sending request for {} to {} took {} ms", new Object[]{streamId, serverAddr, timeTaken});
                    } else {
                        String errorMsg = String.format("Failed to send request for %s to %s: %s", streamId, serverAddr, future.cause());
                        TransportClient.this.logger.error(errorMsg, future.cause());
                        TransportClient.this.channel.close();
                        try {
                            callback.onFailure(streamId, new IOException(errorMsg, future.cause()));
                        }
                        catch (Exception e) {
                            TransportClient.this.logger.error("Uncaught exception in RPC response callback handler!", (Throwable)e);
                        }
                    }
                }
            });
        }
    }

    public long sendRpc(ByteBuffer message, final RpcResponseCallback callback) {
        final String serverAddr = NettyUtils.getRemoteAddress(this.channel);
        final long startTime = System.currentTimeMillis();
        this.logger.trace("Sending RPC to {}", (Object)serverAddr);
        final long requestId = Math.abs(UUID.randomUUID().getLeastSignificantBits());
        this.handler.addRpcRequest(requestId, callback);
        this.channel.writeAndFlush((Object)new RpcRequest(requestId, new NioManagedBuffer(message))).addListener((GenericFutureListener)new ChannelFutureListener(){

            public void operationComplete(ChannelFuture future) throws Exception {
                if (future.isSuccess()) {
                    long timeTaken = System.currentTimeMillis() - startTime;
                    TransportClient.this.logger.trace("Sending request {} to {} took {} ms", new Object[]{requestId, serverAddr, timeTaken});
                } else {
                    String errorMsg = String.format("Failed to send RPC %s to %s: %s", requestId, serverAddr, future.cause());
                    TransportClient.this.logger.error(errorMsg, future.cause());
                    TransportClient.this.handler.removeRpcRequest(requestId);
                    TransportClient.this.channel.close();
                    try {
                        callback.onFailure(new IOException(errorMsg, future.cause()));
                    }
                    catch (Exception e) {
                        TransportClient.this.logger.error("Uncaught exception in RPC response callback handler!", (Throwable)e);
                    }
                }
            }
        });
        return requestId;
    }

    public ByteBuffer sendRpcSync(ByteBuffer message, long timeoutMs) {
        final SettableFuture result = SettableFuture.create();
        this.sendRpc(message, new RpcResponseCallback(){

            @Override
            public void onSuccess(ByteBuffer response) {
                result.set(response);
            }

            @Override
            public void onFailure(Throwable e) {
                result.setException(e);
            }
        });
        try {
            return (ByteBuffer)result.get(timeoutMs, TimeUnit.MILLISECONDS);
        }
        catch (ExecutionException e) {
            throw Throwables.propagate(e.getCause());
        }
        catch (Exception e) {
            throw Throwables.propagate(e);
        }
    }

    public void send(ByteBuffer message) {
        this.channel.writeAndFlush((Object)new OneWayMessage(new NioManagedBuffer(message)));
    }

    public void removeRpcRequest(long requestId) {
        this.handler.removeRpcRequest(requestId);
    }

    public void timeOut() {
        this.timedOut = true;
    }

    @VisibleForTesting
    public TransportResponseHandler getHandler() {
        return this.handler;
    }

    @Override
    public void close() {
        this.channel.close().awaitUninterruptibly(10L, TimeUnit.SECONDS);
    }

    public String toString() {
        return Objects.toStringHelper(this).add("remoteAdress", this.channel.remoteAddress()).add("clientId", this.clientId).add("isActive", this.isActive()).toString();
    }
}

