/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.retry;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import org.neo4j.driver.internal.retry.RetryLogic;
import org.neo4j.driver.internal.retry.RetrySettings;
import org.neo4j.driver.internal.util.Clock;
import org.neo4j.driver.internal.util.Supplier;
import org.neo4j.driver.v1.Logger;
import org.neo4j.driver.v1.Logging;
import org.neo4j.driver.v1.exceptions.ServiceUnavailableException;
import org.neo4j.driver.v1.exceptions.SessionExpiredException;
import org.neo4j.driver.v1.exceptions.TransientException;

public class ExponentialBackoffRetryLogic
implements RetryLogic {
    private static final String RETRY_LOGIC_LOG_NAME = "RetryLogic";
    static final long DEFAULT_MAX_RETRY_TIME_MS = TimeUnit.SECONDS.toMillis(30L);
    private static final long INITIAL_RETRY_DELAY_MS = TimeUnit.SECONDS.toMillis(1L);
    private static final double RETRY_DELAY_MULTIPLIER = 2.0;
    private static final double RETRY_DELAY_JITTER_FACTOR = 0.2;
    private final long maxRetryTimeMs;
    private final long initialRetryDelayMs;
    private final double multiplier;
    private final double jitterFactor;
    private final Clock clock;
    private final Logger log;

    public ExponentialBackoffRetryLogic(RetrySettings settings, Clock clock, Logging logging) {
        this(settings.maxRetryTimeMs(), INITIAL_RETRY_DELAY_MS, 2.0, 0.2, clock, logging);
    }

    ExponentialBackoffRetryLogic(long maxRetryTimeMs, long initialRetryDelayMs, double multiplier, double jitterFactor, Clock clock, Logging logging) {
        this.maxRetryTimeMs = maxRetryTimeMs;
        this.initialRetryDelayMs = initialRetryDelayMs;
        this.multiplier = multiplier;
        this.jitterFactor = jitterFactor;
        this.clock = clock;
        this.log = logging.getLog(RETRY_LOGIC_LOG_NAME);
        this.verifyAfterConstruction();
    }

    @Override
    public <T> T retry(Supplier<T> work) {
        List<Throwable> errors = null;
        long startTime = -1L;
        long nextDelayMs = this.initialRetryDelayMs;
        while (true) {
            try {
                return work.get();
            }
            catch (Throwable error) {
                if (ExponentialBackoffRetryLogic.canRetryOn(error)) {
                    long elapsedTime;
                    long currentTime = this.clock.millis();
                    if (startTime == -1L) {
                        startTime = currentTime;
                    }
                    if ((elapsedTime = currentTime - startTime) < this.maxRetryTimeMs) {
                        long delayWithJitterMs = this.computeDelayWithJitter(nextDelayMs);
                        this.log.error("Transaction failed and will be retried in " + delayWithJitterMs + "ms", error);
                        this.sleep(delayWithJitterMs);
                        nextDelayMs = (long)((double)nextDelayMs * this.multiplier);
                        errors = ExponentialBackoffRetryLogic.recordError(error, errors);
                        continue;
                    }
                }
                ExponentialBackoffRetryLogic.addSuppressed(error, errors);
                throw error;
            }
            break;
        }
    }

    private long computeDelayWithJitter(long delayMs) {
        long jitter = (long)((double)delayMs * this.jitterFactor);
        long min = delayMs - jitter;
        long max = delayMs + jitter;
        return ThreadLocalRandom.current().nextLong(min, max + 1L);
    }

    private void sleep(long delayMs) {
        try {
            this.clock.sleep(delayMs);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("Retries interrupted", e);
        }
    }

    private void verifyAfterConstruction() {
        if (this.maxRetryTimeMs < 0L) {
            throw new IllegalArgumentException("Max retry time should be >= 0: " + this.maxRetryTimeMs);
        }
        if (this.initialRetryDelayMs < 0L) {
            throw new IllegalArgumentException("Initial retry delay should >= 0: " + this.initialRetryDelayMs);
        }
        if (this.multiplier < 1.0) {
            throw new IllegalArgumentException("Multiplier should be >= 1.0: " + this.multiplier);
        }
        if (this.jitterFactor < 0.0 || this.jitterFactor > 1.0) {
            throw new IllegalArgumentException("Jitter factor should be in [0.0, 1.0]: " + this.jitterFactor);
        }
        if (this.clock == null) {
            throw new IllegalArgumentException("Clock should not be null");
        }
    }

    private static boolean canRetryOn(Throwable error) {
        return error instanceof SessionExpiredException || error instanceof ServiceUnavailableException || ExponentialBackoffRetryLogic.isTransientError(error);
    }

    private static boolean isTransientError(Throwable error) {
        if (error instanceof TransientException) {
            String code = ((TransientException)error).code();
            return !"Neo.TransientError.Transaction.Terminated".equals(code) && !"Neo.TransientError.Transaction.LockClientStopped".equals(code);
        }
        return false;
    }

    private static List<Throwable> recordError(Throwable error, List<Throwable> errors) {
        if (errors == null) {
            errors = new ArrayList<Throwable>();
        }
        errors.add(error);
        return errors;
    }

    private static void addSuppressed(Throwable error, List<Throwable> suppressedErrors) {
        if (suppressedErrors != null) {
            for (Throwable suppressedError : suppressedErrors) {
                if (error == suppressedError) continue;
                error.addSuppressed(suppressedError);
            }
        }
    }
}

