/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ExampleJMS.java 18287 2009-08-17 19:35:02Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.example.jms;

import java.io.InputStream;

import javax.jms.ConnectionFactory;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.jms.TextMessage;
import javax.naming.Context;
import javax.naming.InitialContext;

import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.jms.JmsComponent;
import org.ow2.jonas.camel.service.api.ICamelService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.springframework.jms.support.destination.JndiDestinationResolver;

/**
 * A simple example on how to use the camel service on JOnAS 5.
 * 
 * @author Guillaume Renault
 */
public class ExampleJMS {

    /**
     * The logger.
     */
    private Log logger = LogFactory.getLog(this.getClass());

    /**
     * The camel service. Injected by the container.
     */
    private ICamelService camelService = null;

    /**
     * The camel context name of the example.
     */
    private String camelContextName = null;

    /**
     * Start the route.
     */
    public void start() throws Throwable {
        final ClassLoader bundleCL = this.getClass().getClassLoader();
        final ClassLoader oldCL = Thread.currentThread().getContextClassLoader();

        try {
            // Route definition
            RouteBuilder builder = new RouteBuilder() {
                @Override
                public void configure() throws Exception {
                    this.from("registry:queue").process(new Processor() {
                        public void process(final Exchange e) {
                            ExampleJMS.this.logger.info("Received JMS message {0}", e.getIn().getBody());
                        }
                    });
                }
            };

            // Start a new Camel context for the route
            this.camelContextName = this.camelService.startNewContext();
            this.logger.info("Camel context {0} started", camelContextName);

            // Add the registry entries
            InputStream input = bundleCL.getResourceAsStream("jms-registry.xml");
            this.camelService.addRegistry(input, this.camelContextName);

            try {
                // Change the TCCL since JNDI and JMS lookups use the TCCL and
                // not the bundle's ClassLoader.
                //
                // This is needed when looking up for the ConnectionFactory,
                // creating the Camel-JMS route and executing embedded tests.
                Thread.currentThread().setContextClassLoader(bundleCL);

                // Add the JORAM component
                JmsComponent joram = new JmsComponent();
                Context context = new InitialContext();
                ConnectionFactory connectionFactory = (ConnectionFactory) context.lookup("CF");

                joram.setConnectionFactory(connectionFactory);
                JndiDestinationResolver jndiDestinationResolver = new JndiDestinationResolver();
                jndiDestinationResolver.setCache(true);

                joram.setDestinationResolver(jndiDestinationResolver);

                this.camelService.addComponent("joram", joram, this.camelContextName);
                this.logger.debug("JORAM component added");

                // Add the route in the camel context.
                this.camelService.addRoutes(builder, this.camelContextName);

                // Launch kind of a test.
                this.test();
            } finally {
                Thread.currentThread().setContextClassLoader(oldCL);
            }
        } catch (Throwable t) {
            this.logger.error("Cannot start ExampleJMS", t);
            throw t;
        }
    }

    /**
     * Test the route.
     */
    public void test() throws Exception {
        Context context = new InitialContext();
        QueueConnectionFactory connectionFactory = (QueueConnectionFactory) context.lookup("JQCF");
        QueueConnection queueConnection = connectionFactory.createQueueConnection();
        QueueSession queueSession = queueConnection.createQueueSession(false, Session.AUTO_ACKNOWLEDGE);
        Queue queue = (Queue) context.lookup("sampleQueue");
        QueueSender jmsSender = queueSession.createSender(queue);

        for (int i = 0; i < 5; i++) {
            TextMessage message = queueSession.createTextMessage("Test Message: " + i);
            jmsSender.send(message);
        }
    }

    /**
     * Stop the route.
     */
    public void stop() throws Exception {
        try {
            this.camelService.stop(this.camelContextName);
        } finally {
            this.camelContextName = null;
            this.camelService = null;
        }
    }

}
