/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.crt;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.AccessDeniedException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.NotDirectoryException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Date;
import java.util.Locale;
import software.amazon.awssdk.crt.CrtPlatform;
import software.amazon.awssdk.crt.CrtRuntimeException;
import software.amazon.awssdk.crt.Log;

public final class CRT {
    private static final String CRT_LIB_NAME = "aws-crt-jni";
    public static final int AWS_CRT_SUCCESS = 0;
    private static final CrtPlatform s_platform = CRT.findPlatformImpl();

    private static String normalize(String value) {
        if (value == null) {
            return "";
        }
        return value.toLowerCase(Locale.US).replaceAll("[^a-z0-9]+", "");
    }

    private static boolean isAndroid() {
        try {
            Class.forName("android.os.Build");
            return true;
        }
        catch (ClassNotFoundException ex) {
            return false;
        }
    }

    public static String getOSIdentifier() throws UnknownPlatformException {
        if (CRT.isAndroid()) {
            return "android";
        }
        CrtPlatform platform = CRT.getPlatformImpl();
        String name = CRT.normalize(platform != null ? platform.getOSIdentifier() : System.getProperty("os.name"));
        if (name.contains("windows")) {
            return "windows";
        }
        if (name.contains("linux")) {
            return "linux";
        }
        if (name.contains("freebsd")) {
            return "freebsd";
        }
        if (name.contains("macosx")) {
            return "osx";
        }
        if (name.contains("sun os") || name.contains("sunos") || name.contains("solaris")) {
            return "solaris";
        }
        throw new UnknownPlatformException("AWS CRT: OS not supported: " + name);
    }

    public static String getArchIdentifier() throws UnknownPlatformException {
        CrtPlatform platform = CRT.getPlatformImpl();
        String arch = CRT.normalize(platform != null ? platform.getArchIdentifier() : System.getProperty("os.arch"));
        if (arch.matches("^(x8664|amd64|ia32e|em64t|x64|x86_64)$")) {
            return "x86_64";
        }
        if (arch.matches("^(x8632|x86|i[3-6]86|ia32|x32)$")) {
            return CRT.getOSIdentifier() == "android" ? "x86" : "x86_32";
        }
        if (arch.startsWith("armeabi")) {
            if (CRT.getOSIdentifier() == "android") {
                return "armeabi-v7a";
            }
            if (arch.contains("v7")) {
                return "armv7";
            }
            return "armv6";
        }
        if (arch.startsWith("arm64") || arch.startsWith("aarch64")) {
            return CRT.getOSIdentifier() == "android" ? "arm64-v8a" : "armv8";
        }
        if (arch.equals("arm")) {
            return "armv6";
        }
        throw new UnknownPlatformException("AWS CRT: architecture not supported: " + arch);
    }

    private static void loadLibraryFromJar() {
        try {
            Path tmpdirPath;
            String tmpdir = System.getProperty("java.io.tmpdir");
            try {
                tmpdirPath = Paths.get(tmpdir, new String[0]).toAbsolutePath().normalize();
                File tmpdirFile = tmpdirPath.toFile();
                if (tmpdirFile.exists()) {
                    if (!tmpdirFile.isDirectory()) {
                        throw new NotDirectoryException(tmpdirPath.toString());
                    }
                } else {
                    Files.createDirectories(tmpdirPath, new FileAttribute[0]);
                }
                if (!tmpdirFile.canRead() || !tmpdirFile.canWrite()) {
                    throw new AccessDeniedException(tmpdirPath.toString());
                }
            }
            catch (Exception ex) {
                String msg = "java.io.tmpdir=\"" + tmpdir + "\": Invalid directory";
                throw new IOException(msg, ex);
            }
            String prefix = "AWSCRT_" + new Date().getTime();
            String libraryName = System.mapLibraryName(CRT_LIB_NAME);
            String libraryPath = "/" + CRT.getOSIdentifier() + "/" + CRT.getArchIdentifier() + "/" + libraryName;
            Path libTempPath = Files.createTempFile(tmpdirPath, prefix, libraryName, new FileAttribute[0]);
            try (InputStream in = CRT.class.getResourceAsStream(libraryPath);){
                if (in == null) {
                    throw new IOException("Unable to open library in jar for AWS CRT: " + libraryPath);
                }
                Files.deleteIfExists(libTempPath);
                Files.copy(in, libTempPath, new CopyOption[0]);
            }
            File tempSharedLib = libTempPath.toFile();
            if (!tempSharedLib.setExecutable(true)) {
                throw new CrtRuntimeException("Unable to make shared library executable");
            }
            if (!tempSharedLib.setWritable(false)) {
                throw new CrtRuntimeException("Unable to make shared library read-only");
            }
            if (!tempSharedLib.setReadable(true)) {
                throw new CrtRuntimeException("Unable to make shared library readable");
            }
            tempSharedLib.deleteOnExit();
            System.load(libTempPath.toString());
        }
        catch (CrtRuntimeException crtex) {
            System.err.println("Unable to initialize AWS CRT: " + crtex.toString());
            crtex.printStackTrace();
        }
        catch (UnknownPlatformException upe) {
            System.err.println("Unable to determine platform for AWS CRT: " + upe.toString());
            upe.printStackTrace();
        }
        catch (Exception ex) {
            System.err.println("Unable to unpack AWS CRT lib: " + ex.toString());
            ex.printStackTrace();
        }
    }

    private static CrtPlatform findPlatformImpl() {
        String[] platforms;
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        for (String platformImpl : platforms = new String[]{String.format("software.amazon.awssdk.crt.test.%s.CrtPlatformImpl", CRT.getOSIdentifier()), String.format("software.amazon.awssdk.crt.%s.CrtPlatformImpl", CRT.getOSIdentifier())}) {
            try {
                Class<?> platformClass = classLoader.loadClass(platformImpl);
                CrtPlatform instance = (CrtPlatform)platformClass.newInstance();
                return instance;
            }
            catch (ClassNotFoundException platformClass) {
            }
            catch (IllegalAccessException | InstantiationException ex) {
                throw new CrtRuntimeException(ex.toString());
            }
        }
        return null;
    }

    public static CrtPlatform getPlatformImpl() {
        return s_platform;
    }

    private static void jvmInit() {
        CrtPlatform platform = CRT.getPlatformImpl();
        if (platform != null) {
            platform.jvmInit();
        }
    }

    private static native void awsCrtInit(int var0, boolean var1, boolean var2) throws CrtRuntimeException;

    public static native int awsLastError();

    public static native String awsErrorString(int var0);

    public static native String awsErrorName(int var0);

    public static long nativeMemory() {
        return CRT.awsNativeMemory();
    }

    public static native void dumpNativeMemory();

    private static native long awsNativeMemory();

    static void testJniException(boolean throwException) {
        if (throwException) {
            throw new RuntimeException("Testing");
        }
    }

    public static void checkJniExceptionContract(boolean clearException) {
        CRT.nativeCheckJniExceptionContract(clearException);
    }

    private static native void nativeCheckJniExceptionContract(boolean var0);

    static {
        CRT.jvmInit();
        try {
            System.loadLibrary(CRT_LIB_NAME);
        }
        catch (UnsatisfiedLinkError e) {
            CRT.loadLibraryFromJar();
        }
        int memoryTracingLevel = 0;
        try {
            memoryTracingLevel = Integer.parseInt(System.getProperty("aws.crt.memory.tracing"));
        }
        catch (Exception exception) {
            // empty catch block
        }
        boolean debugWait = System.getProperty("aws.crt.debugwait") != null;
        boolean strictShutdown = System.getProperty("aws.crt.strictshutdown") != null;
        CRT.awsCrtInit(memoryTracingLevel, debugWait, strictShutdown);
        try {
            Log.initLoggingFromSystemProperties();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    public static class UnknownPlatformException
    extends RuntimeException {
        UnknownPlatformException(String message) {
            super(message);
        }
    }
}

