package com.aliyun.core.utils;

import com.aliyun.core.logging.ClientLogger;

import java.nio.ByteBuffer;
import java.util.Arrays;

public final class ByteBufferCollector {
    private static final int DEFAULT_INITIAL_SIZE = 1024;
    private static final String INVALID_INITIAL_SIZE = "'initialSize' cannot be equal to or less than 0.";
    private static final String REQUESTED_BUFFER_INVALID = "Required capacity is greater than Integer.MAX_VALUE.";
    private final ClientLogger logger = new ClientLogger(ByteBufferCollector.class);
    private byte[] buffer;
    private int position;

    public ByteBufferCollector() {
        this(DEFAULT_INITIAL_SIZE);
    }

    public ByteBufferCollector(int initialSize) {
        if (initialSize <= 0) {
            throw logger.logExceptionAsError(new IllegalArgumentException(INVALID_INITIAL_SIZE));
        }

        this.buffer = new byte[initialSize];
        this.position = 0;
    }

    public synchronized void write(ByteBuffer byteBuffer) {
        // Null buffer.
        if (byteBuffer == null) {
            return;
        }

        int remaining = byteBuffer.remaining();

        // Nothing to write.
        if (remaining == 0) {
            return;
        }

        ensureCapacity(remaining);
        byteBuffer.get(buffer, position, remaining);
        position += remaining;
    }

    public synchronized byte[] toByteArray() {
        return Arrays.copyOf(buffer, position);
    }

    private void ensureCapacity(int byteBufferRemaining) throws OutOfMemoryError {
        int currentCapacity = buffer.length;
        int requiredCapacity = position + byteBufferRemaining;

        if (((position ^ requiredCapacity) & (byteBufferRemaining ^ requiredCapacity)) < 0) {
            throw logger.logExceptionAsError(new IllegalStateException(REQUESTED_BUFFER_INVALID));
        }

        if (currentCapacity >= requiredCapacity) {
            return;
        }

        int proposedNewCapacity = currentCapacity << 1;

        if ((proposedNewCapacity - requiredCapacity) < 0) {
            proposedNewCapacity = requiredCapacity;
        }

        if (proposedNewCapacity < 0) {
            proposedNewCapacity = Integer.MAX_VALUE - 8;
        }

        buffer = Arrays.copyOf(buffer, proposedNewCapacity);
    }
}
