package com.aliyun.core.utils;

import com.aliyun.core.execption.AliyunException;

import java.io.UncheckedIOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;

/**
 * Operations on {@link String} that are {@code null} safe.
 */
public class StringUtils {

    /**
     * The empty String {@code ""}.
     */
    public static final String EMPTY = "";

    /**
     * {@code StringUtils} instances should NOT be constructed in
     * standard programming.
     */
    private StringUtils() {
    }

    /**
     * Checks if a CharSequence is empty ("") or null.
     *
     * @param cs the CharSequence to check, may be null
     * @return {@code true} if the CharSequence is empty or null
     */
    public static boolean isEmpty(final CharSequence cs) {
        return cs == null || cs.length() == 0;
    }

    /**
     * Checks if a object is empty or null.
     *
     * @param object the String.valueOf(object) to check, may be null
     * @return {@code true} if the String.valueOf(object) is empty or null
     */
    public static boolean isEmpty(Object object) {
        if (null != object) {
            return isEmpty(String.valueOf(object));
        }
        return true;
    }

    /**
     * Checks if a CharSequence is empty (""), null or whitespace only.
     * <p>
     * Whitespace is defined by {@link Character#isWhitespace(char)}.
     *
     * @param cs the CharSequence to check, may be null
     * @return {@code true} if the CharSequence is null, empty or whitespace only
     */
    public static boolean isBlank(final CharSequence cs) {
        if (cs == null || cs.length() == 0) {
            return true;
        }
        for (int i = 0; i < cs.length(); i++) {
            if (!Character.isWhitespace(cs.charAt(i))) {
                return false;
            }
        }
        return true;
    }

    /**
     * The String is trimmed using {@link String#trim()}.
     * Trim removes start and end characters &lt;= 32.
     *
     * @param str the String to be trimmed, may be null
     * @return the trimmed string, {@code null} if null String input
     */
    public static String trim(final String str) {
        return str == null ? null : str.trim();
    }

    /**
     * Compares two Strings, returning {@code true} if they represent
     * equal sequences of characters.
     *
     * @param cs1 the first String, may be {@code null}
     * @param cs2 the second String, may be {@code null}
     * @return {@code true} if the Strings are equal (case-sensitive), or both {@code null}
     * @see Object#equals(Object)
     */
    public static boolean equals(final String cs1, final String cs2) {
        if (cs1 == null || cs2 == null) {
            return false;
        }
        if (cs1.length() != cs2.length()) {
            return false;
        }
        return cs1.equals(cs2);
    }

    /**
     * Join any Strings.
     *
     * @param delimiter the delimiter which split two Strings, should not be {@code null}
     * @param elements  the Strings Iterable, should not be {@code null}
     * @return {@code String}
     */
    public static String join(String delimiter, Iterable<? extends String> elements) {
        Objects.requireNonNull(delimiter);
        Objects.requireNonNull(elements);
        StringBuilder stringBuilder = new StringBuilder();
        for (String value : elements) {
            stringBuilder.append(value);
            stringBuilder.append(delimiter);
        }
        if (stringBuilder.length() > 0) {
            stringBuilder.deleteCharAt(stringBuilder.length() - 1);
        }
        return stringBuilder.toString();
    }

    /**
     * Gets a substring from the specified String avoiding exceptions.
     *
     * @param str   the String to get the substring from, may be null
     * @param start the position to start from, negative means count back from the end of the String by this many characters
     * @param end   the position to end at (exclusive), negative means count back from the end of the String by this many
     *              characters
     * @return substring from start position to end position,
     * {@code null} if null String input
     */
    public static String substring(final String str, int start, int end) {
        if (str == null) {
            return null;
        }

        // handle negatives
        if (end < 0) {
            end = str.length() + end; // remember end is negative
        }
        if (start < 0) {
            start = str.length() + start; // remember start is negative
        }

        // check length next
        if (end > str.length()) {
            end = str.length();
        }

        // if start is greater than end, return ""
        if (start > end) {
            return EMPTY;
        }

        if (start < 0) {
            start = 0;
        }
        if (end < 0) {
            end = 0;
        }

        return str.substring(start, end);
    }

    /**
     * Converts a String to upper case as per {@link String#toUpperCase()}.
     * <p>
     * This uses "ENGLISH" as the locale.
     *
     * @param str the String to upper case, may be null
     * @return the upper cased String, {@code null} if null String input
     */
    public static String upperCase(final String str) {
        if (str == null) {
            return null;
        }
        return str.toUpperCase(Locale.ENGLISH);
    }

    /**
     * Converts a String to lower case as per {@link String#toLowerCase()}.
     * <p>
     * This uses "ENGLISH" as the locale.
     *
     * @param str the String to lower case, may be null
     * @return the lower cased String, {@code null} if null String input
     */
    public static String lowerCase(final String str) {
        if (str == null) {
            return null;
        }
        return str.toLowerCase(Locale.ENGLISH);
    }

    /**
     * Encode the given bytes as a string using the given charset
     *
     * @throws UncheckedIOException with a {@link CharacterCodingException} as the cause if the bytes cannot be encoded using the
     *                              provided charset.
     */
    public static String fromBytes(byte[] bytes, Charset charset) throws UncheckedIOException {
        try {
            return charset.newDecoder().decode(ByteBuffer.wrap(bytes)).toString();
        } catch (CharacterCodingException e) {
            throw new UncheckedIOException("Cannot encode string.", e);
        }
    }

    /**
     * Tests if this string starts with the specified prefix ignoring case considerations.
     *
     * @param str    the string to be tested
     * @param prefix the prefix
     * @return true if the string starts with the prefix ignoring case
     */
    public static boolean startsWithIgnoreCase(String str, String prefix) {
        return str.regionMatches(true, 0, prefix, 0, prefix.length());
    }

    /**
     * Convert a string to boolean safely (as opposed to the less strict {@link Boolean#parseBoolean(String)}). If a customer
     * specifies a boolean value it should be "true" or "false" (case insensitive) or an exception will be thrown.
     */
    public static boolean safeStringToBoolean(String value) {
        if (value.equalsIgnoreCase("true")) {
            return true;
        } else if (value.equalsIgnoreCase("false")) {
            return false;
        }
        throw new IllegalArgumentException("Value was defined as '" + value + "', but should be 'false' or 'true'");
    }

    /**
     * Create a to-string result for the given class name and field map.
     *
     * @param className The name of the class being toString.
     * @param fieldMap  The key and value of the field. Value is ignored if null.
     */
    public static String toAliString(String className, Map<String, Object> fieldMap) {
        StringBuilder result = new StringBuilder(className).append("(");
        fieldMap.forEach((fieldName, field) -> {
            if (field != null) {
                String value;

                if (field.getClass().isArray()) {
                    if (field instanceof byte[]) {
                        value = ((byte[]) field).length == 0 ? "" : "0x" + new String((byte[]) field);
                    } else {
                        value = Arrays.toString((Object[]) field);
                    }
                } else {
                    value = String.valueOf(field);
                }
                result.append(fieldName).append("=").append(value).append(", ");
            }
        });
        return result.append(")").toString();
    }

    public static byte[] toBytes(String str) {
        try {
            return str.getBytes("UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new AliyunException(e);
        }
    }

    /**
     * Convert a bytes to string(utf8)
     *
     * @return the return string
     */
    public static String toString(byte[] bytes) {
        try {
            return new String(bytes, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new AliyunException(e);
        }
    }
}
