package darabonba.core.internal.async;

import darabonba.core.async.AsyncRequestBody;
import org.reactivestreams.Subscriber;
import org.reactivestreams.Subscription;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.channels.AsynchronousFileChannel;
import java.nio.channels.CompletionHandler;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicLong;

public final class FileAsyncRequestBody implements AsyncRequestBody {
    private static final int DEFAULT_CHUNK_SIZE = 16 * 1024;
    private final Path path;
    private final int chunkSizeInBytes;

    public FileAsyncRequestBody(Path path) {
        this(path, DEFAULT_CHUNK_SIZE);
    }

    public FileAsyncRequestBody(Path path, int chunkSizeInBytes) {
        this.path = path;
        this.chunkSizeInBytes = chunkSizeInBytes;
    }

    @Override
    public Optional<Long> contentLength() {
        try {
            return Optional.of(Files.size(path));
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Override
    public Optional<String> contentType() {
        return Optional.empty();
    }

    @Override
    public void subscribe(Subscriber<? super ByteBuffer> s) {
        try {
            AsynchronousFileChannel channel = openInputChannel(this.path);
            Subscription subscription = new FileSubscription(channel, s, chunkSizeInBytes);
            synchronized (subscription) {
                s.onSubscribe(subscription);
            }
        } catch (IOException e) {
            s.onSubscribe(new NoopSubscription(s));
            s.onError(e);
        }
    }

    private static final class FileSubscription implements Subscription {
        private final AsynchronousFileChannel inputChannel;
        private final Subscriber<? super ByteBuffer> subscriber;
        private final int chunkSize;

        private long position = 0;
        private AtomicLong outstandingDemand = new AtomicLong(0);
        private boolean writeInProgress = false;
        private volatile boolean done = false;

        private FileSubscription(AsynchronousFileChannel inputChannel, Subscriber<? super ByteBuffer> subscriber, int chunkSize) {
            this.inputChannel = inputChannel;
            this.subscriber = subscriber;
            this.chunkSize = chunkSize;
        }

        @Override
        public void request(long n) {
            if (done) {
                return;
            }

            if (n < 1) {
                IllegalArgumentException ex =
                        new IllegalArgumentException(subscriber + " violated the Reactive Streams rule 3.9 by requesting a "
                                + "non-positive number of elements.");
                signalOnError(ex);
            } else {
                try {
                    // As governed by rule 3.17, when demand overflows `Long.MAX_VALUE` we treat the signalled demand as
                    // "effectively unbounded"
                    outstandingDemand.getAndUpdate(initialDemand -> {
                        if (Long.MAX_VALUE - initialDemand < n) {
                            return Long.MAX_VALUE;
                        } else {
                            return initialDemand + n;
                        }
                    });

                    synchronized (this) {
                        if (!writeInProgress) {
                            writeInProgress = true;
                            readData();
                        }
                    }
                } catch (Exception e) {
                    signalOnError(e);
                }
            }
        }

        @Override
        public void cancel() {
            synchronized (this) {
                if (!done) {
                    done = true;
                    closeFile();
                }
            }
        }

        private void readData() {
            // It's possible to have another request for data come in after we've closed the file.
            if (!inputChannel.isOpen()) {
                return;
            }

            ByteBuffer buffer = ByteBuffer.allocate(chunkSize);
            inputChannel.read(buffer, position, buffer, new CompletionHandler<Integer, ByteBuffer>() {
                @Override
                public void completed(Integer result, ByteBuffer attachment) {
                    if (result > 0) {
                        attachment.flip();
                        position += attachment.remaining();
                        signalOnNext(attachment);
                        // If we have more permits, queue up another read.
                        if (outstandingDemand.decrementAndGet() > 0) {
                            readData();
                            return;
                        }
                    } else {
                        // Reached the end of the file, notify the subscriber and cleanup
                        signalOnComplete();
                        closeFile();
                    }

                    synchronized (FileSubscription.this) {
                        writeInProgress = false;
                    }
                }

                @Override
                public void failed(Throwable exc, ByteBuffer attachment) {
                    signalOnError(exc);
                    closeFile();
                }
            });
        }

        private void closeFile() {
            try {
                inputChannel.close();
            } catch (IOException e) {
                signalOnError(e);
            }
        }

        private void signalOnNext(ByteBuffer bb) {
            synchronized (this) {
                if (!done) {
                    subscriber.onNext(bb);
                }
            }
        }

        private void signalOnComplete() {
            synchronized (this) {
                if (!done) {
                    subscriber.onComplete();
                    done = true;
                }
            }
        }

        private void signalOnError(Throwable t) {
            synchronized (this) {
                if (!done) {
                    subscriber.onError(t);
                    done = true;
                }
            }
        }
    }

    private static AsynchronousFileChannel openInputChannel(Path path) throws IOException {
        return AsynchronousFileChannel.open(path, StandardOpenOption.READ);
    }
}
