package darabonba.core.sync;

import com.aliyun.core.io.NonCloseableInputStream;
import com.aliyun.core.utils.BinaryUtils;
import com.aliyun.core.utils.FunctionalUtils;
import com.aliyun.core.utils.IOUtils;
import darabonba.core.TeaRequestBody;
import darabonba.core.internal.sync.FileContentStreamProvider;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Optional;

public interface RequestBody extends TeaRequestBody {

    InputStream newStream();

    static RequestBody fromFile(File file) {
        return fromFile(file.toPath(), null);
    }

    static RequestBody fromFile(Path path) {
        return fromFile(path, null);
    }

    static RequestBody fromFile(Path path, String mimetype) {
        return fromContentProvider(new FileContentStreamProvider(path),
                FunctionalUtils.invokeSafely(() -> Files.size(path)),
                mimetype);
    }

    static RequestBody fromInputStream(InputStream inputStream) {
        return fromInputStream(inputStream, null, null);
    }

    static RequestBody fromInputStream(InputStream inputStream, Long contentLength) {
        return fromInputStream(inputStream, contentLength, null);
    }

    static RequestBody fromInputStream(InputStream inputStream, Long contentLength, String mimetype) {
        IOUtils.markStreamWithMaxReadLimit(inputStream);
        InputStream nonCloseable = new NonCloseableInputStream(inputStream);
        return fromContentProvider(() -> {
            if (nonCloseable.markSupported()) {
                FunctionalUtils.invokeSafely(nonCloseable::reset);
            }
            return nonCloseable;
        }, contentLength, mimetype);
    }

    public static RequestBody fromString(String contents) {
        return fromString(contents, StandardCharsets.UTF_8, null);
    }

    static RequestBody fromString(String contents, Charset cs) {
        return fromString(contents, cs, null);
    }

    static RequestBody fromString(String contents, Charset cs, String mimetype) {
        return fromBytesDirect(contents.getBytes(cs), mimetype);
    }

    public static RequestBody fromBytes(byte[] bytes) {
        return fromBytesDirect(Arrays.copyOf(bytes, bytes.length));
    }

    static RequestBody fromBytes(byte[] bytes, String mimetype) {
        return fromBytesDirect(Arrays.copyOf(bytes, bytes.length), mimetype);
    }

    static RequestBody fromByteBuffer(ByteBuffer byteBuffer) {
        return fromBytesDirect(BinaryUtils.copyAllBytesFrom(byteBuffer));
    }

    static RequestBody fromRemainingByteBuffer(ByteBuffer byteBuffer) {
        return fromBytesDirect(BinaryUtils.copyRemainingBytesFrom(byteBuffer));
    }

    static RequestBody empty() {
        return fromBytesDirect(new byte[0]);
    }

    static RequestBody fromBytesDirect(byte[] bytes) {
        return fromBytesDirect(bytes, null);
    }

    static RequestBody fromBytesDirect(byte[] bytes, String mimetype) {
        return fromContentProvider(() -> new ByteArrayInputStream(bytes), Long.valueOf(bytes.length), mimetype);
    }

    static RequestBody fromContentProvider(ContentStreamProvider provider, Long contentLength, String mimetype) {
        return new RequestBody()  {
            @Override
            public Optional<Long> contentLength() {
                return (contentLength != null)? Optional.of(contentLength) : Optional.empty();
            }

            @Override
            public Optional<String> contentType() {
                return (mimetype != null)? Optional.of(mimetype) : Optional.empty();
            }

            @Override
            public InputStream newStream() {
                return provider.newStream();
            }
        };
    }
}
