package darabonba.core.sse;

import com.aliyun.core.http.HttpHeaders;
import com.aliyun.core.http.HttpResponseHandler;
import darabonba.core.ResponseBytes;
import darabonba.core.async.AsyncResponseHandler;
import darabonba.core.async.ByteArrayAsyncResponseHandler;
import org.reactivestreams.Processor;
import org.reactivestreams.Publisher;
import org.reactivestreams.Subscriber;
import org.reactivestreams.Subscription;

import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class SSEHttpResponseHandler implements HttpResponseHandler {
    private volatile AsyncResponseHandler<?, ?> handler = new ByteArrayAsyncResponseHandler<>();

    private static final String symbol = "\\r\\n\\r\\n|\\r\\r|\\n\\n";
    private static final Pattern pattern = Pattern.compile(symbol);

    private final SSEResponseIterator<?> iterator;
    private volatile AsyncResponseHandler<String, ResponseBytes<String>> errorAsyncResponseHandler;

    public SSEHttpResponseHandler(SSEResponseIterator<?> iterator) {
        this.iterator = iterator;
    }

    @Override
    public void onStream(Publisher<ByteBuffer> publisher, int httpStatusCode, HttpHeaders headers) {
        this.iterator.setStatusCode(httpStatusCode);
        this.iterator.setHeaders(headers.toMap());
        if (httpStatusCode / 100 != 2) {
            this.errorAsyncResponseHandler = AsyncResponseHandler.toBytes();
            this.errorAsyncResponseHandler.onStream(publisher);
        } else {
            SSEProcessor proc = new SSEProcessor(this.iterator);
            this.handler.onStream(proc);
            publisher.subscribe(proc);
        }
    }

    @Override
    public void onError(Throwable throwable) {
        this.iterator.endOfFailure(throwable);
        this.handler.onError(throwable);
    }

    public byte[] getErrorBodyByteArrayUnsafe() {
        if (this.errorAsyncResponseHandler == null) {
            return null;
        }
        ResponseBytes<String> result = this.errorAsyncResponseHandler.transform("");
        return result.asByteArrayUnsafe();
    }

    public byte[] getErrorBodyByteArray() {
        if (this.errorAsyncResponseHandler == null) {
            return null;
        }
        ResponseBytes<String> result = this.errorAsyncResponseHandler.transform("");
        return result.asByteArray();
    }

    static class SSEProcessor implements Processor<ByteBuffer, ByteBuffer> {
        protected volatile Subscriber<? super ByteBuffer> subscriber;
        private final StringBuffer stringBuilder = new StringBuffer();

        private final SSEResponseIterator<?> iterator;

        SSEProcessor(SSEResponseIterator<?> iterator) {
            this.iterator = iterator;
        }

        @Override
        public void subscribe(final Subscriber<? super ByteBuffer> subscriber) {
            this.subscriber = subscriber;
        }

        @Override
        public void onSubscribe(Subscription subscription) {
            this.subscriber.onSubscribe(subscription);
        }

        @Override
        public void onNext(ByteBuffer byteBuffer) {
            try {
                if (byteBuffer.hasRemaining()) {
                    String data = new String(byteBuffer.array(), StandardCharsets.UTF_8);
                    Matcher m = pattern.matcher(data);
                    if (m.find()) {
                        String[] list = data.split(symbol, -1);
                        for (int i = 0; i <= list.length - 1; i++) {
                            String event = list[i];
                            if (i == list.length - 1) {
                                this.stringBuilder.append(event);
                            } else {
                                this.iterator.addEvent(Event.parse(this.stringBuilder.append(event).toString()));
                                this.stringBuilder.setLength(0);
                            }
                        }
                    } else {
                        this.stringBuilder.append(data);
                    }
                }
                this.subscriber.onNext(byteBuffer);
            } catch (Exception e) {
                this.iterator.endOfFailure(e);
                this.subscriber.onError(e);
            }
        }

        @Override
        public void onError(Throwable throwable) {
            this.iterator.endOfFailure(throwable);
            this.subscriber.onError(throwable);
        }

        @Override
        public void onComplete() {
            this.iterator.endOfEvent();
            this.subscriber.onComplete();
        }
    }
}
