package darabonba.core;

import com.aliyun.auth.credentials.provider.ICredentialProvider;
import com.aliyun.core.http.HttpHeaders;
import com.aliyun.core.http.HttpMethod;
import com.aliyun.core.http.ProtocolType;
import com.aliyun.core.utils.Copyable;
import darabonba.core.client.ClientConfiguration;
import darabonba.core.client.ClientOption;
import darabonba.core.utils.CommonUtil;

import java.time.Duration;
import java.util.Optional;

public final class TeaConfiguration implements Copyable {

    private final ClientConfiguration clientConfiguration;
    private final RequestConfiguration requestConfiguration;

    public TeaConfiguration(ClientConfiguration clientConfiguration, RequestConfiguration requestConfiguration) {
        this.clientConfiguration = clientConfiguration;
        this.requestConfiguration = requestConfiguration;
    }

    public static TeaConfiguration create() {
        return new TeaConfiguration(ClientConfiguration.create(), RequestConfiguration.create());
    }

    public String serviceName() {
        return this.clientConfiguration.option(ClientOption.SERVICE_NAME);
    }

    public String region() {
        return this.clientConfiguration.option(ClientOption.REGION);
    }

    public String endpoint() {
        if (CommonUtil.isUnset(requestConfiguration)) {
            return this.clientConfiguration.option(ClientOption.ENDPOINT);
        }
        return Optional.ofNullable(requestConfiguration.endpointOverride())
                .orElseGet(() -> this.clientConfiguration.option(ClientOption.ENDPOINT));
    }

    public TeaConfiguration setEndpoint(String endpoint) {
        clientConfiguration.setOption(ClientOption.ENDPOINT, endpoint);
        return this;
    }

    public String endpointType() {
        return this.clientConfiguration.option(ClientOption.ENDPOINT_TYPE);
    }

    public ServiceConfiguration serviceConfiguration() {
        return this.clientConfiguration.option(ClientOption.SERVICE_CONFIGURATION);
    }

    public Duration connectTimeout() {
        if (CommonUtil.isUnset(requestConfiguration)) {
            return this.clientConfiguration.option(ClientOption.CONNECT_TIMEOUT);
        }
        return Optional.ofNullable(requestConfiguration.connectTimeout())
                .orElseGet(() -> this.clientConfiguration.option(ClientOption.CONNECT_TIMEOUT));
    }

//    public Duration readTimeout() {
//        if (CommonUtil.isUnset(requestConfiguration)) {
//            return this.clientConfiguration.option(ClientOption.READ_TIMEOUT);
//        }
//        return Optional.ofNullable(requestConfiguration.readTimeout())
//                .orElseGet(() -> this.clientConfiguration.option(ClientOption.READ_TIMEOUT));
//    }

    public Duration responseTimeout() {
        if (CommonUtil.isUnset(requestConfiguration)) {
            return this.clientConfiguration.option(ClientOption.RESPONSE_TIMEOUT);
        }
        return Optional.ofNullable(requestConfiguration.responseTimeout())
                .orElseGet(() -> this.clientConfiguration.option(ClientOption.RESPONSE_TIMEOUT));
    }

//    public Duration writeTimeout() {
//        if (CommonUtil.isUnset(requestConfiguration)) {
//            return this.clientConfiguration.option(ClientOption.WRITE_TIMEOUT);
//        }
//        return Optional.ofNullable(requestConfiguration.writeTimeout())
//                .orElseGet(() -> this.clientConfiguration.option(ClientOption.WRITE_TIMEOUT));
//    }

    public String protocol() {
        return Optional.ofNullable(this.clientConfiguration.option(ClientOption.HTTP_PROTOCOL))
                .orElseGet(() -> ProtocolType.HTTPS);
    }

    public HttpMethod method() {
        if (CommonUtil.isUnset(requestConfiguration)) {
            return this.clientConfiguration.option(ClientOption.HTTP_METHOD);
        }
        return Optional.ofNullable(requestConfiguration.httpMethod())
                .orElseGet(() -> this.clientConfiguration.option(ClientOption.HTTP_METHOD));
    }

    public HttpHeaders httpHeaders() {
        HttpHeaders headers = new HttpHeaders();
        if (!CommonUtil.isUnset(this.clientConfiguration.option(ClientOption.ADDITIONAL_HTTP_HEADERS))) {
            headers.putAll(this.clientConfiguration.option(ClientOption.ADDITIONAL_HTTP_HEADERS));
        }
        if (!CommonUtil.isUnset(requestConfiguration) && !CommonUtil.isUnset(requestConfiguration.httpHeaders())) {
            headers.putAll(requestConfiguration.httpHeaders());
        }
        return headers;
    }

    public ICredentialProvider credentialProvider() {
        return this.clientConfiguration.option(ClientOption.CREDENTIALS_PROVIDER);
    }

    public ClientConfiguration clientConfiguration() {
        return this.clientConfiguration;
    }

    public RequestConfiguration requestConfiguration() {
        return this.requestConfiguration;
    }

    @Override
    public TeaConfiguration copy() {
        return new TeaConfiguration(this.clientConfiguration.copy(), this.requestConfiguration.copy());
    }
}
