package darabonba.core.policy.retry.backoff;


import darabonba.core.policy.retry.RetryPolicyContext;
import java.time.Duration;
import java.util.Random;

import static com.aliyun.core.utils.Validate.isNotNegative;


public final class FullJitterBackoffStrategy implements BackoffStrategy {
    private static final Duration BASE_DELAY_CEILING = Duration.ofDays(3);
    private static final Duration MAX_BACKOFF_CEILING = Duration.ofDays(3);

    private final Duration baseDelay;
    private final Duration maxBackoffTime;
    private final Random random;

    private FullJitterBackoffStrategy(BuilderImpl builder) {
        this(builder.baseDelay, builder.maxBackoffTime, new Random());
    }

    FullJitterBackoffStrategy(final Duration baseDelay, final Duration maxBackoffTime, final Random random) {
        this.baseDelay = Duration.ofNanos(Math.min(isNotNegative(baseDelay, "baseDelay").toNanos(), BASE_DELAY_CEILING.toNanos()));
        this.maxBackoffTime = Duration.ofNanos(Math.min(isNotNegative(maxBackoffTime, "maxBackoffTime").toNanos(), MAX_BACKOFF_CEILING.toNanos()));
        this.random = random;
    }

    @Override
    public Duration computeDelayBeforeNextRetry(RetryPolicyContext context) {
        int ceil = calculateExponentialDelay(context.retriesAttempted(), baseDelay, maxBackoffTime);
        return Duration.ofMillis(random.nextInt(ceil));
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public interface Builder {
        Builder baseDelay(Duration baseDelay);

        Duration baseDelay();

        Builder maxBackoffTime(Duration maxBackoffTime);

        Duration maxBackoffTime();

        FullJitterBackoffStrategy build();
    }

    private static final class BuilderImpl implements Builder {

        private Duration baseDelay;
        private Duration maxBackoffTime;

        private BuilderImpl() {
        }

        @Override
        public Builder baseDelay(Duration baseDelay) {
            this.baseDelay = baseDelay;
            return this;
        }

        public void setBaseDelay(Duration baseDelay) {
            baseDelay(baseDelay);
        }

        @Override
        public Duration baseDelay() {
            return baseDelay;
        }

        @Override
        public Builder maxBackoffTime(Duration maxBackoffTime) {
            this.maxBackoffTime = maxBackoffTime;
            return this;
        }

        public void setMaxBackoffTime(Duration maxBackoffTime) {
            maxBackoffTime(maxBackoffTime);
        }

        @Override
        public Duration maxBackoffTime() {
            return maxBackoffTime;
        }

        @Override
        public FullJitterBackoffStrategy build() {
            return new FullJitterBackoffStrategy(this);
        }
    }
}
