/*
 * BaseConfiguration.java
 * 
 * Created 10.12.2007.
 * 
 * Copyright (c) 2007 Johann Burkard (<mailto:jb@eaio.com>) <http://eaio.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
 * documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the
 * Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
package com.eaio.configuration;

import static com.eaio.util.Resource.close;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.lang.SystemUtils;
import org.apache.commons.lang.text.StrBuilder;
import org.apache.log4j.helpers.OptionConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Base configuration class.
 * <p>
 * TODO: Serialize correctly.
 * 
 * @author <a href="mailto:jb@eaio.com">Johann Burkard</a>
 * @version $Id: BaseConfiguration.java 5373 2012-12-12 18:19:20Z johann $
 */
public class BaseConfiguration extends ConcurrentHashMap<String, String> implements Configuration {

    private static final long serialVersionUID = 3749446309723647414L;

    private final Logger LOG = LoggerFactory.getLogger(getClass()); // Use getClass() so that the extending class' name is used.

    private final String PROPERTIES_SUFFIX = ".properties";

    private final File userHome = new File(SystemUtils.USER_HOME);

    private final File userDir = new File(SystemUtils.USER_DIR);

    private static String[] additionalPaths = System.getProperty("configuration.paths", "").split(
            SystemUtils.PATH_SEPARATOR);

    private final String key;

    private final boolean substitute;

    /**
     * Constructor for BaseConfiguration. Performs variable substitution.
     * 
     * @param key the name of the properties files
     * @see #BaseConfiguration(String, boolean)
     */
    public BaseConfiguration(String key) {
        this(key, true);
    }

    /**
     * 
     * Constructor for BaseConfiguration.
     * 
     * @param key the name of the properties files
     * @param substitute
     */
    public BaseConfiguration(String key, boolean substitute) {
        super();
        this.key = key;
        this.substitute = substitute;
        load();
    }

    /**
     * @see com.eaio.configuration.Configuration#load()
     */
    public void load() {
        Properties p = new Properties();
        p.put("key", key);

        tryLoading(buildName(null), p);

        InetAddress[] localHosts = localHosts();
        for (int i = 0; i < localHosts.length; ++i) {
            tryLoading(buildName(localHosts[i].getHostName()), p);
            tryLoading(buildName(localHosts[i].getHostAddress()), p);
        }

        if (substitute) {
            substituteVariables(p);
        }

        for (String key : p.stringPropertyNames()) {
            put(key, p.getProperty(key));
        }

        LOG.debug("loaded configuration for {}", key);
    }

    private void substituteVariables(Properties p) {
        for (Object key : p.keySet()) {
            String val = p.getProperty((String) key);
            if (val.contains("${")) {
                p.setProperty((String) key, OptionConverter.substVars(val, p));
            }
        }
    }

    void tryLoading(String resource, Properties p) {
        File f = null;
        if (userHome != null) {
            f = returnNewFileIfFileAndReadable(f, new File(userHome, resource));
        }
        if (userDir != null && !userDir.equals(userHome)) {
            f = returnNewFileIfFileAndReadable(f, new File(userDir, resource));
        }
        if (additionalPaths != null) {
            for (int i = 0; i < additionalPaths.length; ++i) {
                File additional = new File(additionalPaths[i]);
                if (!additional.equals(userHome) && !additional.equals(userDir)) {
                    f = returnNewFileIfFileAndReadable(f, new File(additional, resource));
                }
            }
        }

        String actualResource;

        InputStream in = null;
        if (f == null) {
            actualResource = new StrBuilder(resource.length() + 1).append('/').append(resource).toString();
            in = BaseConfiguration.class.getResourceAsStream(actualResource);
        }
        else {
            actualResource = f.getAbsolutePath();
            try {
                in = new FileInputStream(f);
            }
            catch (FileNotFoundException ex) {
                // Can't really be since checked before
                LOG.error("could not find file {} even though checked before", f, ex);
            }
        }
        if (in != null) {
            try {
                LOG.debug("loading from {}", actualResource);
                p.load(in);
            }
            catch (IOException ex) {
                LOG.warn("while loading " + actualResource + " from " + resource, ex);
            }
            finally {
                close(in);
            }
        }

    }

    /**
     * Returns <code>newFile</code> if existing, not a directory and readable. <code>curr</code> otherwise.
     * 
     * @param curr the current file
     * @param newFile the new file
     * @return newFile or curr
     */
    File returnNewFileIfFileAndReadable(File curr, File newFile) {
        LOG.trace("checking {}", newFile.getAbsolutePath());
        return newFile.isFile() && newFile.canRead() ? newFile : curr;
    }

    String buildName(String middle) {
        StrBuilder out = new StrBuilder();
        out.append(key);
        if (middle != null) {
            out.append('.');
            out.append(middle);
        }
        out.append(PROPERTIES_SUFFIX);
        return out.toString();
    }

    /**
     * Returns a list of local hosts.
     * 
     * @return a list of local hosts
     */
    private InetAddress[] localHosts() {
        InetAddress localHost = null;
        InetAddress[] out = null;

        try {
            localHost = InetAddress.getLocalHost();
            if (localHost != null) {
                out = InetAddress.getAllByName(localHost.getHostName());
            }
        }
        catch (UnknownHostException ex) {
            LOG.warn("unknown host", ex);
        }

        if (out == null) {
            if (localHost != null) {
                out = new InetAddress[] { localHost };
            }
            else {
                out = new InetAddress[0];
            }
        }

        return out;
    }

}
