/**
 * 
 */
package com.eaio.io;

import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;

/**
 * {@link FilterOutputStream} that only serves a slice of the data written to it.
 *
 * @author <a href="mailto:jb&#64;eaio.com">Johann Burkard</a>
 * @version $Id: RangeOutputStream.java 5118 2012-08-26 13:26:52Z johann $
 */
public class RangeOutputStream extends FilterOutputStream {
	
	private long start, end;

	/**
	 * @param out the original output stream
	 * @param start the start position
	 */
	public RangeOutputStream(OutputStream out, long start) {
		super(out);
        assert start >= 0;
        this.start = start;
        this.end = Long.MIN_VALUE;
	}
	
	/**
	 * @param out the original output stream
	 * @param start the start position
	 * @param end the end position
	 */
	public RangeOutputStream(OutputStream out, long start, long end) {
		super(out);
		assert start >= 0;
		assert end >= start;
		this.start = start;
		this.end = end;
	}

	@Override
	public void write(int b) throws IOException {
		--start;
		
		if (end == Long.MIN_VALUE) { // Only start is set
			if (start < 0L) {
				out.write(b);				
			}
		}
		else {
			--end;
			
			if (start < 0L && end >= 0L) {
				super.write(b);				
			}
		}
	}

	/**
	 * Only partially delegates to {@link FilterOutputStream#write(byte[], int, int)} because of decreased write performance.
	 */
	@Override
	public void write(byte[] b, int off, int len) throws IOException {
		if ((off | len | (b.length - (len + off)) | (off + len)) < 0)
		    throw new IndexOutOfBoundsException();
		
		if (end == Long.MIN_VALUE) { // Only start is set
			if (start >= len) {
				start -= len;
			}
			else if (start < 0L) {
				out.write(b, off, len);
				start -= len;
			}
			else {
				super.write(b, off, len);
			}
		}
		else {
			
			if (start < 0L && end >= len) {
				out.write(b, off, len);
				start -= len;
				end -= len;
			}
			else if (start >= len || end < 0) {
				start -= len;
				end -= len;
			}
			else {
				super.write(b, off, len);
			}
			
		}
	}
	
}
