/*
 * Copyright (c) 2021 SAP SE or an SAP affiliate company. All rights reserved.
 */

package com.google.gson.internal.bind;

import java.io.IOException;
import java.lang.reflect.Type;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.annotations.Beta;
import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;
import com.google.gson.TypeAdapter;
import com.google.gson.TypeAdapterFactory;
import com.google.gson.internal.$Gson$Types;
import com.google.gson.internal.ConstructorConstructor;
import com.google.gson.internal.JsonReaderInternalAccess;
import com.google.gson.internal.ObjectConstructor;
import com.google.gson.reflect.TypeToken;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonToken;
import com.google.gson.stream.JsonWriter;

import io.vavr.Tuple;
import io.vavr.Tuple2;

/**
 * Experimental GSON type adapter factory to manage serialization and deserialization of Map types. It's a variant of
 * {@link MapTypeAdapterFactory}. The main feature is additional access to custom TypeAdapters that can be registered
 * for native JSON types, see {@link JsonToken}. This class becomes obsolete once
 * https://github.com/google/gson/pull/1290 is merged.
 */
@Beta
public final class CustomMapTypeAdapterFactory implements TypeAdapterFactory
{
    private final ConstructorConstructor constructorConstructor = new ConstructorConstructor(Collections.emptyMap());

    private final Map<JsonToken, Tuple2<Predicate<Object>, TypeAdapter<?>>> customAdapters = new LinkedHashMap<>();

    /**
     * Register a custom type adapter for native JSON primitive types. Used to override the default GSON behavior.
     * 
     * @param type
     *            The primitive JSON type.
     * @param ableToWrite
     *            A predicate to check whether a generic object can be used to serialize using provided adapter.
     * @param adapter
     *            The actual type adapter to be used for the primitive JSON type.
     */
    public void registerAdapter(
        @Nonnull final JsonToken type,
        @Nonnull final Predicate<Object> ableToWrite,
        @Nonnull final TypeAdapter<?> adapter )
    {
        customAdapters.put(type, Tuple.of(ableToWrite, adapter));
    }

    @SuppressWarnings( { "unchecked", "rawtypes" } )
    @Override
    @Nullable
    public <T> TypeAdapter<T> create( @Nonnull final Gson gson, @Nonnull final TypeToken<T> typeToken )
    {
        final Type type = typeToken.getType();
        if( !Map.class.isAssignableFrom(typeToken.getRawType()) ) {
            return null;
        }
        final Type[] keyAndValueTypes = $Gson$Types.getMapKeyAndValueTypes(type, $Gson$Types.getRawType(type));
        final Type keyType = keyAndValueTypes[0];
        final Type valueType = keyAndValueTypes[1];

        return new CustomMapTypeAdapterFactory.Adapter(
            gson,
            keyType,
            getKeyTypeAdapter(gson, keyType),
            valueType,
            getValueTypeAdapter(gson, valueType),
            constructorConstructor.get(typeToken));
    }

    private TypeAdapter<?> getValueTypeAdapter( @Nonnull final Gson gson, @Nonnull final Type valueType )
    {
        final TypeAdapter<?> valueAdapter = gson.getAdapter(TypeToken.get(valueType));
        return new TypeAdapter<Object>()
        {
            @SuppressWarnings( "unchecked" )
            @Override
            public void write( @Nonnull final JsonWriter out, @Nullable final Object value )
                throws IOException
            {
                for( final Tuple2<Predicate<Object>, TypeAdapter<?>> adapterEntry : customAdapters.values() ) {
                    if( adapterEntry._1().test(value) ) {
                        ((TypeAdapter<Object>) adapterEntry._2()).write(out, value);
                        return;
                    }
                }
                ((TypeAdapter<Object>) valueAdapter).write(out, value);
            }

            @Override
            @Nullable
            public Object read( @Nonnull final JsonReader in )
                throws IOException
            {
                final Tuple2<Predicate<Object>, TypeAdapter<?>> adapterEntry = customAdapters.get(in.peek());
                if( adapterEntry != null ) {
                    return adapterEntry._2().read(in);
                }
                return valueAdapter.read(in);
            }
        };
    }

    private TypeAdapter<?> getKeyTypeAdapter( @Nonnull final Gson gson, @Nonnull final Type keyType )
    {
        return keyType == boolean.class || keyType == Boolean.class
            ? TypeAdapters.BOOLEAN_AS_STRING
            : gson.getAdapter(TypeToken.get(keyType));
    }

    private static final class Adapter<K, V> extends TypeAdapter<Map<K, V>>
    {
        private final TypeAdapter<K> keyTypeAdapter;
        private final TypeAdapter<V> valueTypeAdapter;
        private final ObjectConstructor<? extends Map<K, V>> constructor;

        private Adapter(
            @Nonnull final Gson context,
            @Nullable final Type keyType,
            @Nonnull final TypeAdapter<K> keyTypeAdapter,
            @Nullable final Type valueType,
            @Nonnull final TypeAdapter<V> valueTypeAdapter,
            @Nonnull final ObjectConstructor<? extends Map<K, V>> constructor )
        {
            this.keyTypeAdapter = new TypeAdapterRuntimeTypeWrapper<>(context, keyTypeAdapter, keyType);
            this.valueTypeAdapter = new TypeAdapterRuntimeTypeWrapper<>(context, valueTypeAdapter, valueType);
            this.constructor = constructor;
        }

        @Override
        @Nullable
        public Map<K, V> read( @Nonnull final JsonReader in )
            throws IOException
        {
            final JsonToken peek = in.peek();
            if( peek == JsonToken.NULL ) {
                in.nextNull();
                return null;
            }

            final Map<K, V> map = constructor.construct();

            if( peek == JsonToken.BEGIN_ARRAY ) {
                in.beginArray();
                while( in.hasNext() ) {
                    in.beginArray(); // entry array
                    final K key = keyTypeAdapter.read(in);
                    final V value = valueTypeAdapter.read(in);
                    final V replaced = map.put(key, value);
                    if( replaced != null ) {
                        throw new JsonSyntaxException("duplicate key: " + key);
                    }
                    in.endArray();
                }
                in.endArray();
            } else {
                in.beginObject();
                while( in.hasNext() ) {
                    JsonReaderInternalAccess.INSTANCE.promoteNameToValue(in);
                    final K key = keyTypeAdapter.read(in);
                    final V value = valueTypeAdapter.read(in);
                    final V replaced = map.put(key, value);
                    if( replaced != null ) {
                        throw new JsonSyntaxException("duplicate key: " + key);
                    }
                }
                in.endObject();
            }
            return map;
        }

        @Override
        public void write( @Nonnull final JsonWriter out, @Nullable final Map<K, V> map )
            throws IOException
        {
            if( map == null ) {
                out.nullValue();
                return;
            }

            out.beginObject();
            for( final Map.Entry<K, V> entry : map.entrySet() ) {
                out.name(String.valueOf(entry.getKey()));
                valueTypeAdapter.write(out, entry.getValue());
            }
            out.endObject();
        }
    }
}
