/*
 * Decompiled with CFR 0.152.
 */
package com.xxl.tool.concurrent;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimeWheel {
    private static Logger logger = LoggerFactory.getLogger(TimeWheel.class);
    private final int ticksCount;
    private final long tickDuration;
    private final long wheelDuration;
    private final Map<Integer, Set<TimeTask>> taskSlots;
    private volatile int currentTick = 0;
    private volatile long lastTickTime;
    private final AtomicBoolean running = new AtomicBoolean(false);
    private final ExecutorService taskExecutor;

    public TimeWheel(int ticksCount, long tickDuration) {
        this(ticksCount, tickDuration, 10, 200, 2000);
    }

    public TimeWheel(int ticksCount, long tickDuration, int taskExecutorCoreSize, int taskExecutorMaxSize, int taskExecutorQueueSize) {
        this(ticksCount, tickDuration, new ThreadPoolExecutor(taskExecutorCoreSize, taskExecutorMaxSize, 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(taskExecutorQueueSize)));
    }

    public TimeWheel(int ticksCount, long tickDuration, ExecutorService taskExecutor) {
        if (ticksCount < 1 || tickDuration < 1L) {
            throw new IllegalArgumentException("ticksCount and tickDuration must be greater than 0");
        }
        this.ticksCount = ticksCount;
        this.tickDuration = tickDuration;
        this.wheelDuration = (long)ticksCount * tickDuration;
        this.taskSlots = new ConcurrentHashMap<Integer, Set<TimeTask>>();
        for (int i = 0; i < ticksCount; ++i) {
            this.taskSlots.put(i, ConcurrentHashMap.newKeySet());
        }
        this.taskExecutor = taskExecutor;
    }

    public void start() {
        if (this.running.compareAndSet(false, true)) {
            long now = System.currentTimeMillis();
            this.lastTickTime = now - now % 1000L;
            Thread wheelThread = new Thread(() -> {
                while (this.running.get()) {
                    long current = System.currentTimeMillis();
                    long diff = current - this.lastTickTime;
                    if (diff >= this.tickDuration) {
                        int ticksToMove = (int)(diff / this.tickDuration);
                        for (int i = 0; i < ticksToMove; ++i) {
                            int tickToProcess = (this.currentTick + i) % this.ticksCount;
                            this.executeTasks(tickToProcess);
                        }
                        this.currentTick = (this.currentTick + ticksToMove) % this.ticksCount;
                        this.lastTickTime += (long)ticksToMove * this.tickDuration;
                    }
                    try {
                        Thread.sleep(Math.max(1L, this.tickDuration - (System.currentTimeMillis() - this.lastTickTime)));
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        break;
                    }
                }
            });
            wheelThread.setDaemon(true);
            wheelThread.start();
            Runtime.getRuntime().addShutdownHook(new Thread(() -> this.stop()));
            logger.info(">>>>>>>>>>> TimeWheel[hashCode = " + this.hashCode() + "] started");
        }
    }

    public void stop() {
        this.running.set(false);
        this.taskExecutor.shutdown();
        try {
            if (!this.taskExecutor.awaitTermination(10L, TimeUnit.SECONDS)) {
                this.taskExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.taskExecutor.shutdownNow();
            logger.error(">>>>>>>>>>> TimeWheel[hashCode = " + this.hashCode() + "] stop error:{}", (Object)e.getMessage(), (Object)e);
        }
        logger.info(">>>>>>>>>>> TimeWheel[hashCode = " + this.hashCode() + "] stopped.");
    }

    public boolean submitTask(long executeTime, Runnable task) {
        long now = System.currentTimeMillis();
        if (executeTime <= now || executeTime > now + this.wheelDuration) {
            return false;
        }
        TimeTask timeTask = new TimeTask(executeTime, task);
        int targetTick = this.calculateTick(executeTime);
        this.taskSlots.get(targetTick).add(timeTask);
        return true;
    }

    private int calculateTick(long executeTime) {
        long delay = executeTime - System.currentTimeMillis();
        int ticks = (int)(delay / this.tickDuration);
        return (this.currentTick + ticks) % this.ticksCount;
    }

    private void executeTasks(int tick) {
        Set<TimeTask> tasks = this.taskSlots.get(tick);
        long now = System.currentTimeMillis();
        tasks.removeIf(task -> {
            if (((TimeTask)task).executeTime <= now) {
                try {
                    this.taskExecutor.submit(((TimeTask)task).task);
                }
                catch (Exception e) {
                    logger.error(e.getMessage(), (Throwable)e);
                }
                return true;
            }
            return false;
        });
    }

    private static class TimeTask {
        private final long executeTime;
        private final Runnable task;

        public TimeTask(long executeTime, Runnable task) {
            this.executeTime = executeTime;
            this.task = task;
        }
    }
}

