/*
 * Decompiled with CFR 0.152.
 */
package org.molgenis.data.validation;

import com.google.common.collect.Lists;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.internal.constraintvalidators.EmailValidator;
import org.molgenis.data.Entity;
import org.molgenis.data.Range;
import org.molgenis.data.meta.AttributeType;
import org.molgenis.data.meta.model.Attribute;
import org.molgenis.data.meta.model.EntityType;
import org.molgenis.data.util.EntityUtils;
import org.molgenis.data.validation.ConstraintViolation;
import org.molgenis.data.validation.ExpressionValidator;
import org.molgenis.util.UnexpectedEnumException;
import org.springframework.stereotype.Component;

@Component
public class EntityAttributesValidator {
    private final ExpressionValidator expressionValidator;
    private EmailValidator emailValidator;

    public EntityAttributesValidator(ExpressionValidator expressionValidator) {
        this.expressionValidator = Objects.requireNonNull(expressionValidator);
    }

    public Set<ConstraintViolation> validate(Entity entity, EntityType meta) {
        Set<ConstraintViolation> violations = this.checkNullableExpressions(entity, meta);
        violations.addAll(this.checkValidationExpressions(entity, meta));
        for (Attribute attr : meta.getAtomicAttributes()) {
            ConstraintViolation violation = null;
            AttributeType attrType = attr.getDataType();
            switch (attrType) {
                case EMAIL: {
                    violation = this.checkEmail(entity, attr, meta);
                    break;
                }
                case BOOL: {
                    violation = EntityAttributesValidator.checkBoolean(entity, attr, meta);
                    break;
                }
                case DATE: {
                    violation = EntityAttributesValidator.checkDate(entity, attr, meta);
                    break;
                }
                case DATE_TIME: {
                    violation = EntityAttributesValidator.checkDateTime(entity, attr, meta);
                    break;
                }
                case DECIMAL: {
                    violation = EntityAttributesValidator.checkDecimal(entity, attr, meta);
                    break;
                }
                case HYPERLINK: {
                    violation = this.checkHyperlink(entity, attr, meta);
                    break;
                }
                case INT: {
                    violation = EntityAttributesValidator.checkInt(entity, attr, meta);
                    if (violation != null || attr.getRange() == null) break;
                    violation = EntityAttributesValidator.checkRange(entity, attr, meta);
                    break;
                }
                case LONG: {
                    violation = EntityAttributesValidator.checkLong(entity, attr, meta);
                    if (violation != null || attr.getRange() == null) break;
                    violation = EntityAttributesValidator.checkRange(entity, attr, meta);
                    break;
                }
                case ENUM: {
                    violation = this.checkEnum(entity, attr, meta);
                    break;
                }
                case HTML: {
                    violation = EntityAttributesValidator.checkText(entity, attr, meta, AttributeType.HTML);
                    break;
                }
                case SCRIPT: {
                    violation = EntityAttributesValidator.checkText(entity, attr, meta, AttributeType.SCRIPT);
                    break;
                }
                case TEXT: {
                    violation = EntityAttributesValidator.checkText(entity, attr, meta, AttributeType.TEXT);
                    break;
                }
                case STRING: {
                    violation = EntityAttributesValidator.checkText(entity, attr, meta, AttributeType.STRING);
                    break;
                }
                case CATEGORICAL: 
                case FILE: 
                case XREF: {
                    violation = this.checkXref(entity, attr, meta);
                    break;
                }
                case CATEGORICAL_MREF: 
                case MREF: 
                case ONE_TO_MANY: {
                    violation = this.checkMref(entity, attr, meta);
                    break;
                }
                case COMPOUND: {
                    break;
                }
                default: {
                    throw new UnexpectedEnumException((Enum)attrType);
                }
            }
            if (violation == null) continue;
            violations.add(violation);
        }
        return violations;
    }

    private ConstraintViolation checkMref(Entity entity, Attribute attr, EntityType entityType) {
        Iterable refEntities;
        try {
            refEntities = entity.getEntities(attr.getName());
        }
        catch (Exception e) {
            return this.createConstraintViolation(entity, attr, entityType, "Not a valid entity, expected an entity list.");
        }
        if (refEntities == null) {
            return this.createConstraintViolation(entity, attr, entityType, "Not a valid entity, expected an entity list.");
        }
        for (Entity refEntity : refEntities) {
            if (refEntity == null) {
                return this.createConstraintViolation(entity, attr, entityType, "Not a valid entity, null is not allowed");
            }
            if (refEntity.getEntityType().getId().equals(attr.getRefEntity().getId())) continue;
            return this.createConstraintViolation(entity, attr, entityType, "Not a valid entity type.");
        }
        return null;
    }

    private ConstraintViolation checkXref(Entity entity, Attribute attr, EntityType entityType) {
        Entity refEntity;
        try {
            refEntity = entity.getEntity(attr.getName());
        }
        catch (Exception e) {
            return this.createConstraintViolation(entity, attr, entityType, "Not a valid entity.");
        }
        if (refEntity == null) {
            return null;
        }
        if (!refEntity.getEntityType().getId().equals(attr.getRefEntity().getId())) {
            return this.createConstraintViolation(entity, attr, entityType, "Not a valid entity type.");
        }
        return null;
    }

    private Set<ConstraintViolation> checkNullableExpressions(Entity entity, EntityType entityType) {
        ArrayList<String> nullableExpressions = new ArrayList<String>();
        ArrayList<Attribute> expressionAttributes = new ArrayList<Attribute>();
        for (Attribute attribute : entityType.getAtomicAttributes()) {
            String nullableExpression = attribute.getNullableExpression();
            if (nullableExpression == null) continue;
            expressionAttributes.add(attribute);
            nullableExpressions.add(nullableExpression);
        }
        LinkedHashSet<ConstraintViolation> violations = new LinkedHashSet<ConstraintViolation>();
        if (!nullableExpressions.isEmpty()) {
            List<Boolean> results = this.expressionValidator.resolveBooleanExpressions(nullableExpressions, entity);
            for (int i = 0; i < results.size(); ++i) {
                if (results.get(i).booleanValue() || !EntityUtils.isNullValue((Entity)entity, (Attribute)((Attribute)expressionAttributes.get(i)))) continue;
                violations.add(this.createConstraintViolation(entity, (Attribute)expressionAttributes.get(i), entityType, String.format("Offended nullable expression: %s", nullableExpressions.get(i))));
            }
        }
        return violations;
    }

    private Set<ConstraintViolation> checkValidationExpressions(Entity entity, EntityType meta) {
        ArrayList<String> validationExpressions = new ArrayList<String>();
        ArrayList<Attribute> expressionAttributes = new ArrayList<Attribute>();
        for (Attribute attribute : meta.getAtomicAttributes()) {
            if (!StringUtils.isNotBlank((CharSequence)attribute.getValidationExpression())) continue;
            expressionAttributes.add(attribute);
            validationExpressions.add(attribute.getValidationExpression());
        }
        LinkedHashSet<ConstraintViolation> violations = new LinkedHashSet<ConstraintViolation>();
        if (!validationExpressions.isEmpty()) {
            List<Boolean> results = this.expressionValidator.resolveBooleanExpressions(validationExpressions, entity);
            for (int i = 0; i < results.size(); ++i) {
                if (results.get(i).booleanValue()) continue;
                violations.add(this.createConstraintViolation(entity, (Attribute)expressionAttributes.get(i), meta, String.format("Offended validation expression: %s", validationExpressions.get(i))));
            }
        }
        return violations;
    }

    private ConstraintViolation checkEmail(Entity entity, Attribute attribute, EntityType entityType) {
        String email = entity.getString(attribute.getName());
        if (email == null) {
            return null;
        }
        if (this.emailValidator == null) {
            this.emailValidator = new EmailValidator();
        }
        if (!this.emailValidator.isValid((CharSequence)email, null)) {
            return this.createConstraintViolation(entity, attribute, entityType, "Not a valid e-mail address.");
        }
        if ((long)email.length() > AttributeType.EMAIL.getMaxLength()) {
            return EntityAttributesValidator.createConstraintViolation(entity, attribute, entityType);
        }
        return null;
    }

    private static ConstraintViolation checkBoolean(Entity entity, Attribute attribute, EntityType entityType) {
        try {
            entity.getBoolean(attribute.getName());
            return null;
        }
        catch (Exception e) {
            return EntityAttributesValidator.createConstraintViolation(entity, attribute, entityType);
        }
    }

    private static ConstraintViolation checkDateTime(Entity entity, Attribute attribute, EntityType entityType) {
        try {
            entity.getInstant(attribute.getName());
            return null;
        }
        catch (Exception e) {
            return EntityAttributesValidator.createConstraintViolation(entity, attribute, entityType);
        }
    }

    private static ConstraintViolation checkDate(Entity entity, Attribute attribute, EntityType entityType) {
        try {
            entity.getLocalDate(attribute.getName());
            return null;
        }
        catch (Exception e) {
            return EntityAttributesValidator.createConstraintViolation(entity, attribute, entityType);
        }
    }

    private static ConstraintViolation checkDecimal(Entity entity, Attribute attribute, EntityType entityType) {
        try {
            entity.getDouble(attribute.getName());
            return null;
        }
        catch (Exception e) {
            return EntityAttributesValidator.createConstraintViolation(entity, attribute, entityType);
        }
    }

    private ConstraintViolation checkHyperlink(Entity entity, Attribute attribute, EntityType entityType) {
        String link = entity.getString(attribute.getName());
        if (link == null) {
            return null;
        }
        try {
            new URI(link);
        }
        catch (URISyntaxException e) {
            return this.createConstraintViolation(entity, attribute, entityType, "Not a valid hyperlink.");
        }
        if ((long)link.length() > AttributeType.HYPERLINK.getMaxLength()) {
            return EntityAttributesValidator.createConstraintViolation(entity, attribute, entityType);
        }
        return null;
    }

    private static ConstraintViolation checkInt(Entity entity, Attribute attribute, EntityType entityType) {
        try {
            entity.getInt(attribute.getName());
            return null;
        }
        catch (Exception e) {
            return EntityAttributesValidator.createConstraintViolation(entity, attribute, entityType);
        }
    }

    private static ConstraintViolation checkLong(Entity entity, Attribute attribute, EntityType entityType) {
        try {
            entity.getLong(attribute.getName());
            return null;
        }
        catch (Exception e) {
            return EntityAttributesValidator.createConstraintViolation(entity, attribute, entityType);
        }
    }

    private static ConstraintViolation checkRange(Entity entity, Attribute attr, EntityType entityType) {
        Long value;
        Range range = attr.getRange();
        switch (attr.getDataType()) {
            case INT: {
                Integer intValue = entity.getInt(attr.getName());
                value = intValue != null ? Long.valueOf(intValue.longValue()) : null;
                break;
            }
            case LONG: {
                value = entity.getLong(attr.getName());
                break;
            }
            default: {
                throw new RuntimeException(String.format("Range not allowed for data type [%s]", attr.getDataType().toString()));
            }
        }
        if (value != null && (range.getMin() != null && value < range.getMin() || range.getMax() != null && value > range.getMax())) {
            return EntityAttributesValidator.createConstraintViolation(entity, attr, entityType);
        }
        return null;
    }

    private static ConstraintViolation checkText(Entity entity, Attribute attribute, EntityType meta, AttributeType fieldType) {
        String text = entity.getString(attribute.getName());
        if (text == null) {
            return null;
        }
        if ((long)text.length() > fieldType.getMaxLength()) {
            return EntityAttributesValidator.createConstraintViolation(entity, attribute, meta);
        }
        return null;
    }

    private ConstraintViolation checkEnum(Entity entity, Attribute attribute, EntityType entityType) {
        List enumOptions;
        String value = entity.getString(attribute.getName());
        if (value != null && !(enumOptions = attribute.getEnumOptions()).contains(value)) {
            return this.createConstraintViolation(entity, attribute, entityType, "Value must be one of " + enumOptions.toString());
        }
        return null;
    }

    private static ConstraintViolation createConstraintViolation(Entity entity, Attribute attribute, EntityType entityType) {
        Long maxLength;
        String message = String.format("Invalid %s value '%s' for attribute '%s' of entity '%s'.", attribute.getDataType().toString().toLowerCase(), entity.get(attribute.getName()), attribute.getLabel(), entityType.getId());
        Range range = attribute.getRange();
        if (range != null) {
            message = message + String.format("Value must be between %d and %d", range.getMin(), range.getMax());
        }
        if ((maxLength = attribute.getDataType().getMaxLength()) != null) {
            message = message + String.format("Value must be less than or equal to %d characters", maxLength);
        }
        return new ConstraintViolation(message);
    }

    private ConstraintViolation createConstraintViolation(Entity entity, Attribute attribute, EntityType entityType, String message) {
        Object value = this.getDataValuesForType(entity, attribute);
        String dataValue = value != null ? value.toString() : null;
        String fullMessage = String.format("Invalid [%s] value [%s] for attribute [%s] of entity [%s] with type [%s].", attribute.getDataType().toString().toLowerCase(), dataValue, attribute.getLabel(), entity.getLabelValue(), entityType.getId());
        fullMessage = fullMessage + " " + message;
        return new ConstraintViolation(fullMessage);
    }

    private Object getDataValuesForType(Entity entity, Attribute attribute) {
        String attributeName = attribute.getName();
        switch (attribute.getDataType()) {
            case DATE: {
                return entity.getLocalDate(attributeName);
            }
            case DATE_TIME: {
                return entity.getInstant(attributeName);
            }
            case BOOL: {
                return entity.getBoolean(attributeName);
            }
            case DECIMAL: 
            case INT: 
            case LONG: {
                return entity.getInt(attributeName);
            }
            case EMAIL: 
            case HYPERLINK: 
            case ENUM: 
            case HTML: 
            case SCRIPT: 
            case TEXT: 
            case STRING: {
                return entity.getString(attributeName);
            }
            case CATEGORICAL: 
            case FILE: 
            case XREF: {
                Entity refEntity = entity.getEntity(attributeName);
                if (refEntity != null) {
                    return refEntity.getIdValue();
                }
                return "";
            }
            case CATEGORICAL_MREF: 
            case MREF: {
                ArrayList mrefValues = Lists.newArrayList();
                for (Entity mrefEntity : entity.getEntities(attributeName)) {
                    if (mrefEntity == null) continue;
                    mrefValues.add(mrefEntity.getIdValue().toString());
                }
                return mrefValues;
            }
            case COMPOUND: {
                return "";
            }
        }
        return "";
    }
}

