package com.aliyun.sdk.gateway.pop.clients;

import com.aliyun.core.utils.SdkAutoCloseable;
import com.aliyun.sdk.gateway.pop.Configuration;
import com.aliyun.sdk.gateway.pop.auth.SignatureVersion;
import com.aliyun.sdk.gateway.pop.models.CommonRequest;
import com.aliyun.sdk.gateway.pop.models.CommonResponse;
import com.aliyun.sdk.gateway.pop.sse.ResponseBodyIterator;
import darabonba.core.*;
import darabonba.core.client.ClientConfiguration;
import darabonba.core.client.ClientExecutionParams;
import darabonba.core.sse.SSEHttpResponseHandler;

import java.util.concurrent.CompletableFuture;

public final class CommonAsyncClient implements SdkAutoCloseable {
    private final TeaAsyncHandler handler;

    CommonAsyncClient(ClientConfiguration configuration) {
        this.handler = new TeaAsyncHandler(configuration);
    }

    public static CommonAsyncClientBuilder builder() {
        return new CommonAsyncClientBuilder().serviceConfiguration(Configuration.create().setSignatureVersion(SignatureVersion.V3));
    }

    static CommonAsyncClient create() {
        return builder().build();
    }

    @Override
    public void close() {
        this.handler.close();
    }

    public ResponseIterable<String> callSseApi(CommonRequest request) {
        this.handler.validateRequestModel(request);
        TeaRequest teaRequest = TeaRequest.create()
                .setProduct(request.getProduct())
                .setVersion(request.getVersion())
                .setStyle(RequestStyle.SSE)
                .setAction(request.getAction())
                .setMethod(request.getHttpMethod())
                .setPathRegex(request.getPath())
                .setReqBodyType(request.getRequestBodyType())
                .setBodyType(request.getResponseBodyType())
                .setBodyIsForm(request.isFormData())
                .formModel(request);
        ResponseBodyIterator iterator = ResponseBodyIterator.create();
        ClientExecutionParams<RequestModel, TeaModel> params = new ClientExecutionParams<>()
                .withInput(request)
                .withRequest(teaRequest)
                .withHttpResponseHandler(new SSEHttpResponseHandler(iterator));
        this.handler.execute(params);
        return new ResponseIterable<>(iterator);
    }

    public CompletableFuture<CommonResponse> callApi(CommonRequest request) {
        try {
            this.handler.validateRequestModel(request);
            TeaRequest teaRequest = TeaRequest.create()
                    .setProduct(request.getProduct())
                    .setVersion(request.getVersion())
                    .setStyle(request.getPath().equals("/") ? RequestStyle.RPC : RequestStyle.RESTFUL)
                    .setAction(request.getAction())
                    .setMethod(request.getHttpMethod())
                    .setPathRegex(request.getPath())
                    .setReqBodyType(request.getRequestBodyType())
                    .setBodyType(request.getResponseBodyType())
                    .setBodyIsForm(request.isFormData())
                    .formModel(request);
            ClientExecutionParams<CommonRequest, CommonResponse> params = new ClientExecutionParams<CommonRequest, CommonResponse>()
                    .withInput(request)
                    .withRequest(teaRequest)
                    .withOutput(CommonResponse.create());
            return this.handler.execute(params);
        } catch (Exception e) {
            CompletableFuture<CommonResponse> future = new CompletableFuture<>();
            future.completeExceptionally(e);
            return future;
        }
    }

}
