package com.aliyun.sdk.gateway.pop.auth.signer;

import com.aliyun.auth.signature.signer.SignAlgorithmHmacSHA256;
import com.aliyun.auth.signature.signer.SignAlgorithmHmacSM3;
import com.aliyun.sdk.gateway.pop.auth.SignatureAlgorithm;
import com.aliyun.sdk.gateway.pop.auth.SignatureVersion;
import org.bouncycastle.crypto.macs.HMac;
import org.bouncycastle.crypto.params.KeyParameter;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import java.io.UnsupportedEncodingException;
import java.security.*;

public class PopV4Signer implements PopSigner {
    private final String ENCODING = "UTF-8";
    private final String HASH_SHA256 = "SHA-256";
    private final String HASH_SM3 = "SM3";
    private final String PEM_BEGIN = "-----BEGIN RSA PRIVATE KEY-----\n";
    private final String PEM_END = "\n-----END RSA PRIVATE KEY-----";
    private final SignatureAlgorithm DEFAULT_ALGORITHM = SignatureAlgorithm.ACS3_HMAC_SHA256;
    private final SignatureAlgorithm algorithm;

    public PopV4Signer() {
        this.algorithm = DEFAULT_ALGORITHM;
    }

    public PopV4Signer(SignatureAlgorithm algorithm) {
        if (algorithm != null) {
            this.algorithm = algorithm;
        } else {
            this.algorithm = DEFAULT_ALGORITHM;
        }
    }

    @Override
    public byte[] signString(String stringToSign, byte[] signingKey) {
        try {
            switch (algorithm) {
                case ACS4_HMAC_SM3:
                    HMac sm3HMAC = SignAlgorithmHmacSM3.HMAC_SM3.getMac();
                    SecretKey key = new SecretKeySpec(signingKey, SignAlgorithmHmacSM3.HMAC_SM3.toString());
                    byte[] sm3Sign = new byte[sm3HMAC.getMacSize()];
                    byte[] inputBytes = stringToSign.getBytes(ENCODING);
                    sm3HMAC.init(new KeyParameter(key.getEncoded()));
                    sm3HMAC.update(inputBytes, 0, inputBytes.length);
                    sm3HMAC.doFinal(sm3Sign, 0);
                    return sm3Sign;
                case ACS4_HMAC_SHA256:
                default:
                    Mac sha256HMAC = SignAlgorithmHmacSHA256.HmacSHA256.getMac();
                    sha256HMAC.init(new SecretKeySpec(signingKey, SignAlgorithmHmacSHA256.HmacSHA256.toString()));
                    return sha256HMAC.doFinal(stringToSign.getBytes(ENCODING));
            }
        } catch (UnsupportedEncodingException | InvalidKeyException e) {
            throw new com.aliyun.auth.signature.exception.SignatureException(e.toString());
        }

    }

    @Override
    public byte[] signString(String stringToSign, String signingKey) {
        try {
            return signString(stringToSign, signingKey.getBytes(ENCODING));
        } catch (UnsupportedEncodingException e) {
            throw new com.aliyun.auth.signature.exception.SignatureException(e.toString());
        }
    }

    @Override
    public byte[] hash(byte[] raw) {
        if (null == raw) {
            return null;
        }
        try {
            MessageDigest digest;
            switch (algorithm) {
                case ACS3_HMAC_SM3:
                    BouncyCastleProvider provider = new BouncyCastleProvider();
                    digest = MessageDigest.getInstance(HASH_SM3, provider);
                    break;
                case ACS3_HMAC_SHA256:
                case ACS3_RSA_SHA256:
                default:
                    digest = MessageDigest.getInstance(HASH_SHA256);
            }
            return digest.digest(raw);
        } catch (NoSuchAlgorithmException e) {
            throw new com.aliyun.auth.signature.exception.SignatureException(e.toString());
        }
    }

    @Override
    public String getContent() {
        switch (algorithm) {
            case ACS4_HMAC_SM3:
                return "x-acs-content-sm3";
            case ACS3_HMAC_SHA256:
            default:
                return "x-acs-content-sha256";
        }
    }

    @Override
    public SignatureAlgorithm getSignerName() {
        return algorithm;
    }

    @Override
    public SignatureVersion getSignerVersion() {
        return SignatureVersion.V4;
    }
}
