package darabonba.core.internal.async;

import darabonba.core.async.AsyncRequestBody;
import org.reactivestreams.Subscriber;
import org.reactivestreams.Subscription;

import java.nio.ByteBuffer;
import java.util.Optional;

public final class ByteArrayAsyncRequestBody implements AsyncRequestBody {

    private final byte[] bytes;

    public ByteArrayAsyncRequestBody(byte[] bytes) {
        this.bytes = bytes.clone();
    }

    @Override
    public Optional<Long> contentLength() {
        return Optional.of((long) bytes.length);
    }

    @Override
    public Optional<String> contentType() {
        return Optional.empty();
    }

    @Override
    public void subscribe(Subscriber<? super ByteBuffer> s) {
        if (s == null) {
            throw new NullPointerException("Subscription is null.");
        }

        try {
            s.onSubscribe(
                    new Subscription() {
                        private boolean done = false;

                        @Override
                        public void request(long n) {
                            if (done) {
                                return;
                            }
                            if (n > 0) {
                                done = true;
                                s.onNext(ByteBuffer.wrap(bytes));
                                s.onComplete();
                            } else {
                                s.onError(new IllegalArgumentException("non-positive requests are not allowed!"));
                            }
                        }

                        @Override
                        public void cancel() {
                            synchronized (this) {
                                if (!done) {
                                    done = true;
                                }
                            }
                        }
                    }
            );
        } catch (Throwable ex) {
            new IllegalStateException(ex)
                    .printStackTrace();
        }
    }
}
