package com.clickhouse.client.data;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import com.clickhouse.client.ClickHouseChecker;
import com.clickhouse.client.ClickHouseValue;
import com.clickhouse.client.ClickHouseValues;

/**
 * Wraper class of double.
 */
public class ClickHouseDoubleValue implements ClickHouseValue {
    /**
     * Create a new instance representing null value.
     *
     * @return new instance representing null value
     */
    public static ClickHouseDoubleValue ofNull() {
        return ofNull(null);
    }

    /**
     * Update given value to null or create a new instance if {@code ref} is null.
     * 
     * @param ref object to update, could be null
     * @return same object as {@code ref} or a new instance if it's null
     */
    public static ClickHouseDoubleValue ofNull(ClickHouseValue ref) {
        return ref instanceof ClickHouseDoubleValue ? ((ClickHouseDoubleValue) ref).set(true, 0D)
                : new ClickHouseDoubleValue(true, 0D);
    }

    /**
     * Wrap the given value.
     *
     * @param value value
     * @return object representing the value
     */
    public static ClickHouseDoubleValue of(double value) {
        return of(null, value);
    }

    /**
     * Update value of the given object or create a new instance if {@code ref} is
     * null.
     *
     * @param ref   object to update, could be null
     * @param value value
     * @return same object as {@code ref} or a new instance if it's null
     */
    public static ClickHouseDoubleValue of(ClickHouseValue ref, double value) {
        return ref instanceof ClickHouseDoubleValue ? ((ClickHouseDoubleValue) ref).set(false, value)
                : new ClickHouseDoubleValue(false, value);
    }

    private boolean isNull;
    private double value;

    protected ClickHouseDoubleValue(boolean isNull, double value) {
        set(isNull, value);
    }

    protected ClickHouseDoubleValue set(boolean isNull, double value) {
        this.isNull = isNull;
        this.value = isNull ? 0L : value;

        return this;
    }

    /**
     * Gets value.
     *
     * @return value
     */
    public double getValue() {
        return value;
    }

    @Override
    public ClickHouseDoubleValue copy(boolean deep) {
        return new ClickHouseDoubleValue(isNull, value);
    }

    @Override
    public boolean isNullOrEmpty() {
        return isNull;
    }

    @Override
    public byte asByte() {
        return (byte) value;
    }

    @Override
    public short asShort() {
        return (short) value;
    }

    @Override
    public int asInteger() {
        return (int) value;
    }

    @Override
    public long asLong() {
        return (long) value;
    }

    @Override
    public BigInteger asBigInteger() {
        return isNull ? null : BigDecimal.valueOf(value).toBigInteger();
    }

    @Override
    public float asFloat() {
        return (float) value;
    }

    @Override
    public double asDouble() {
        return value;
    }

    @Override
    public BigDecimal asBigDecimal(int scale) {
        if (isNull) {
            return null;
        }

        BigDecimal dec = BigDecimal.valueOf(value);
        if (value == 0D || isInfinity() || isNaN()) {
            dec = dec.setScale(scale);
        } else {
            int diff = scale - dec.scale();
            if (diff > 0) {
                dec = dec.divide(BigDecimal.TEN.pow(diff + 1));
            } else if (diff < 0) {
                dec = dec.setScale(scale);
            }
        }
        return dec;
    }

    @Override
    public Object asObject() {
        return isNull ? null : getValue();
    }

    @Override
    public String asString(int length, Charset charset) {
        if (isNull) {
            return null;
        }

        String str = String.valueOf(value);
        if (length > 0) {
            ClickHouseChecker.notWithDifferentLength(str.getBytes(charset == null ? StandardCharsets.UTF_8 : charset),
                    length);
        }

        return str;
    }

    @Override
    public ClickHouseDoubleValue resetToNullOrEmpty() {
        return set(true, 0D);
    }

    @Override
    public String toSqlExpression() {
        if (isNullOrEmpty()) {
            return ClickHouseValues.NULL_EXPR;
        } else if (isNaN()) {
            return ClickHouseValues.NAN_EXPR;
        } else if (value == Float.POSITIVE_INFINITY) {
            return ClickHouseValues.INF_EXPR;
        } else if (value == Float.NEGATIVE_INFINITY) {
            return ClickHouseValues.NINF_EXPR;
        }

        return String.valueOf(value);
    }

    @Override
    public ClickHouseDoubleValue update(boolean value) {
        return set(false, value ? 1 : 0);
    }

    @Override
    public ClickHouseDoubleValue update(char value) {
        return set(false, value);
    }

    @Override
    public ClickHouseDoubleValue update(byte value) {
        return set(false, value);
    }

    @Override
    public ClickHouseDoubleValue update(short value) {
        return set(false, value);
    }

    @Override
    public ClickHouseDoubleValue update(int value) {
        return set(false, value);
    }

    @Override
    public ClickHouseDoubleValue update(long value) {
        return set(false, value);
    }

    @Override
    public ClickHouseDoubleValue update(float value) {
        return set(false, value);
    }

    @Override
    public ClickHouseDoubleValue update(double value) {
        return set(false, value);
    }

    @Override
    public ClickHouseDoubleValue update(BigInteger value) {
        return value == null ? resetToNullOrEmpty() : set(false, value.doubleValue());
    }

    @Override
    public ClickHouseDoubleValue update(BigDecimal value) {
        return value == null ? resetToNullOrEmpty() : set(false, value.doubleValue());
    }

    @Override
    public ClickHouseDoubleValue update(Enum<?> value) {
        return value == null ? resetToNullOrEmpty() : set(false, value.ordinal());
    }

    @Override
    public ClickHouseDoubleValue update(String value) {
        return value == null ? resetToNullOrEmpty() : set(false, Double.parseDouble(value));
    }

    @Override
    public ClickHouseDoubleValue update(ClickHouseValue value) {
        return value == null ? resetToNullOrEmpty() : set(false, value.asDouble());
    }

    @Override
    public ClickHouseDoubleValue update(Object value) {
        if (value instanceof Number) {
            return set(false, ((Number) value).doubleValue());
        } else if (value instanceof ClickHouseValue) {
            return set(false, ((ClickHouseValue) value).asDouble());
        }

        ClickHouseValue.super.update(value);
        return this;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) { // too bad this is a mutable class :<
            return true;
        } else if (obj == null || getClass() != obj.getClass()) {
            return false;
        }

        ClickHouseDoubleValue v = (ClickHouseDoubleValue) obj;
        return isNull == v.isNull && value == v.value;
    }

    @Override
    public int hashCode() {
        // not going to use Objects.hash(isNull, value) due to autoboxing
        long l = Double.doubleToLongBits(value);
        return (31 + (isNull ? 1231 : 1237)) * 31 + (int) (l ^ (l >>> 32));
    }

    @Override
    public String toString() {
        return ClickHouseValues.convertToString(this);
    }
}
