package com.els.base.auth.service.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.auth.dao.MenuMapper;
import com.els.base.auth.entity.Menu;
import com.els.base.auth.entity.MenuExample;
import com.els.base.auth.entity.OperatorExample;
import com.els.base.auth.entity.RoleRight;
import com.els.base.auth.entity.RoleRightExample;
import com.els.base.auth.service.ButtonService;
import com.els.base.auth.service.IMenuTipGenerator;
import com.els.base.auth.service.MenuService;
import com.els.base.auth.service.OperatorService;
import com.els.base.auth.service.RoleRightService;
import com.els.base.auth.utils.AuthConstant;
import com.els.base.auth.utils.MenuComparator;
import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.utils.SpringContextHolder;

@Service
public class MenuServiceImpl implements MenuService {

	private static final int MAX_CALLCOUNT = 5;

	@Resource
	private MenuMapper menuMapper;

	@Resource
	private RoleRightService roleRightService;

	@Resource
	private OperatorService operatorService;

	@Resource
	private ButtonService buttonService;

	// 添加菜单
	@Transactional
	@CacheEvict(value = "menu", allEntries = true)
	@Override
	public void addObj(Menu menu) {
		Assert.isNotNull(menu, "菜单数据不能为空");
		Assert.isNotBlank(menu.getMenuCode(), "菜单编码不能为空");
		Assert.isNotBlank(menu.getMenuName(), "菜单名称不能为空");
		Assert.isNotBlank(menu.getMenuType(), "菜单类型不能为空");
		
		if (menu.getMenuCode().length() > 30) {
			throw new CommonException("菜单栏目代码长度不能超过30", "length_canot_exceed", "菜单栏目代码", "30");
		}
		if (menu.getMenuName().length() > 100) {
			throw new CommonException("名称长度不能超过100", "length_canot_exceed", "名称", "100");
		}
		MenuExample menuExample = new MenuExample();
		menuExample.createCriteria().andMenuCodeEqualTo(menu.getMenuCode());
		if (this.menuMapper.countByExample(menuExample) > 0) {
			throw new CommonException("菜单code值已存在", "base_is_exists", "code值");
		}

		menuMapper.insertSelective(menu);
	}

	@CacheEvict(value = "menu", allEntries = true)
	@Override
	public void deleteObjById(String menuId) {
		// 递归删除子节点
		deleteChildrenMenu(menuId, 0);
	}

	@Transactional
	@CacheEvict(value = "menu", allEntries = true)
	@Override
	public void modifyObj(Menu menu) {
		Assert.isNotBlank(menu.getId(), "菜单id不能为空");

		if (menu.getMenuCode().length() > 30) {
			throw new CommonException("菜单栏目代码长度不能超过30", "length_canot_exceed", "菜单栏目代码", "30");
		}
		if (menu.getMenuName().length() > 100) {
			throw new CommonException("名称长度不能超过100", "length_canot_exceed", "名称", "100");
		}

		if (StringUtils.isNotBlank(menu.getMenuCode())) {

			MenuExample menuExample = new MenuExample();
			menuExample.createCriteria()
				.andMenuCodeEqualTo(menu.getMenuCode())
				.andIdNotEqualTo(menu.getId());
			
			if (this.menuMapper.countByExample(menuExample) > 0) {
				throw new CommonException("菜单code值已存在", "base_is_exists", "code值");
			}
		}

		menu.setMenuType(null);
		menuMapper.updateByPrimaryKeySelective(menu);

	}

	@Cacheable(value = "menu", keyGenerator = "redisKeyGenerator")
	@Override
	public Menu queryObjById(String id) {
		return this.menuMapper.selectByPrimaryKey(id);
	}

	@Cacheable(value = "menu", keyGenerator = "redisKeyGenerator")
	@Override
	public List<Menu> queryAllObjByExample(MenuExample example) {
		return this.menuMapper.selectByExample(example);
	}

	@Cacheable(value = "menu", keyGenerator = "redisKeyGenerator")
	@Override
	public PageView<Menu> queryObjByPage(MenuExample example) {
		PageView<Menu> pageView = example.getPageView();
		List<Menu> list = this.menuMapper.selectByExampleByPage(example);
		pageView.setQueryResult(list);
		return pageView;
	}

	/** 
	 * 递归调用删除所有的子菜单
	 * @param menuId
	 * @param callCount
	 */
	private void deleteChildrenMenu(String menuId, int callCount) {
		callCount++;
		// 判断递归深度
		if (callCount > MAX_CALLCOUNT) {
			return;
		}
		// List<Menu> menuChildren = menuMapper.queryMenuByParentId(menuId);
		MenuExample menuExample = new MenuExample();
		menuExample.createCriteria().andParentIdEqualTo(menuId);
		menuExample.setOrderByClause("sort_no ASC");

		List<Menu> menuChildren = this.menuMapper.selectByExample(menuExample);

		for (Menu subMenu : menuChildren) {
			deleteChildrenMenu(subMenu.getId(), callCount);
		}

		this.menuMapper.deleteByPrimaryKey(menuId);

		OperatorExample operatorExample = new OperatorExample();
		operatorExample.createCriteria().andMenuIdEqualTo(menuId);
		this.operatorService.deleteByExample(operatorExample);
		
		this.buttonService.deleteObjByMenuId(menuId);
	}

	@Cacheable(value = "menu", keyGenerator = "redisKeyGenerator")
	@Override
	public List<Menu> queryAllMenuWithAuthRoleIds(List<String> currentRoleIds, List<String> targetRoleIds, String menuType) {
		if (CollectionUtils.isEmpty(currentRoleIds)) {
			throw new CommonException("当前的角色为空");
		}
		
		if (StringUtils.isBlank(menuType)) {
			// 默认查普通菜单啊
			throw new CommonException("菜单类型不能为空");
		}
		
		MenuExample menuExample = new MenuExample();
		MenuExample.Criteria criteria = menuExample.createCriteria();
		criteria.andMenuTypeEqualTo(menuType);
		if (!currentRoleIds.contains(AuthConstant.SYS_MANAGER_ROLE.getId())) {
			List<String> currentMenuIds = this.queryMenuIdByRoleId(currentRoleIds, menuType);
			if(CollectionUtils.isEmpty(currentMenuIds)) return null;
			criteria.andIdIn(currentMenuIds);
		}
		
		List<Menu> currentMenus = this.queryAllObjByExample(menuExample);
		if (CollectionUtils.isEmpty(currentMenus)) {
			return null;
		}
		
		if (!targetRoleIds.contains(AuthConstant.SYS_MANAGER_ROLE.getId())
				&& !currentRoleIds.equals(targetRoleIds)) {
			
			List<String> targetMenuId = this.queryMenuIdByRoleId(targetRoleIds, menuType);
			for(Menu menu : currentMenus){
				menu.setAuthorized(targetMenuId.contains(menu.getId()));
			}
		}
		
		List<Menu> allMenu = new ArrayList<>();
		for(Menu menu : currentMenus){
			if (!Menu.ROOT_ID.equals(menu.getParentId())) {
				continue;
			}
			
			allMenu.add(menu);
			
			List<Menu> children = this.findChildren(menu, currentMenus);
			menu.setChildren(children);
		}
		Collections.sort(allMenu, new MenuComparator());
		
		this.buttonService.setAuthDataInMenu(currentMenus, currentRoleIds, targetRoleIds);
		this.operatorService.setAuthDataInMenu(currentMenus, currentRoleIds, targetRoleIds);
		return allMenu;
		
	}

	private List<Menu> findChildren(Menu parent, List<Menu> currentMenus) {
		
		List<Menu> allMenu = new ArrayList<>();
		
		for(Menu child : currentMenus){
			if (!parent.getId().equals(child.getParentId())) {
				continue;
			}
			allMenu.add(child);
			
			List<Menu> children = this.findChildren(child, currentMenus);
			child.setChildren(children);
		}
		Collections.sort(allMenu, new MenuComparator());
		return allMenu;
	}

	@Cacheable(value = "menu", keyGenerator = "redisKeyGenerator")
	@Override
	public List<Menu> queryAuthedMenuForRoleIds(List<String> roleIds, String menuType) {
		return this.queryAllMenuWithAuthRoleIds(roleIds, roleIds, menuType);
	}
	
	private List<String> queryMenuIdByRoleId(List<String> roleIds, String menuType) {
		
        RoleRightExample roleRightExample = new RoleRightExample();
        roleRightExample.createCriteria()
        	.andRoleIdIn(roleIds)
        	.andTypeEqualTo(menuType);

        List<RoleRight> roleRightList = this.roleRightService.queryAllObjByExample(roleRightExample);
        List<String> authMenuIds = new ArrayList<>();

        for (RoleRight roleRight : roleRightList) {
            authMenuIds.add(roleRight.getKeyId());
        }

        return authMenuIds;
    }
	
	@CacheEvict(value = "menu", allEntries = true)
	@Override
	public void deleteByExample(MenuExample example) {
		Assert.isNotNull(example, "参数不能为空");
    	Assert.isNotEmpty(example.getOredCriteria(), "批量删除不能全表删除");
    	
		this.menuMapper.deleteByExample(example);
	}

	@CacheEvict(value = "menu", allEntries = true)
	@Transactional
	@Override
	public void addAll(List<Menu> list) {
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		
		list.forEach(record->{
			this.menuMapper.insertSelective(record);
		});
	}

	@Override
	public List<Menu> queryMenuTip(List<String> roleIdList, String menuType) {
		if (CollectionUtils.isEmpty(roleIdList)) {
			return null;
		}
		
		Map<String, IMenuTipGenerator> beans = SpringContextHolder.getBeans(IMenuTipGenerator.class);
		if (MapUtils.isEmpty(beans)) {
			return null;
		}
		List<IMenuTipGenerator> tipGenerators = beans.entrySet().stream()
				.map(Entry::getValue).collect(Collectors.toList());
		
		MenuExample menuExample = new MenuExample();
		MenuExample.Criteria criteria = menuExample.createCriteria();
		criteria.andMenuTypeEqualTo(menuType);
		if (!roleIdList.contains(AuthConstant.SYS_MANAGER_ROLE.getId())) {
			List<String> currentMenuIds = this.queryMenuIdByRoleId(roleIdList, menuType);
			if(CollectionUtils.isEmpty(currentMenuIds)) return null;
			criteria.andIdIn(currentMenuIds);
		}
		
		List<Menu> menuList = this.queryAllObjByExample(menuExample);
		if (CollectionUtils.isEmpty(menuList)) {
			return null;
		}
		
		tipGenerators.stream()
			.sorted(Comparator.comparing(IMenuTipGenerator::getPriority).reversed())
			.forEach(tipGenerator->{
				menuList.stream()
					.filter(menu-> tipGenerator.isMatchMenuCode(menu.getMenuCode()))
					.forEach(menu-> menu.setTip(tipGenerator.generate(menu)));
			});
		
		menuList.stream().filter(menu->StringUtils.isNotBlank(menu.getTip()));
		
		return menuList;
	}

}
