package com.els.base.auth.service.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.auth.dao.RoleMapper;
import com.els.base.auth.dao.RoleRightMapper;
import com.els.base.auth.entity.AuthorizationData;
import com.els.base.auth.entity.Button;
import com.els.base.auth.entity.ButtonExample;
import com.els.base.auth.entity.Operator;
import com.els.base.auth.entity.OperatorExample;
import com.els.base.auth.entity.Role;
import com.els.base.auth.entity.RoleExample;
import com.els.base.auth.entity.RoleRight;
import com.els.base.auth.entity.RoleRightExample;
import com.els.base.auth.service.ButtonService;
import com.els.base.auth.service.MenuService;
import com.els.base.auth.service.OperatorService;
import com.els.base.auth.service.RoleRightService;
import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;

/**
 * Created by LiBing on 2014-07-22 17:45
 */
@Service
public class RoleRightServiceImpl implements RoleRightService {

	@Resource
	protected MenuService menuService;
	
	@Resource
	protected ButtonService buttonService;

    @Resource
    private RoleRightMapper roleRightMapper;

    @Resource
    private RoleMapper roleMapper;

    @Resource
    private OperatorService operatorService;

    @CacheEvict(value = {"roleRightCache", "menu" }, allEntries = true)
    @Override
    public void addObj(RoleRight roleRight) {
        roleRightMapper.insertSelective(roleRight);
    }

    @CacheEvict(value = {"roleRightCache", "menu" }, allEntries = true)
    @Override
    public void deleteObjById(String id) {
        this.roleRightMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value = {"roleRightCache", "menu" }, allEntries = true)
    @Override
    public void modifyObj(RoleRight t) {
        if (StringUtils.isBlank(t.getId()) || StringUtils.isBlank(t.getKeyId())
                || StringUtils.isBlank(t.getRoleId()) || StringUtils.isBlank(t.getType())) {
            throw new IllegalArgumentException("参数不全，没法修改");
        }
        this.roleRightMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value = "roleRightCache", keyGenerator="redisKeyGenerator")
    @Override
    public RoleRight queryObjById(String id) {
        return this.roleRightMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value = "roleRightCache", keyGenerator="redisKeyGenerator")
    @Override
    public List<RoleRight> queryAllObjByExample(RoleRightExample example) {
        return this.roleRightMapper.selectByExample(example);
    }

    @Cacheable(value = "roleRightCache", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<RoleRight> queryObjByPage(RoleRightExample example) {
        PageView<RoleRight> pageView = example.getPageView();
        pageView.setQueryResult(this.roleRightMapper.selectByExampleByPage(example));
        return pageView;
    }

    @Cacheable(value = "roleRightCache", keyGenerator="redisKeyGenerator")
    @Override
    public List<Role> getRoleByOperatorUrl(String url) {
        OperatorExample operatorExample = new OperatorExample();
        operatorExample.createCriteria().andOperUrlEqualTo(url);
        if (!url.contains(".html")) {
            operatorExample.or().andOperUrlEqualTo(url + ".html");
        }

        List<Operator> operList = this.operatorService.queryAllObjByExample(operatorExample);
        List<String> operIdList = new ArrayList<>();
        for (int i = 0; i < operList.size(); i++) {
            operIdList.add(operList.get(i).getId().toString());
        }

        if (CollectionUtils.isEmpty(operIdList)) {
            return null;
        }

        RoleRightExample roleRightExample = new RoleRightExample();
        roleRightExample.createCriteria().andKeyIdIn(operIdList)
        	.andTypeEqualTo(AuthorizationData.TYPE_OPERATOR);

        List<RoleRight> roleRightList = this.roleRightMapper.selectByExample(roleRightExample);

        List<String> roleIdList = new ArrayList<>();
        for (int i = 0; i < roleRightList.size(); i++) {
            roleIdList.add(roleRightList.get(i).getRoleId());
        }

        if (CollectionUtils.isEmpty(roleIdList)) {
            return null;
        }

        RoleExample roleExample = new RoleExample();
        roleExample.createCriteria().andIdIn(roleIdList);
        List<Role> roleList = this.roleMapper.selectByExample(roleExample);
        return roleList;
    }
    
    @CacheEvict(value = {"roleRightCache", "menu" }, allEntries = true)
	@Override
	public void deleteByExample(RoleRightExample example) {
    	Assert.isNotNull(example, "参数不能为空");
    	Assert.isNotEmpty(example.getOredCriteria(), "批量删除不能全表删除");
    	
		this.roleRightMapper.deleteByExample(example);
	}

    @CacheEvict(value = {"roleRightCache", "menu" }, allEntries = true)
    @Transactional
	@Override
	public void addAll(List<RoleRight> list) {
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		
		list.forEach(record->{
			this.roleRightMapper.insertSelective(record);
		});
	}

    @CacheEvict(value = {"roleRightCache", "menu" }, allEntries = true)
    @Transactional
    @Override
    public void authorized(String roleId, List<RoleRight> rights) {
        this.authorized(roleId, rights, AuthorizationData.TYPE_MENU);
    }

    @CacheEvict(value = {"roleRightCache", "menu" }, allEntries = true)
	@Override
	public void authorized(String roleId, List<RoleRight> rights, String menuType) {
    	
    	if (StringUtils.isBlank(menuType)) {
			throw new CommonException("菜单类型不能为空");
		}
    	
        for (RoleRight roleRight : rights) {
            if (StringUtils.isBlank(roleRight.getType())) {
                throw new CommonException("参数[type]不能为空", "base_canot_be_null","参数[type]");
            }
            if (StringUtils.isBlank(roleRight.getKeyId())) {
                throw new CommonException("参数[keyId]不能为空", "base_canot_be_null","参数[keyId]");
            }
            roleRight.setRoleId(roleId);
        }
        
        //删除该菜单下的所有权限
    	this.deleteRight(roleId, rights, menuType);

        for (RoleRight roleRight : rights) {
            this.roleRightMapper.insertSelective(roleRight);
        }
	}

    /**
     * 删除改菜单类型下面的所有权限
     * @param roleId
     * @param rights
     * @param menuType
     */
	private void deleteRight(String roleId, List<RoleRight> rights, String menuType) {
		if (CollectionUtils.isNotEmpty(rights)) {
			
			boolean isMatch = rights.stream()
					.anyMatch(right-> right.getType().equals(menuType));
				
			if (!isMatch) {
				throw new CommonException("菜单类型不一致");
			}
		}
		
		RoleRightExample example = new RoleRightExample();
        example.createCriteria()
        	.andRoleIdEqualTo(roleId)
        	.andTypeEqualTo(menuType);
        
        List<RoleRight> roleRightList = this.roleRightMapper.selectByExample(example);
        if (CollectionUtils.isEmpty(roleRightList)) {
			return;
		}
        
        List<String> roleRightIds = new ArrayList<>();
        
        List<String> menuIds = roleRightList.stream()
        		.map(RoleRight::getKeyId).collect(Collectors.toList());
        roleRightIds.addAll(menuIds);
		
		ButtonExample buttonExample = new ButtonExample();
		buttonExample.createCriteria().andMenuIdIn(menuIds);
		
		List<Button> buttonList = buttonService.queryAllObjByExample(buttonExample);
		if (CollectionUtils.isNotEmpty(buttonList)) {
			buttonList.forEach(button-> roleRightIds.add(button.getId()));
		}
		
		OperatorExample operatorExample = new OperatorExample();
		operatorExample.createCriteria().andMenuIdIn(menuIds);
		
		List<Operator> operatorList = this.operatorService.queryAllObjByExample(operatorExample);
		if (CollectionUtils.isNotEmpty(operatorList)) {
			operatorList.forEach(operator-> roleRightIds.add(operator.getId()));
		}
		
		RoleRightExample roleRightExample = new RoleRightExample();
		roleRightExample.createCriteria()
			.andKeyIdIn(roleRightIds)
			.andRoleIdEqualTo(roleId);
		
		this.roleRightMapper.deleteByExample(roleRightExample);
	}
}
