package com.els.base.auth.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.auth.dao.RoleMapper;
import com.els.base.auth.dao.UserGroupRoleRefMapper;
import com.els.base.auth.entity.AuthorizationData;
import com.els.base.auth.entity.Operator;
import com.els.base.auth.entity.Role;
import com.els.base.auth.entity.RoleExample;
import com.els.base.auth.entity.RoleRight;
import com.els.base.auth.entity.RoleRightExample;
import com.els.base.auth.entity.UserGroup;
import com.els.base.auth.entity.UserGroupRoleRef;
import com.els.base.auth.entity.UserGroupRoleRefExample;
import com.els.base.auth.entity.UserRole;
import com.els.base.auth.entity.UserRoleExample;
import com.els.base.auth.service.OperatorService;
import com.els.base.auth.service.RoleRightService;
import com.els.base.auth.service.RoleService;
import com.els.base.auth.service.UserGroupService;
import com.els.base.auth.service.UserRoleService;
import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.utils.SpringContextHolder;

/**
 * Created by LiBing on 2014/6/5.
 */
@Service("roleService")
public class RoleServiceImpl implements RoleService {
    @Autowired
    private RoleMapper roleMapper;

    @Resource
    private UserGroupRoleRefMapper userGroupRoleRefMapper;

    @Resource
    private RoleRightService roleRightService;

    @Resource
    private OperatorService operatorService;

    @Resource
    private UserRoleService userRoleService;

    @Resource
    private UserGroupService userGroupService;

    @CacheEvict(value = { "roleCache", "roleRightCache" }, allEntries = true)
    @Override
    public void addObj(Role role) {
    	if (StringUtils.isBlank(role.getRoleCode())) {
    		throw new CommonException("code值不能为空", "base_canot_be_null","code值");
		}
    	if (StringUtils.isBlank(role.getCompanyId())) {
			throw new CommonException("companyId不能为空");
		}
    	
    	RoleExample example = new RoleExample();
		example.createCriteria()
		    .andRoleCodeEqualTo(role.getRoleCode())
		    .andCompanyIdEqualTo(role.getCompanyId());
		
		example.or()
		    .andRoleCodeEqualTo(role.getRoleCode())
		    .andCompanyIdIsNull();
		
		if (this.roleMapper.countByExample(example) >0 ) {
			throw new CommonException("code值已存在", "base_is_exists","code值");
		}
    	if(role.getRoleName().length()>50){
    	    throw new CommonException("角色名长度不能超过50", "length_canot_exceed", "角色名","50");
    	}
    	if(role.getRoleCode().length()>20){
            throw new CommonException("角色编码长度不能超过20", "length_canot_exceed", "角色编码","20");
        }
    	
    	if (role.getRoleLevel() == null || role.getRoleLevel() <0) {
			role.setRoleLevel(Role.ROLE_LEVEL_CUSTOM);
		}
    	
    	if (!Role.ROLE_LEVEL_SYS.equals(role.getRoleLevel())
    			&& !Role.ROLE_LEVEL_CUSTOM.equals(role.getRoleLevel())) {
			throw new CommonException("非法的角色级别");
		}
		
        Date date = new Date();
        role.setCreateTime(date);
        role.setUpdateTime(date);
        roleMapper.insertSelective(role);
    }

    @CacheEvict(value = { "roleCache", "roleRightCache" }, allEntries = true)
    @Override
    public void deleteObjById(String roleId) {
        if (roleId.equals("1")) {
            throw new CommonException("管理员,不允许操作", "admin_not_allow_oper");
        }

        roleMapper.deleteByPrimaryKey(roleId);
    }

    @CacheEvict(value = { "roleCache", "roleRightCache" }, allEntries = true)
    @Override
    public void modifyObj(Role role) {
    	if(StringUtils.isBlank(role.getId())){
			throw new CommonException("id 不能为空", "id_is_blank");
		}
    	
    	if(role.getRoleName().length()>50){
            throw new CommonException("角色名长度不能超过50", "length_canot_exceed", "角色名","50");
        }
        if(role.getRoleCode().length()>20){
            throw new CommonException("角色编码长度不能超过20", "length_canot_exceed", "角色编码","20");
        }
    	
    	RoleExample example = new RoleExample();
        
        if (StringUtils.isNotBlank(role.getRoleCode())) {
        	example.createCriteria()
        	    .andRoleCodeEqualTo(role.getRoleCode())
			    .andIdNotEqualTo(role.getId())
			    .andCompanyIdIsNull();
        	
        	if (StringUtils.isNotBlank(role.getCompanyId())) {
        		example.or()
        		.andRoleCodeEqualTo(role.getRoleCode())
        		.andIdNotEqualTo(role.getId())
        		.andCompanyIdEqualTo(role.getCompanyId());
    		}
			
			if (this.roleMapper.countByExample(example) >0 ) {
				throw new CommonException("code值已存在", "base_is_exists","code值");
			}
		}
        
        Date date = new Date();
        role.setUpdateTime(date);
        roleMapper.updateByPrimaryKeySelective(role);
    }

    @Cacheable(value = "roleCache", keyGenerator="redisKeyGenerator")
    @Override
    public Map<String, Set<String>> queryUserOperatorAction(boolean isAdmin, List<String> roleIds) {
        Map<String, Set<String>> rightMap = new HashMap<String, Set<String>>();
        // 是否具有管理员权限
        if (isAdmin) {
            rightMap.put("menuAll", new HashSet<String>());
        } else {

            RoleRightExample roleRightExample = new RoleRightExample();
            roleRightExample.createCriteria().andRoleIdIn(roleIds)
                    .andTypeEqualTo(AuthorizationData.TYPE_OPERATOR);

            List<RoleRight> rights = this.roleRightService.queryAllObjByExample(roleRightExample);
            for (RoleRight right : rights) {
                Operator operator = this.operatorService.queryObjById(right.getKeyId());

                String menuCode = operator.getMenuCode();
                String operCode = operator.getOperCode();
                Set<String> operSet = null;
                if (rightMap.containsKey(menuCode)) {
                    operSet = rightMap.get(menuCode);
                } else {
                    operSet = new HashSet<String>();
                }
                operSet.add(operCode);
                rightMap.put(menuCode, operSet);
            }
        }
        return rightMap;
    }

    @Cacheable(value = "roleCache", keyGenerator="redisKeyGenerator")
    @Override
    public List<Role> queryUserOwnAndCreateRoles(boolean isAdmin, String userId) {
        RoleExample roleExample = new RoleExample();

        // 需要根据用户身份取出对应的角色
        if (isAdmin) {
            return this.roleMapper.selectByExample(roleExample);
        } else {
            UserRoleExample userRoleExample = new UserRoleExample();
            userRoleExample.createCriteria().andUserIdEqualTo(userId);

            List<UserRole> list = userRoleService.queryAllObjByExample(userRoleExample);
            List<String> roleIdList = new ArrayList<>();
            for (UserRole userRole : list) {
                roleIdList.add(userRole.getRoleId());
            }

            roleExample.createCriteria().andCreateUserIdEqualTo(userId);
            roleExample.or().andIdIn(roleIdList);

            return this.queryAllObjByExample(roleExample);
        }
    }

    @Cacheable(value = "roleCache", keyGenerator="redisKeyGenerator")
    @Override
    public Role queryRoleById(String roleId) {
        return roleMapper.selectByPrimaryKey(roleId);
    }

    @Cacheable(value = "roleCache", keyGenerator="redisKeyGenerator")
    @Override
    public List<Role> queryUserOwnRoles(String userId) {
        UserRoleExample userRoleExample = new UserRoleExample();
        userRoleExample.createCriteria().andUserIdEqualTo(userId);

        List<UserRole> list = this.userRoleService.queryAllObjByExample(userRoleExample);
        if (CollectionUtils.isEmpty(list)) {
            return null;
        }

        List<String> roleIdList = new ArrayList<>();
        for (UserRole role : list) {
            roleIdList.add(role.getRoleId());
        }

        RoleExample roleExample = new RoleExample();
        roleExample.createCriteria().andIdIn(roleIdList);
        return this.queryAllObjByExample(roleExample);
    }

    @Cacheable(value = "roleCache", keyGenerator="redisKeyGenerator")
    @Override
    public Role queryObjById(String id) {
        return this.roleMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value = "roleCache", keyGenerator="redisKeyGenerator")
    @Override
    public List<Role> queryAllObjByExample(RoleExample example) {
        return this.roleMapper.selectByExample(example);
    }

    @Cacheable(value = "roleCache", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<Role> queryObjByPage(RoleExample example) {
        PageView<Role> pageView = example.getPageView();
        pageView.setQueryResult(this.roleMapper.selectByExampleByPage(example));
        return pageView;
    }

    @Override
    public List<Role> queryAllRoleByUserGroup(String userGroupId) {
        // 1、查询组织已经组织的上级
        List<String> userGroupIds = new ArrayList<>();

        UserGroup userGroup = SpringContextHolder.getOneBean(UserGroupService.class).queryObjById(userGroupId);
        if (userGroup == null) {
            return null;
        }
        userGroupIds.add(userGroup.getId());

        UserGroup parents = SpringContextHolder.getOneBean(UserGroupService.class).queryParents(userGroupId);
        while (parents != null) {
            userGroupIds.add(parents.getId());
            parents = parents.getParent();
        }

        // 根据组织或组织上级，查出关联的roleid
        UserGroupRoleRefExample example = new UserGroupRoleRefExample();
        example.createCriteria().andOganizationIdIn(userGroupIds);

        List<UserGroupRoleRef> refList = this.userGroupRoleRefMapper.selectByExample(example);
        if (CollectionUtils.isEmpty(refList)) {
            return null;
        }
        List<String> roleIds = new ArrayList<>();
        for (UserGroupRoleRef ref : refList) {
            roleIds.add(ref.getRoleId());
        }

        // 根据roleid查出role
        RoleExample roleExample = new RoleExample();
        roleExample.createCriteria().andIdIn(roleIds);
        return this.queryAllObjByExample(roleExample);
    }

    @CacheEvict(value = { "roleCache", "roleRightCache" }, allEntries = true)
	@Override
	public void deleteByExample(RoleExample example) {
		Assert.isNotNull(example, "参数不能为空");
    	Assert.isNotEmpty(example.getOredCriteria(), "批量删除不能全表删除");
    	
    	this.roleMapper.deleteByExample(example);
	}

    @CacheEvict(value = { "roleCache", "roleRightCache" }, allEntries = true)
    @Transactional
	@Override
	public void addAll(List<Role> list) {
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		
		list.forEach(record ->{
			this.roleMapper.insertSelective(record);
		});
	}

    @CacheEvict(value = { "roleCache", "roleRightCache" }, allEntries = true)
	@Override
	public void modifyByExample(Role role, RoleExample example) {
    	Assert.isNotNull(example, "参数不能为空");
    	Assert.isNotEmpty(example.getOredCriteria(), "批量删除不能全表删除");
    	
    	this.roleMapper.updateByExampleSelective(role, example);
	}
}
