package com.els.base.auth.service.impl;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.auth.dao.UserGroupMapper;
import com.els.base.auth.dao.UserGroupUserRefMapper;
import com.els.base.auth.entity.UserGroup;
import com.els.base.auth.entity.UserGroupExample;
import com.els.base.auth.entity.UserGroupUserRef;
import com.els.base.auth.entity.UserGroupUserRefExample;
import com.els.base.auth.service.UserGroupService;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.user.User;
import com.els.base.core.entity.user.UserExample;
import com.els.base.core.service.user.UserService;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.Constant;

@Service("userGroupService")
public class UserGroupServiceImpl implements UserGroupService {
    @Resource
    protected UserGroupMapper userGroupMapper;
    
    @Resource
    protected UserGroupUserRefMapper userGroupUserRefMapper;
    
    @Resource
    protected UserService userService;

    @CacheEvict(value={"userGroupCache"}, allEntries=true)
    @Override
    public void addObj(UserGroup t) {
        this.userGroupMapper.insertSelective(t);
    }

    @CacheEvict(value={"userGroupCache"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.userGroupMapper.deleteByPrimaryKey(id);
    }

    @CacheEvict(value={"userGroupCache"}, allEntries=true)
    @Override
    public void modifyObj(UserGroup t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        this.userGroupMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value="userGroupCache", keyGenerator="redisKeyGenerator")
    @Override
    public UserGroup queryObjById(String id) {
        return this.userGroupMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value="userGroupCache", keyGenerator="redisKeyGenerator")
    @Override
    public List<UserGroup> queryAllObjByExample(UserGroupExample example) {
        return this.userGroupMapper.selectByExample(example);
    }

    @Cacheable(value="userGroupCache", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<UserGroup> queryObjByPage(UserGroupExample example) {
        PageView<UserGroup> pageView = example.getPageView();
        pageView.setQueryResult(this.userGroupMapper.selectByExampleByPage(example));
        return pageView;
    }
    
    @CacheEvict(value={"userGroupCache"}, allEntries=true)
	@Override
	public void addUserIntoUserGroup(String userId, String userGroupId) {
		UserGroupUserRef ref = new UserGroupUserRef();
		ref.setUserId(userId);
		ref.setUserGroupId(userGroupId);
		this.userGroupUserRefMapper.insertSelective(ref);
	}

    @Cacheable(value="userGroupCache", keyGenerator="redisKeyGenerator")
	@Override
	public UserGroup queryUserGroupsByUserId(String userId) {
		UserGroupUserRefExample userUserGroupRefExample = new UserGroupUserRefExample();
		userUserGroupRefExample.createCriteria().andUserIdEqualTo(userId);
		
		List<UserGroupUserRef> refList = this.userGroupUserRefMapper.selectByExample(userUserGroupRefExample);
		if (CollectionUtils.isEmpty(refList)) {
			return null;
		}
		
		return this.queryParents(refList.get(0).getUserGroupId());
	}

    @Cacheable(value="userGroupCache", keyGenerator="redisKeyGenerator")
	@Override
	public List<User> queryUserByUserGroupId(String userGroupId) {
		List<UserGroup> userGroups = this.queryAllChildren(userGroupId);
		
		List<String> userGroupIds = new ArrayList<>();
		for (UserGroup o: userGroups) {
			userGroupIds.add(o.getId());
		}
		
		UserGroupUserRefExample refExample = new UserGroupUserRefExample();
		refExample.createCriteria().andUserGroupIdIn(userGroupIds);
		
		List<UserGroupUserRef> refList = this.userGroupUserRefMapper.selectByExample(refExample);
		
		List<String> userIds = new ArrayList<>();
		for (UserGroupUserRef ref: refList) {
			userIds.add(ref.getUserId());
		}
		
		UserExample userExample = new UserExample();
		userExample.createCriteria().andIdIn(userIds).andIsEnableEqualTo(Constant.YES_INT);
		return this.userService.queryAllObjByExample(userExample);
	}

    @Cacheable(value="userGroupCache", keyGenerator="redisKeyGenerator")
	@Override
	public UserGroup queryParents(String userGroupId) {
		UserGroup current = this.userGroupMapper.selectByPrimaryKey(userGroupId);
		UserGroup parent = current.getParent();
		while (parent != null && parent.getParentId() != null && Constant.YES_INT.equals(parent.getIsEnable())) {
			UserGroup parentTmp = this.userGroupMapper.selectByPrimaryKey(parent.getId());
			parent.setParent(parentTmp);
			parent = parentTmp;
		}
		
		return current;
	}

    @Cacheable(value="userGroupCache", keyGenerator="redisKeyGenerator")
	@Override
	public List<UserGroup> queryAllChildren(String userGroupId) {
		UserGroupExample userGroupExample = new UserGroupExample();
		userGroupExample.createCriteria()
		                   .andParentIdEqualTo(userGroupId)
		                   .andIsEnableEqualTo(Constant.YES_INT);
		
		List<UserGroup> children = this.userGroupMapper.selectByExample(userGroupExample);
		if (CollectionUtils.isEmpty(children)) {
			return null;
		}
		
		List<UserGroup> tmpChildren = new ArrayList<>();
		for(UserGroup o: children){
			List<UserGroup> subChildren = this.queryAllChildren(o.getId());
			if (CollectionUtils.isEmpty(subChildren)) {
				continue;
			}
			CollectionUtils.addAll(tmpChildren, subChildren.iterator());
		}
		
		CollectionUtils.addAll(children, tmpChildren.iterator());
		return children;
	}

    @Cacheable(value="userGroupCache", keyGenerator="redisKeyGenerator")
	@Override
	public List<UserGroup> querySubChildren(String userGroupId) {
		UserGroupExample userGroupExample = new UserGroupExample();
		userGroupExample.createCriteria()
		                   .andParentIdEqualTo(userGroupId)
		                   .andIsEnableEqualTo(Constant.YES_INT);
		
		return this.userGroupMapper.selectByExample(userGroupExample);
	}

    @CacheEvict(value={"userGroupCache"}, allEntries=true)
	@Override
	public void deleteByExample(UserGroupExample example) {
		Assert.isNotNull(example, "参数不能为空");
    	Assert.isNotEmpty(example.getOredCriteria(), "批量删除不能全表删除");
    	
    	this.userGroupMapper.deleteByExample(example);		
	}

    @CacheEvict(value={"userGroupCache"}, allEntries=true)
    @Transactional
	@Override
	public void addAll(List<UserGroup> list) {
		if (CollectionUtils.isEmpty(list)) {
			return;
		}
		
		list.forEach(record ->{
			this.userGroupMapper.insertSelective(record);
		});
	}
}