package com.els.base.auth.web.controller;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.auth.entity.Role;
import com.els.base.auth.entity.RoleExample;
import com.els.base.auth.entity.UserRole;
import com.els.base.auth.entity.UserRoleExample;
import com.els.base.auth.service.RoleService;
import com.els.base.auth.service.UserRoleService;
import com.els.base.auth.utils.AuthConstant;
import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;

/**
 * Created by libing on 2014-06-02 08:57.
 */
@Api("权限管理-用户角色关联管理")
@Controller
@RequestMapping("userRole")
public class UserRoleController {
    @Autowired
    private UserRoleService rightService;
    
    @Resource
    protected RoleService roleService;

    @ApiOperation(value = "给用户增加角色", httpMethod = "POST")
    @RequestMapping("service/addUserRoleRight")
    @ResponseBody
    public ResponseResult<String> addUserRoleRight(String roleId, String userId) {
        UserRoleExample example = new UserRoleExample();
        example.createCriteria().andUserIdEqualTo(userId).andRoleIdEqualTo(roleId);

        if (CollectionUtils.isNotEmpty(this.rightService.queryAllObjByExample(example))) {
            return ResponseResult.success();
        }

        UserRole userRole = new UserRole();
        userRole.setRoleId(roleId);
        userRole.setUserId(userId);
        rightService.addObj(userRole);

        return ResponseResult.success();
    }

    @ApiOperation(value = "给用户删除角色", httpMethod = "POST")
    @RequestMapping("service/deleteUserRoleRight")
    @ResponseBody
    public ResponseResult<String> deleteUserRight(String rightId) {
        rightService.deleteObjById(rightId);

        return ResponseResult.success();
    }

    @ApiOperation(value = "修改用户的角色", httpMethod = "POST")
    @RequestMapping("/modifyUserRoleRight")
    @ResponseBody
    public ResponseResult<String> modifyUserRight(String rightId, String roleId, String userId) {
        UserRole userRole = new UserRole();
        userRole.setId(rightId);
        userRole.setRoleId(roleId);
        userRole.setUserId(userId);
        rightService.modifyObj(userRole);

        return ResponseResult.success();
    }

    @ApiOperation(value = "给用户添加多个角色", httpMethod = "POST")
    @RequestMapping("service/addUserRoleList")
    @ResponseBody
    public ResponseResult<String> addUserRoleList(String roleIds, String userId) {
        if (StringUtils.isBlank(roleIds)) {
            throw new CommonException("角色数据不能为空", "base_canot_be_null","角色数据");
        }
        
        List<String> allRole = new ArrayList<>();
        allRole.addAll(Arrays.asList(roleIds.split(",")));
        
        if (!this.isMangerUser()) {
        	// 如果不是系统管理，不能更改系统角色的分配，
        	// 把用户的系统角色找出来，并放到要分配的角色中，进行分配
        	List<String> sysRoleIds = this.querySysRoleOfUser(userId);
        	if (CollectionUtils.isNotEmpty(sysRoleIds)) {
        		allRole.addAll(sysRoleIds);
        	}
		}
        
        String[] roleIdArray = allRole.stream().distinct().toArray(String[]::new);
        
        this.rightService.resetUserRole(userId, roleIdArray);
        return ResponseResult.success();
    }

    /**
     * 检查当前用户是否系统管理员
     * @return
     */
    private boolean isMangerUser() {
    	List<Role> roleList = SpringSecurityUtils.getLoginUserRoleList();
    	if (CollectionUtils.isEmpty(roleList)) {
			return false;
		}
    	
    	for (Role role : roleList) {
    		if (AuthConstant.SYS_MANAGER_ROLE.getRoleCode().equals(role.getRoleCode())) {
				return true;
			}
		}
    	
    	return false;
	}

	/**
     * 找出该用户下所有的系统角色
     * @param userId
     * @return
     */
	private List<String> querySysRoleOfUser(String userId) {
		List<String> roleIdList = this.rightService.queryUserOwnRoleIds(userId);
		if (CollectionUtils.isEmpty(roleIdList)) {
			return null;
		}
		
		RoleExample roleExample = new RoleExample();
		roleExample.createCriteria().andIdIn(roleIdList)
			.andRoleLevelEqualTo(Role.ROLE_LEVEL_SYS);
		List<Role> roleList = this.roleService.queryAllObjByExample(roleExample);
		if (CollectionUtils.isEmpty(roleList)) {
			return null;
		}
		
		return roleList.stream().map(Role::getId).collect(Collectors.toList());
	}
}
