package com.els.base.auth.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

import com.els.base.auth.dao.OperatorMapper;
import com.els.base.auth.entity.AuthorizationData;
import com.els.base.auth.entity.Menu;
import com.els.base.auth.entity.Operator;
import com.els.base.auth.entity.OperatorExample;
import com.els.base.auth.entity.RoleRight;
import com.els.base.auth.entity.RoleRightExample;
import com.els.base.auth.service.MenuService;
import com.els.base.auth.service.OperatorService;
import com.els.base.auth.service.RoleRightService;
import com.els.base.auth.utils.AuthConstant;
import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;

@Service
public class OperatorServiceImpl implements OperatorService {

    @Resource
    private OperatorMapper operatorMapper;

    @Resource
    private RoleRightService roleRightService;
    
    @Resource
    private MenuService menuService;

    @CacheEvict(value = { "operatorCache" }, allEntries = true)
    @Override
    public void addObj(Operator operator) {
    	Assert.isNotBlank(operator.getMenuId(), "菜单id不能为空");
    	Assert.isNotBlank(operator.getOperCode(), "operatecode不能为空");
    	
        OperatorExample operatorExample = new OperatorExample();
        operatorExample.createCriteria().andOperCodeEqualTo(operator.getOperCode());

        if (this.operatorMapper.countByExample(operatorExample) > 0) {
            throw new CommonException("操作code值已存在", "base_is_exists","code值");
        }

        if (StringUtils.isBlank(operator.getMenuCode())) {
        	Menu menu = this.menuService.queryObjById(operator.getMenuId());
			operator.setMenuCode(menu.getMenuCode());
			operator.setMenuName(menu.getMenuName());
		}
        
        operator.setCreateTime(new Date());
        operatorMapper.insertSelective(operator);
    }

    @CacheEvict(value = { "operatorCache" }, allEntries = true)
    @Override
    public void deleteObjById(String id) {
        operatorMapper.deleteByPrimaryKey(id);
        
        RoleRightExample roleRightExample = new RoleRightExample();
        roleRightExample.createCriteria()
            .andKeyIdEqualTo(id)
            .andTypeEqualTo(AuthorizationData.TYPE_OPERATOR);
        
        this.roleRightService.deleteByExample(roleRightExample);
    }

    @CacheEvict(value = { "operatorCache" }, allEntries = true)
    @Override
    public void modifyObj(Operator operator) {
        if (operator == null || StringUtils.isBlank(operator.getId())) {
            throw new IllegalArgumentException("id不能为空");
        }

        OperatorExample operatorExample = new OperatorExample();
        operatorExample.createCriteria().andOperCodeEqualTo(operator.getOperCode()).andIdNotEqualTo(operator.getId());

        if (this.operatorMapper.countByExample(operatorExample) > 0) {
            throw new CommonException("操作code值已存在", "base_is_exists","code值");
        }
        
        operator.setUpdateTime(new Date());
        operatorMapper.updateByPrimaryKeySelective(operator);
    }

    @Override
    public void editOperatorsByMenuId(Operator operator) {
        if (StringUtils.isNotBlank(operator.getId())) {
            this.modifyObj(operator);
        } else {
            this.addObj(operator);
        }

    }

    @Cacheable(value = "operatorCache", keyGenerator="redisKeyGenerator")
    @Override
    public Operator queryObjById(String id) {
        return this.operatorMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value = "operatorCache", keyGenerator="redisKeyGenerator")
    @Override
    public List<Operator> queryAllOperatorsByMenuId(String menuId) {
        OperatorExample example = new OperatorExample();
        example.createCriteria().andMenuIdEqualTo(menuId);
        return operatorMapper.selectByExample(example);
    }

    @Cacheable(value = "operatorCache", keyGenerator="redisKeyGenerator")
    @Override
    public List<Operator> queryAllObjByExample(OperatorExample example) {
        return this.operatorMapper.selectByExample(example);
    }

    @Cacheable(value = "operatorCache", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<Operator> queryObjByPage(OperatorExample example) {
        PageView<Operator> pageView = example.getPageView();
        pageView.setQueryResult(this.operatorMapper.selectByExampleByPage(example));
        return pageView;
    }

    @CacheEvict(value = { "operatorCache" }, allEntries = true)
    @Override
    public void deleteByExample(OperatorExample example) {
        this.operatorMapper.deleteByExample(example);
    }

    @CacheEvict(value = { "operatorCache" }, allEntries = true)
    @Override
    public void modifyByExample(Operator operator, OperatorExample example) {
        this.operatorMapper.updateByExampleSelective(operator, example);
    }

	private List<Operator> queryByRoleIds(List<String> currentRoleIds, List<String> targetRoleIds) {
		if (CollectionUtils.isEmpty(currentRoleIds)) {
			return null;
		}
		
		//1、查询当前角色已经拥有权限的请求url
		OperatorExample operatorExample = new OperatorExample();
		OperatorExample.Criteria criteria = operatorExample.createCriteria();
		
		if (!currentRoleIds.contains(AuthConstant.SYS_MANAGER_ROLE.getId())) {
			List<String> currentOperatorIds = this.queryOperatorIdsByRoleIds(currentRoleIds);
			if (CollectionUtils.isEmpty(currentOperatorIds)) {
				return null;
			}
			criteria.andIdIn(currentOperatorIds);
		}
		
		List<Operator> operatorList = this.operatorMapper.selectByExample(operatorExample);
		if(CollectionUtils.isEmpty(operatorList)){
			return operatorList;
		}
		
		if (targetRoleIds.contains(AuthConstant.SYS_MANAGER_ROLE.getId())
				|| currentRoleIds.equals(targetRoleIds)) {
			//如果角色中包含系统管理员，就不用过滤授权
			//如果授权的角色就是当前角色，也不用过滤授权
			return operatorList;
		}
		
		//2、要目标角色已经拥有的请求url
		List<String> targetOperatorIds = this.queryOperatorIdsByRoleIds(targetRoleIds);
		for(int i=0; i<operatorList.size(); i++){
			Operator operator = operatorList.get(i);
			operator.setAuthorized(targetOperatorIds.contains(operator.getId()));
		}
		return operatorList;
	}

	private List<String> queryOperatorIdsByRoleIds(List<String> roleIds) {
		if (CollectionUtils.isEmpty(roleIds)) {
			return new ArrayList<>();
		}
		
		RoleRightExample roleRightExample = new RoleRightExample();
		roleRightExample.createCriteria().andRoleIdIn(roleIds).andTypeEqualTo(AuthorizationData.TYPE_OPERATOR);
		
		List<RoleRight> roleRightList = this.roleRightService.queryAllObjByExample(roleRightExample);
		if (CollectionUtils.isEmpty(roleRightList)) {
			return new ArrayList<>();
		}
		
		List<String> operatorIds = new ArrayList<>();
		for(RoleRight roleRight : roleRightList){
			operatorIds.add(roleRight.getKeyId());
		}
		return operatorIds;
	}

	@Override
	public void setAuthDataInMenu(List<Menu> allMenu, List<String> currentRoleIds, List<String> targetRoleIds) {
		if (CollectionUtils.isEmpty(allMenu) || CollectionUtils.isEmpty(currentRoleIds)) {
			return;
		}
		
		List<Operator> allOperator = this.queryByRoleIds(currentRoleIds, targetRoleIds);
		if (CollectionUtils.isEmpty(allOperator)) {
			return;
		}
		this.setOperatorListInMenu(allMenu, allOperator);
	}

	private void setOperatorListInMenu(List<Menu> allMenu, List<Operator> allOperator) {
		if (CollectionUtils.isEmpty(allMenu) || CollectionUtils.isEmpty(allOperator)) {
			return;
		}
		
		for(Menu menu : allMenu){
			List<Operator> operatorsInMenu = menu.getOperatorList();
			if (operatorsInMenu == null) {
				operatorsInMenu = new ArrayList<>();
				menu.setOperatorList(operatorsInMenu);
			}
			
			for(Operator operator : allOperator){
				if (menu.getId().equals(operator.getMenuId())) {
					operatorsInMenu.add(operator);
				}
			}
			
			this.setOperatorListInMenu(menu.getChildren(), allOperator);
		}
	}

}
