package com.els.base.auth.web.controller;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.auth.entity.Role;
import com.els.base.auth.entity.RoleExample;
import com.els.base.auth.service.RoleService;
import com.els.base.auth.utils.SpringSecurityUtils;
import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.project.ProjectUtils;
import com.els.base.core.utils.query.QueryParam;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

/**
 * Created by libing on 2014-06-02 08:57.
 */
@Api("权限管理-角色管理")
@Controller
@RequestMapping("role")
public class RoleController {

    @Autowired
    private RoleService roleService;

    @ApiOperation(value = "添加角色", httpMethod = "POST")
    @RequestMapping("service/addRole")
    @ResponseBody
    public ResponseResult<String> addRole(@RequestBody Role role) {
        role.setId(null);
        role.setProjectId(ProjectUtils.getProjectId());
        role.setCreateUserId(SpringSecurityUtils.getLoginUserId());
        this.roleService.addObj(role);

        return ResponseResult.success();
    }

    @ApiOperation(value = "删除角色", httpMethod = "POST")
    @RequestMapping("service/deleteRole")
    @ResponseBody
    public ResponseResult<String> deleteRole(
            @ApiParam(value = "角色id") @RequestParam(required = true) String roleId) {
    	Role role = this.roleService.queryObjById(roleId);
    	if (role == null) {
			throw new CommonException("该数据不存在", "do_not_exists","角色数据");
		}
    	
    	if (Role.ROLE_LEVEL_SYS.equals(role.getRoleLevel()) && !SpringSecurityUtils.isManagerLoginUser()) {
			throw new CommonException("该角色是系统角色无法修改", "can_not_delete_sys_role");
		}

        this.roleService.deleteObjById(roleId);

        return ResponseResult.success();
    }

    @ApiOperation(value = "修改角色信息", httpMethod = "POST")
    @RequestMapping("service/modifyRole")
    @ResponseBody
    public ResponseResult<String> modifyRole(@RequestBody Role role) {

    	Role temp = this.roleService.queryObjById(role.getId());
    	
    	if (Role.ROLE_LEVEL_SYS.equals(temp.getRoleLevel()) && !SpringSecurityUtils.isManagerLoginUser()) {
			throw new CommonException("该角色是系统角色无法修改", "can_not_delete_sys_role");
		}
    	
    	role.setRoleLevel(null);
    	role.setProjectId(temp.getProjectId());
    	role.setCompanyId(temp.getCompanyId());
        this.roleService.modifyObj(role);

        return ResponseResult.success();
    }

    @ApiOperation(value = "查询所有角色，并标记是否已授权", httpMethod = "GET")
    @RequestMapping("service/queryAllRoleWithAuthTag")
    @ResponseBody
    public ResponseResult<List<Role>> queryAllRoleWithAuthTag(@ApiParam(value = "用户id") String userId) {
        if (StringUtils.isBlank(userId)) {
            throw new CommonException("id为空，不能操作", "id_is_blank");
        }

        List<Role> allRoles = this.roleService.queryAllObjByExample(null);
        List<Role> rolesOfUser = this.roleService.queryUserOwnRoles(userId);
        if (CollectionUtils.isEmpty(rolesOfUser)) {
            return ResponseResult.success(allRoles);
        }

        List<String> roleIdsOfUser = new ArrayList<>();
        for (Role role : rolesOfUser) {
            roleIdsOfUser.add(role.getId());
        }

        for (Role role : allRoles) {
            role.setAuthorized(roleIdsOfUser.contains(role.getId()));
        }

        return ResponseResult.success(allRoles);
    }

    @ApiOperation(value = "分页查询所有的角色", httpMethod = "POST")
    @RequestMapping("service/queryAllRoleByPage")
    @ResponseBody
    public ResponseResult<PageView<Role>> queryAllRoleByPage(
            @ApiParam(value = "所在页", defaultValue = "1") @RequestParam(defaultValue = "1") int pageNo,
            @ApiParam(value = "每页数量", defaultValue = "10") @RequestParam(defaultValue = "10") int pageSize,
            @ApiParam(value = "查询条件") @RequestBody(required = false) List<QueryParam> queryParams) {

//    	SpringSecurityUtils
    	
        RoleExample roleExample = new RoleExample();
        roleExample.setPageView(new PageView<Role>(pageNo, pageSize));
        RoleExample.Criteria criteria = roleExample.createCriteria();
        criteria.andProjectIdEqualTo(ProjectUtils.getProjectId());
        
        RoleExample.Criteria or = roleExample.or();
        or.andProjectIdIsNull();
        if (CollectionUtils.isNotEmpty(queryParams)) {
            CriteriaUtils.addCriterion(criteria, queryParams);
        }

        roleExample.setOrderByClause("CREATE_TIME DESC");
        PageView<Role> pageData = roleService.queryObjByPage(roleExample);
        return ResponseResult.success(pageData);
    }

}
