package com.els.base.auth.web.security.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.security.access.ConfigAttribute;
import org.springframework.security.web.FilterInvocation;
import org.springframework.security.web.access.intercept.FilterInvocationSecurityMetadataSource;
import org.springframework.stereotype.Component;

import com.els.base.auth.config.SecurityConfiguration;
import com.els.base.auth.entity.Role;
import com.els.base.auth.entity.RoleExample;
import com.els.base.auth.service.RoleRightService;
import com.els.base.auth.service.RoleService;
import com.els.base.auth.utils.AuthConstant;

/**
 * 操作权限生成器，针对于用户操作(例如：url请求),定义操作所需要的权限。
 * @author hzy
 *
 */
@Component("customFilterInvocationSecurityMetadataSource")
public class CustomFilterInvocationSecurityMetadataSource implements FilterInvocationSecurityMetadataSource {

	private static Pattern pattern = Pattern.compile("(.+?)\\?.+");
	
	@Resource
	private RoleService roleService;
	
	@Resource
	private RoleRightService roleRightService;

	@Override
	public Collection<ConfigAttribute> getAttributes(Object object) throws IllegalArgumentException {
		final HttpServletRequest request = ((FilterInvocation) object).getRequest();
		if (this.isAnonymousAccess(request)) {
			return Arrays.asList(AuthConstant.ANONYMOUS_ROLE);
		}
		
		String url = ((FilterInvocation) object).getRequestUrl().trim();
		if (StringUtils.isBlank(url)) {
			return null;
		}
		if ("/".equals(url)) {
			return null;
		}
		
		if (url.contains("?")) {
			Matcher matcher = pattern.matcher(url);
			url = matcher.find() ? matcher.group(1) : url; 
		}
		
		List<Role> roleList = this.roleRightService.getRoleByOperatorUrl(url);
		Collection<ConfigAttribute> tmpList = new ArrayList<>();
		if (CollectionUtils.isNotEmpty(roleList)) {
			CollectionUtils.addAll(tmpList, roleList.iterator());
			
		}else {
			//如果没有配置，就最低限度是默认角色
			tmpList.add(AuthConstant.DEFAULT_ROLE);
		}
		
		return tmpList;
	}

	private boolean isAnonymousAccess(HttpServletRequest request) {
		if(CollectionUtils.isEmpty(SecurityConfiguration.getAnonymousMatchers())){
			return false;
		}
		
		// 符合任意一个匿名访问的路径
		return SecurityConfiguration.getAnonymousMatchers().stream()
				.anyMatch(matcher-> matcher.matches(request));
	}

	@Override
	public Collection<ConfigAttribute> getAllConfigAttributes() {
		RoleExample roleExample = new RoleExample();
//		List<Role> roleList = this.roleMapper.selectByExample(roleExample);
		List<Role> roleList = this.roleService.queryAllObjByExample(roleExample);
		
		Collection<ConfigAttribute> tmpList = new ArrayList<>();
		CollectionUtils.addAll(tmpList, roleList.iterator());
		return tmpList;
		
	}

	@Override
	public boolean supports(Class<?> clazz) {
		// TODO Auto-generated method stub
		return false;
	}

	
}
