package com.els.base.auth.service.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.auth.dao.MenuMapper;
import com.els.base.auth.entity.AuthorizationData;
import com.els.base.auth.entity.Menu;
import com.els.base.auth.entity.MenuExample;
import com.els.base.auth.entity.OperatorExample;
import com.els.base.auth.entity.RoleRight;
import com.els.base.auth.entity.RoleRightExample;
import com.els.base.auth.service.ButtonService;
import com.els.base.auth.service.MenuService;
import com.els.base.auth.service.OperatorService;
import com.els.base.auth.service.RoleRightService;
import com.els.base.auth.utils.AuthConstant;
import com.els.base.auth.utils.MenuComparator;
import com.els.base.core.entity.PageView;
import com.els.base.core.exception.CommonException;

@Service
public class MenuServiceImpl implements MenuService {

	private static final int MAX_CALLCOUNT = 5;

	@Resource
	private MenuMapper menuMapper;

	@Resource
	private RoleRightService roleRightService;

	@Resource
	private OperatorService operatorService;

	@Resource
	private ButtonService buttonService;

	// 添加菜单
	@Transactional
	@CacheEvict(value = "menu", allEntries = true)
	@Override
	public void addObj(Menu menu) {
		MenuExample menuExample = new MenuExample();
		menuExample.createCriteria().andMenuCodeEqualTo(menu.getMenuCode());
		if (this.menuMapper.countByExample(menuExample) > 0) {
			throw new CommonException("菜单code值已存在", "base_is_exists", "code值");
		}
		if (menu.getMenuCode().length() > 30) {
			throw new CommonException("菜单栏目代码长度不能超过30", "length_canot_exceed", "菜单栏目代码", "30");
		}
		if (menu.getMenuName().length() > 100) {
			throw new CommonException("名称长度不能超过100", "length_canot_exceed", "名称", "100");
		}

		menuMapper.insertSelective(menu);

	}

	@CacheEvict(value = "menu", allEntries = true)
	@Override
	public void deleteObjById(String menuId) {
		// 递归删除子节点
		deleteChildrenMenu(menuId, 0);
	}

	@Transactional
	@CacheEvict(value = "menu", allEntries = true)
	@Override
	public void modifyObj(Menu menu) {
		if (StringUtils.isBlank(menu.getId())) {
			throw new CommonException("菜单id不能为空", "id_is_blank");
		}

		if (menu.getMenuCode().length() > 30) {
			throw new CommonException("菜单栏目代码长度不能超过30", "length_canot_exceed", "菜单栏目代码", "30");
		}
		if (menu.getMenuName().length() > 100) {
			throw new CommonException("名称长度不能超过100", "length_canot_exceed", "名称", "100");

		}

		Menu source = this.queryObjById(menu.getId());
		if (StringUtils.isNotBlank(menu.getMenuCode()) && !menu.getMenuCode().equals(source.getMenuCode())) {

			MenuExample menuExample = new MenuExample();
			menuExample.createCriteria().andMenuCodeEqualTo(menu.getMenuCode());
			if (this.menuMapper.countByExample(menuExample) > 0) {
				throw new CommonException("菜单code值已存在", "base_is_exists", "code值");
			}
		}

		menuMapper.updateByPrimaryKeySelective(menu);

	}

	@Cacheable(value = "menu", keyGenerator = "redisKeyGenerator")
	@Override
	public Menu queryObjById(String id) {
		return this.menuMapper.selectByPrimaryKey(id);
	}

	@Cacheable(value = "menu", keyGenerator = "redisKeyGenerator")
	@Override
	public List<Menu> queryAllObjByExample(MenuExample example) {
		return this.menuMapper.selectByExample(example);
	}

	@Cacheable(value = "menu", keyGenerator = "redisKeyGenerator")
	@Override
	public PageView<Menu> queryObjByPage(MenuExample example) {
		PageView<Menu> pageView = example.getPageView();
		List<Menu> list = this.menuMapper.selectByExampleByPage(example);
		pageView.setQueryResult(list);
		return pageView;
	}

//	@Cacheable(value = "menu", keyGenerator = "redisKeyGenerator")
	private List<Menu> getChildrendMenu(String parentMenuId, List<String> visiableMenuId, List<String> authedMenuIds, boolean isFilter) {
		if (visiableMenuId == null) {
			if (isFilter) {
				return null;
			}
			visiableMenuId = new ArrayList<>();
		}
		if (authedMenuIds == null) {
			authedMenuIds = new ArrayList<>();
		}
		
		MenuExample menuExample = new MenuExample();
		MenuExample.Criteria criteria = menuExample.createCriteria().andParentIdEqualTo(parentMenuId);
		
		if (isFilter) {
			criteria.andIdIn(visiableMenuId);
		}
		
		menuExample.setOrderByClause("sort_no ASC");
		List<Menu> childrenMenus = this.menuMapper.selectByExample(menuExample);

		for (int i = 0; CollectionUtils.isNotEmpty(childrenMenus) && i < childrenMenus.size(); i++) {
			Menu subMenu = childrenMenus.get(i);
			if (!authedMenuIds.equals(visiableMenuId) && isFilter) {
				subMenu.setAuthorized(authedMenuIds.contains(subMenu.getId()));
			}
			
			subMenu.setChildren(this.getChildrendMenu(subMenu.getId(), visiableMenuId, authedMenuIds, isFilter));
		}
		return childrenMenus;
	}

	// 递归调用删除所有的子菜单
	private void deleteChildrenMenu(String menuId, int callCount) {
		callCount++;
		// 判断递归深度
		if (callCount > MAX_CALLCOUNT) {
			return;
		}
		// List<Menu> menuChildren = menuMapper.queryMenuByParentId(menuId);
		MenuExample menuExample = new MenuExample();
		menuExample.createCriteria().andParentIdEqualTo(menuId);
		menuExample.setOrderByClause("sort_no ASC");

		List<Menu> menuChildren = this.menuMapper.selectByExample(menuExample);

		for (Menu subMenu : menuChildren) {
			deleteChildrenMenu(subMenu.getId(), callCount);
		}

		this.menuMapper.deleteByPrimaryKey(menuId);

		OperatorExample operatorExample = new OperatorExample();
		operatorExample.createCriteria().andMenuIdEqualTo(menuId);
		this.operatorService.deleteByExample(operatorExample);
		
		this.buttonService.deleteObjByMenuId(menuId);
	}

	@Cacheable(value = "menu", keyGenerator = "redisKeyGenerator")
	@Override
	public List<Menu> queryAllMenuWithAuthRoleIds(List<String> currentRoleIds, List<String> targetRoleIds) {
		if (CollectionUtils.isEmpty(currentRoleIds)) {
			throw new CommonException("当前的角色为空");
		}
		
		MenuExample menuExample = new MenuExample();
		MenuExample.Criteria criteria = menuExample.createCriteria();
		if (!currentRoleIds.contains(AuthConstant.SYS_MANAGER_ROLE.getId())) {
			List<String> currentMenuIds = this.queryMenuIdByRoleId(currentRoleIds);
			if(CollectionUtils.isEmpty(currentMenuIds)) return null;
			criteria.andIdIn(currentMenuIds);
		}
		
		List<Menu> currentMenus = this.queryAllObjByExample(menuExample);
		if (CollectionUtils.isEmpty(currentMenus)) {
			return null;
		}
		
		if (!targetRoleIds.contains(AuthConstant.SYS_MANAGER_ROLE.getId())
				&& !currentRoleIds.equals(targetRoleIds)) {
			
			List<String> targetMenuId = this.queryMenuIdByRoleId(targetRoleIds);
			for(Menu menu : currentMenus){
				menu.setAuthorized(targetMenuId.contains(menu.getId()));
			}
		}
		
		List<Menu> allMenu = new ArrayList<>();
		for(Menu menu : currentMenus){
			if (!Menu.ROOT_ID.equals(menu.getParentId())) {
				continue;
			}
			
			allMenu.add(menu);
			
			List<Menu> children = this.findChildren(menu, currentMenus);
			menu.setChildren(children);
		}
		Collections.sort(allMenu, new MenuComparator());
		
		this.buttonService.setAuthDataInMenu(allMenu, currentRoleIds, targetRoleIds);
		this.operatorService.setAuthDataInMenu(allMenu, currentRoleIds, targetRoleIds);
		return allMenu;
		
	}

	private List<Menu> findChildren(Menu parent, List<Menu> currentMenus) {
		
		List<Menu> allMenu = new ArrayList<>();
		
		for(Menu child : currentMenus){
			if (!parent.getId().equals(child.getParentId())) {
				continue;
			}
			allMenu.add(child);
			
			List<Menu> children = this.findChildren(child, currentMenus);
			child.setChildren(children);
		}
		Collections.sort(allMenu, new MenuComparator());
		return allMenu;
	}

	@Cacheable(value = "menu", keyGenerator = "redisKeyGenerator")
	@Override
	public List<Menu> queryAuthedMenuForRoleIds(List<String> roleIds) {
		return this.queryAllMenuWithAuthRoleIds(roleIds, roleIds);
	}
	
	private List<String> queryMenuIdByRoleId(List<String> roleIds) {
        RoleRightExample roleRightExample = new RoleRightExample();
        roleRightExample.createCriteria().andRoleIdIn(roleIds).andTypeEqualTo(AuthorizationData.TYPE_MENU);

        List<RoleRight> roleRightList = this.roleRightService.queryAllObjByExample(roleRightExample);
        List<String> authMenuIds = new ArrayList<>();

        for (RoleRight roleRight : roleRightList) {
            authMenuIds.add(roleRight.getKeyId());
        }

        return authMenuIds;
    }

}
