package com.els.base.auth.config;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.annotation.Resource;
import javax.servlet.Filter;

import org.springframework.context.annotation.Bean;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.ProviderManager;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.builders.WebSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.security.web.access.ExceptionTranslationFilter;
import org.springframework.security.web.access.intercept.FilterSecurityInterceptor;
import org.springframework.security.web.authentication.AuthenticationFailureHandler;
import org.springframework.security.web.authentication.AuthenticationSuccessHandler;
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter;
import org.springframework.security.web.authentication.logout.LogoutSuccessHandler;
import org.springframework.security.web.session.ConcurrentSessionFilter;
import org.springframework.security.web.session.InvalidSessionStrategy;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;

import com.els.base.auth.web.security.impl.CustomAccessDecisionManagerImpl;
import com.els.base.auth.web.security.impl.CustomAccessDeniedHandler;
import com.els.base.auth.web.security.impl.CustomAuthenticationEntryPoint;
import com.els.base.auth.web.security.impl.CustomCaptchaAuthenticationFilter;
import com.els.base.auth.web.security.impl.CustomDaoAuthenticationProvider;
import com.els.base.auth.web.security.impl.CustomFilterInvocationSecurityMetadataSource;
import com.els.base.auth.web.security.impl.CustomInvalidSessionStrategy;
import com.els.base.auth.web.security.impl.CustomLogoutSuccessHandler;
import com.els.base.auth.web.security.impl.LocaleChangeAndSaveFilter;

@EnableWebSecurity
public class SecurityConfiguration extends WebSecurityConfigurerAdapter {

	@Resource(name="customPasswordEncoder")
	PasswordEncoder passwordEncoder;
	
	@Resource(name="customUserDetailsService")
	UserDetailsService userDetailsService;

	@Resource(name="customAuthenticationFailureHandler")
	AuthenticationFailureHandler failureHandler;

	@Resource(name="customAuthenticationSuccessHandler")
	protected AuthenticationSuccessHandler successHandler;
	
	@Resource(name="accessDeniedHandler")
	protected CustomAccessDeniedHandler accessDeniedHandler;
	
	@Resource(name="customAccessDecisionManager")
	protected CustomAccessDecisionManagerImpl customAccessDecisionManager;
	
	@Resource(name="customFilterInvocationSecurityMetadataSource")
	protected CustomFilterInvocationSecurityMetadataSource customFilterInvocationSecurityMetadataSource;
	
	private static List<RequestMatcher> anonymousMatchers = new ArrayList<RequestMatcher>();
	
	private static void configAnonymousMatchers(String... antPatterns) {
		for (String pattern : antPatterns) {
			anonymousMatchers.add(new AntPathRequestMatcher(pattern));
		}
	}
	
	public static List<RequestMatcher> getAnonymousMatchers() {
		return anonymousMatchers;
	}

	@Override
	public void configure(WebSecurity web) throws Exception {
		web.ignoring()
			// 资源文件，api接口，移动端文件，登录页面不拦截
			.antMatchers("/resources/**", "/swagger/**", "/mobileview/**", "/login.html*")
			// api接口数据，验证码请求
			.antMatchers("/captcha/**", "/v2/api-docs");
	}

	@Override
	protected void configure(HttpSecurity http) throws Exception {
		http.addFilterBefore(this.getLocaleChangeAndSaveFilter(), ConcurrentSessionFilter.class)
			.addFilterBefore(this.getCaptchaAuthenticaionFilter(), UsernamePasswordAuthenticationFilter.class)
			.addFilterBefore(this.getExceptionTranslationFilter(), FilterSecurityInterceptor.class)
			.addFilterBefore(this.getCustomFilterSecurityInterceptor(), FilterSecurityInterceptor.class);
		
		http.sessionManagement().sessionFixation().migrateSession();
		http.sessionManagement().invalidSessionStrategy(this.getInvalidSessionStrategy());
		http.sessionManagement().sessionAuthenticationErrorUrl("/login.html");
		http.sessionManagement().maximumSessions(1).maxSessionsPreventsLogin(false).expiredUrl("/login.html");
		
		http.formLogin()
			.failureHandler(this.failureHandler)
			
		.and()
			.logout()
			.invalidateHttpSession(true)
			.logoutUrl("/logout")
			.logoutSuccessHandler(this.getCustomLogoutSuccessHandler())
			
		.and()
			.csrf().disable();
			
		configAnonymousMatchers("/*/front/**");
	}

	@Bean("customLogoutSuccessHandler")
	protected LogoutSuccessHandler getCustomLogoutSuccessHandler() {
		CustomLogoutSuccessHandler handler = new CustomLogoutSuccessHandler();
		handler.setDefaultTargetUrl("/logout.html");
		return handler;
	}

	private Filter getCustomFilterSecurityInterceptor() {
		FilterSecurityInterceptor filter = new FilterSecurityInterceptor();
		filter.setAuthenticationManager(this.getAuthenticationManager());
		filter.setAccessDecisionManager(customAccessDecisionManager);
		filter.setSecurityMetadataSource(customFilterInvocationSecurityMetadataSource);
		return filter;
	}

	@Bean("captchaAuthenticaionFilter")
	protected Filter getCaptchaAuthenticaionFilter() {
		CustomCaptchaAuthenticationFilter filter = new CustomCaptchaAuthenticationFilter();
		filter.setAuthenticationManager(this.getAuthenticationManager());
		filter.setAuthenticationFailureHandler(failureHandler);
		filter.setAuthenticationSuccessHandler(successHandler);
		filter.setFilterProcessesUrl("/loginForSpringSecurity");
		return filter;
	}

	@Bean("authenticationManager")
	protected AuthenticationManager getAuthenticationManager() {
		
		List<AuthenticationProvider> providers = Arrays.asList(this.getAuthenticationProvider());
		AuthenticationManager authenticationManager = new ProviderManager(providers);
		
		return authenticationManager;
	}

	@Bean("daoAuthenticationProvider")
	protected AuthenticationProvider getAuthenticationProvider() {

		CustomDaoAuthenticationProvider authenticationProvider = new CustomDaoAuthenticationProvider();
		authenticationProvider.setPasswordEncoder(passwordEncoder);
		authenticationProvider.setUserDetailsService(userDetailsService);
		authenticationProvider.setHideUserNotFoundExceptions(true);
		
		return authenticationProvider;
	}

	@Bean("customExceptionTranslationFilter")
	protected Filter getExceptionTranslationFilter() {
		ExceptionTranslationFilter filter = new ExceptionTranslationFilter(this.getAuthenticationEntryPoint());
		filter.setAccessDeniedHandler(this.accessDeniedHandler);
		
		return filter;
	}

	@Bean("authenticationEntryPoint")
	protected AuthenticationEntryPoint getAuthenticationEntryPoint() {
		AuthenticationEntryPoint entryPoint = new CustomAuthenticationEntryPoint("/login.html");
		return entryPoint;
	}

	@Bean("localeChangeAndSaveFilter")
	protected Filter getLocaleChangeAndSaveFilter() {
		return new LocaleChangeAndSaveFilter();
	}

	@Bean("customInvalidSessionStrategy")
	protected InvalidSessionStrategy getInvalidSessionStrategy(){
		CustomInvalidSessionStrategy strategy =  new CustomInvalidSessionStrategy("/login.html");
		return strategy;
	}

}
