package com.els.base.auth.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.els.base.auth.dao.ButtonMapper;
import com.els.base.auth.entity.AuthorizationData;
import com.els.base.auth.entity.Button;
import com.els.base.auth.entity.ButtonExample;
import com.els.base.auth.entity.Menu;
import com.els.base.auth.entity.RoleRight;
import com.els.base.auth.entity.RoleRightExample;
import com.els.base.auth.service.ButtonService;
import com.els.base.auth.service.RoleRightService;
import com.els.base.auth.utils.AuthConstant;
import com.els.base.core.entity.PageView;
import com.els.base.core.utils.Assert;

@Service("defaultButtonService")
public class ButtonServiceImpl implements ButtonService {
    @Resource
    protected ButtonMapper buttonMapper;
    
    @Resource
    private RoleRightService roleRightService;

    @CacheEvict(value={"button", "menu"}, allEntries=true)
    @Override
    public void addObj(Button button) {
    	Assert.isNotBlank(button.getBtnCode(), "code值不能为空");
    	Assert.isNotBlank(button.getMenuId(), "menuId不能为空");
    	
    	button.setCreateTime(new Date());
    	button.setUpdateTime(new Date());
        this.buttonMapper.insertSelective(button);
    }
    
    @Transactional
    @CacheEvict(value={"button", "menu"}, allEntries=true)
    @Override
    public void deleteObjById(String id) {
        this.buttonMapper.deleteByPrimaryKey(id);
        
        RoleRightExample roleRightExample = new RoleRightExample();
        roleRightExample.createCriteria()
            .andKeyIdEqualTo(id)
            .andTypeEqualTo(AuthorizationData.TYPE_BUTTON);
        
        this.roleRightService.deleteByExample(roleRightExample);
    }

    @CacheEvict(value={"button", "menu"}, allEntries=true)
    @Override
    public void modifyObj(Button t) {
        if (StringUtils.isBlank(t.getId())) {
            throw new NullPointerException("id 为空，无法更新");
        }
        t.setUpdateTime(new Date());
        this.buttonMapper.updateByPrimaryKeySelective(t);
    }

    @Cacheable(value="button", keyGenerator="redisKeyGenerator")
    @Override
    public Button queryObjById(String id) {
        return this.buttonMapper.selectByPrimaryKey(id);
    }

    @Cacheable(value="button", keyGenerator="redisKeyGenerator")
    @Override
    public List<Button> queryAllObjByExample(ButtonExample example) {
        return this.buttonMapper.selectByExample(example);
    }

    @Cacheable(value="button", keyGenerator="redisKeyGenerator")
    @Override
    public PageView<Button> queryObjByPage(ButtonExample example) {
        PageView<Button> pageView = example.getPageView();
        pageView.setQueryResult(this.buttonMapper.selectByExampleByPage(example));
        return pageView;
    }
    
	@Override
	public void setAuthDataInMenu(List<Menu> allMenu, List<String> currentRoleIds, List<String> targetRoleIds) {
		if (CollectionUtils.isEmpty(allMenu) || CollectionUtils.isEmpty(currentRoleIds)) {
			return;
		}
		
		List<Button> allButtons = this.queryByRoleIds(currentRoleIds, targetRoleIds);
		if (CollectionUtils.isEmpty(allButtons)) {
			return;
		}
		this.setButtonListInMenu(allMenu, allButtons);
	}

	private void setButtonListInMenu(List<Menu> allMenu, List<Button> allButtonList) {
		if (CollectionUtils.isEmpty(allMenu) || CollectionUtils.isEmpty(allButtonList)) {
			return;
		}
		
		for(Menu menu : allMenu){
			List<Button> buttonsInMenu = menu.getButtonList();
			if (buttonsInMenu == null) {
				buttonsInMenu = new ArrayList<>();
				menu.setButtonList(buttonsInMenu);
			}
			
			for(Button button : allButtonList){
				if (menu.getId().equals(button.getMenuId())) {
					buttonsInMenu.add(button);
				}
			}
			
			this.setButtonListInMenu(menu.getChildren(), allButtonList);
		}
	}

	private List<Button> queryByRoleIds(List<String> currentRoleIds, List<String> targetRoleIds) {
		if (CollectionUtils.isEmpty(currentRoleIds)) {
			return null;
		}
		
		//1、查询当前角色拥有的按钮
		ButtonExample operatorExample = new ButtonExample();
		ButtonExample.Criteria criteria = operatorExample.createCriteria();
		
		if (!currentRoleIds.contains(AuthConstant.SYS_MANAGER_ROLE.getId())) {
			
			List<String> currentButtonIds = this.queryButtonIdsByRoleIds(currentRoleIds);
			if (CollectionUtils.isEmpty(currentButtonIds)) {
				return null;
			}
			criteria.andIdIn(currentButtonIds);
		}
		
		List<Button> buttonList = this.buttonMapper.selectByExample(operatorExample);
		if (CollectionUtils.isEmpty(buttonList)) {
			return buttonList;
		}
		
		if (targetRoleIds.contains(AuthConstant.SYS_MANAGER_ROLE.getId())
				|| currentRoleIds.equals(targetRoleIds)) {
			
			//如果角色中包含系统管理员，就不用过滤授权
			//如果授权的角色就是当前角色，也不用过滤授权
			return buttonList;
		}
		
		List<String> targetButtonIds = this.queryButtonIdsByRoleIds(targetRoleIds);
		for(int i=0;  i<buttonList.size(); i++){
			Button button = buttonList.get(i);
			button.setAuthorized(targetButtonIds.contains(button.getId()));
		}
		return buttonList;
	}
	
	private List<String> queryButtonIdsByRoleIds(List<String> roleIds){
		if (CollectionUtils.isEmpty(roleIds)) {
			return new ArrayList<>();
		}
		
		RoleRightExample roleRightExample = new RoleRightExample();
		roleRightExample.createCriteria().andRoleIdIn(roleIds).andTypeEqualTo(AuthorizationData.TYPE_BUTTON);
		
		List<RoleRight> roleRightList = this.roleRightService.queryAllObjByExample(roleRightExample);
		if (CollectionUtils.isEmpty(roleRightList)) {
			return new ArrayList<>();
		}
		
		List<String> buttonIds = new ArrayList<>();
		for(RoleRight roleRight : roleRightList){
			buttonIds.add(roleRight.getKeyId());
		}
		return buttonIds;
	}

	@CacheEvict(value={"button", "menu"}, allEntries=true)
	@Override
	public void deleteObjByMenuId(String menuId) {
		ButtonExample buttonExample = new ButtonExample();
		buttonExample.createCriteria().andMenuIdEqualTo(menuId);
		this.buttonMapper.deleteByExample(buttonExample);
	}
}