package com.els.base.core.web.controller.dictionary;

import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.entity.dictionary.DicGroup;
import com.els.base.core.entity.dictionary.DicGroupExample;
import com.els.base.core.exception.CommonException;
import com.els.base.core.service.dictionary.DicGroupService;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.query.QueryParam;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

@Api(value = "字典组")
@Controller
@RequestMapping("dicGroup")
public class DicGroupController {

    @Resource
    protected DicGroupService dicGroupService;

    @ApiOperation(httpMethod = "POST", value = "创建字典组")
    @RequestMapping("service/create")
    @ResponseBody
    public ResponseResult<String> create(@RequestBody DicGroup dicGroup) {
        dicGroupValidtion(dicGroup);

        this.dicGroupService.addObj(dicGroup);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "编辑字典组")
    @RequestMapping("service/edit")
    @ResponseBody
    public ResponseResult<String> edit(@RequestBody DicGroup dicGroup) {
        dicGroupValidtion(dicGroup);

        this.dicGroupService.modifyObj(dicGroup);
        return ResponseResult.success();
    }

    @ApiOperation(httpMethod = "POST", value = "删除字典组")
    @RequestMapping("service/deleteById")
    @ResponseBody
    public ResponseResult<String> deleteById(String id) {
        this.dicGroupService.deleteObjById(id);
        return ResponseResult.success();
    }

    private void dicGroupValidtion(DicGroup dicGroup) {
        if (dicGroup == null || StringUtils.isBlank(dicGroup.getCode())
                || StringUtils.isBlank(dicGroup.getName())) {
            throw new CommonException("参数不完整，请填写必要的信息","parameter_incomplete");

        }

        if (StringUtils.isBlank(dicGroup.getCode()) || !dicGroup.getCode().matches("[\\w\\d\\_\\-]+")) {
        	throw new CommonException("字典组code格式不正确", "base_invaild_format","字典组code");
        }

        DicGroupExample example = new DicGroupExample();
        DicGroupExample.Criteria cri = example.createCriteria();
        cri.andCodeEqualTo(dicGroup.getCode());
        if (dicGroup.getId() != null) {
            cri.andIdNotEqualTo(dicGroup.getId());
        }

        List<DicGroup> list = this.dicGroupService.queryAllObjByExample(example);
        if (list.size() > 0) {
            throw new CommonException("code必须唯一，该字典组中已经存在相同的code","base_is_exists","code");
        }
    }

    @ApiOperation(httpMethod = "GET", value = "查询字典组")
    @RequestMapping("service/findByPage")
    @ResponseBody
    public ResponseResult<PageView<DicGroup>> findByPage(
            @ApiParam(value = "所在页", defaultValue = "1") @RequestParam(defaultValue = "1") int pageNo,
            @ApiParam(value = "每页的数量", defaultValue = "10") @RequestParam(defaultValue = "10") int pageSize,
            @ApiParam(value = "查询条件") @RequestBody(required=false) List<QueryParam> queryParams) {

        DicGroupExample example = new DicGroupExample();
        example.setPageView(new PageView<DicGroup>(pageNo, pageSize));
        example.setOrderByClause("ID DESC");

        DicGroupExample.Criteria criteria = example.createCriteria();
        if (CollectionUtils.isNotEmpty(queryParams)) {
            CriteriaUtils.addCriterion(criteria, queryParams);
        }

        PageView<DicGroup> pageView = this.dicGroupService.queryObjByPage(example);
        return ResponseResult.success(pageView);
    }
    
}
