package com.els.base.schedule.web.controller;

import java.util.List;

import javax.annotation.Resource;

import org.quartz.SchedulerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.els.base.core.entity.PageView;
import com.els.base.core.entity.ResponseResult;
import com.els.base.core.exception.CommonException;
import com.els.base.core.utils.Assert;
import com.els.base.core.utils.CriteriaUtils;
import com.els.base.core.utils.query.QueryParamWapper;
import com.els.base.schedule.entity.ScheduleJob;
import com.els.base.schedule.entity.ScheduleJobExample;
import com.els.base.schedule.plugin.QuartzManager;
import com.els.base.schedule.service.ScheduleJobService;
import com.els.base.utils.SpringContextHolder;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

//@ConditionalOnProperty(name="job.is.enable", havingValue="true", matchIfMissing=true)
@Api(value="定时任务")
@Controller
@RequestMapping("scheduleJob")
public class ScheduleJobController {
	
	private static final Logger logger = LoggerFactory.getLogger(ScheduleJobController.class);
	
	@Resource
	protected ScheduleJobService scheduleJobService;

	private QuartzManager quartzManager;

	private QuartzManager getQuartzManager(){
		if (quartzManager == null) {
			quartzManager = SpringContextHolder.getOneBean(QuartzManager.class);
		}

		if (quartzManager == null) {
			throw new CommonException("系统未开启定时任务功能");
		}
		return quartzManager;
	}

	@ApiOperation(httpMethod="POST", value="创建定时任务")
	@RequestMapping("service/create")
	@ResponseBody
	public ResponseResult<String> create(@RequestBody ScheduleJob scheduleJob)  {
		
		this.scheduleJobService.addObj(scheduleJob);
		return ResponseResult.success(scheduleJob.getId());
	}

	@ApiOperation(httpMethod="POST", value="编辑定时任务")
	@RequestMapping("service/edit")
	@ResponseBody
	public ResponseResult<String> edit(@RequestBody ScheduleJob scheduleJob) {
		
		this.scheduleJobService.modifyObj(scheduleJob);
		return ResponseResult.success();
	}

	@ApiOperation(httpMethod="POST", value="删除定时任务")
	@RequestMapping("service/deleteById")
	@ResponseBody
	public ResponseResult<String> deleteById(String id) {
		
		this.scheduleJobService.deleteObjById(id);
		return ResponseResult.success();
	}


	@ApiOperation(httpMethod="POST", value="删除定时任务")
	@RequestMapping("service/deleteByIds")
	@ResponseBody
	public ResponseResult<String> deleteByIds(@RequestBody List<String> ids) {
		Assert.isNotEmpty(ids,"请选中需要操作的单据！");

		for (String id : ids) {
			this.deleteById(id);
		}
		return ResponseResult.success();
	}

	@ApiOperation(httpMethod="GET", value="查询定时任务")
	@RequestMapping("service/findByPage")
	@ResponseBody
	public ResponseResult<PageView<ScheduleJob>> findByPage(
			@ApiParam(value="所在页",defaultValue="1" ) @RequestParam(defaultValue="1") int pageNo,
			@ApiParam(value="每页数量", defaultValue="10") @RequestParam(defaultValue="10") int pageSize,
			@ApiParam(value="查询条件") @RequestBody(required=false) QueryParamWapper queryParams) {

		ScheduleJobExample example = new ScheduleJobExample();
		example.setPageView(new PageView<ScheduleJob>(pageNo, pageSize));

		ScheduleJobExample.Criteria criteria = example.createCriteria();
		if (queryParams != null) {
			CriteriaUtils.addCriterion(criteria, queryParams);
		}
		return ResponseResult.success(this.scheduleJobService.queryObjByPage(example));
	}

	@ApiOperation(httpMethod="GET", value="现在执行任务")
	@RequestMapping("service/runJobNow")
	@ResponseBody
	public ResponseResult<String> runJobNow(String id) {
		ScheduleJob job = this.scheduleJobService.queryObjById(id);
		Assert.isNotNull(job, "定时任务不存在");
		try {
			getQuartzManager().runJobNow(job);
		} catch (SchedulerException e) {
			logger.error("执行失败", e);
			throw new CommonException("执行失败", "base_fail");
		}
		return ResponseResult.success();
	}

	@ApiOperation(httpMethod="GET", value="停止任务")
	@RequestMapping("service/pauseJob")
	@ResponseBody
	public ResponseResult<String> pauseJob(String id){

		ScheduleJob job = this.scheduleJobService.queryObjById(id);
		Assert.isNotNull(job, "定时任务不存在");
		try {
			getQuartzManager().pauseJob(job);

		} catch (SchedulerException e) {
			logger.error("停止任务失败", e);
			throw new CommonException("job 停止任务失败","base_fail");
		}

		return ResponseResult.success();
	}

	@ApiOperation(httpMethod="GET", value="启动任务")
	@RequestMapping("service/resumeJob")
	@ResponseBody
	public ResponseResult<String> resumeJob(String id){

		ScheduleJob job = this.scheduleJobService.queryObjById(id);
		Assert.isNotNull(job, "定时任务不存在");
		
		try {
			getQuartzManager().resumeJob(job);

		} catch (SchedulerException e) {
			logger.error("启动任务失败", e);
			throw new CommonException("job 启动任务失败","base_fail");
		}

		return ResponseResult.success();
	}
	
}