package com.doudian.open.core;

import com.doudian.open.constant.SDKConstant;
import com.doudian.open.core.http.DefaultHttpClient;
import com.doudian.open.core.http.HttpClient;
import com.doudian.open.core.http.HttpRequest;
import com.doudian.open.core.http.HttpResponse;
import com.doudian.open.core.threadpool.DoudianOpThreadPool;
import com.doudian.open.utils.JsonUtil;
import com.doudian.open.exception.DoudianOpException;
import com.doudian.open.utils.Logger;
import com.doudian.open.utils.SignUtil;


import java.util.concurrent.Callable;
import java.util.concurrent.Future;


public class DefaultDoudianOpClient implements DoudianOpClient{

    private static final Logger LOG = Logger.getLogger(DefaultDoudianOpClient.class);

    private static final DefaultDoudianOpClient DEFAULT_DOUDIAN_OP_CLIENT = new DefaultDoudianOpClient();

    private HttpClient httpClient = DefaultHttpClient.getDefaultClient();

    @SuppressWarnings("unchecked")
    @Override
    public <T, R> T request(DoudianOpRequest<R> request, AccessToken accessToken) {
        String appKey = request.getConfig().getAppKey();
        String appSecret = request.getConfig().getAppSecret();
        String paramJson = JsonUtil.toJson(request.getParam());
        String requestUrlPath = getRequestUrlPath(request.getUrlPath());

        String timestamp = String.valueOf(System.currentTimeMillis());
        String requestUrlPattern = "%s/%s?app_key=%s&method=%s&v=2&sign=%s&timestamp=%s&access_token=%s";
        String method = requestUrlPath.replaceAll("/", ".");
        String sign = SignUtil.sign(appKey, appSecret, method, timestamp, paramJson, "2");
        String accessTokenStr = "";
        if(accessToken != null){
            accessTokenStr = accessToken.getAccessToken();
        }

        //参数校验
        if(request.getConfig().getFieldCheck()) {
            ParameterChecker.check(request.getParam());
        }

        String requestUrl = null;
        requestUrl = String.format(requestUrlPattern, request.getConfig().getOpenRequestUrl(), requestUrlPath, appKey, method, sign, timestamp, accessTokenStr);
        HttpRequest httpRequest = HttpRequest.build(requestUrl, paramJson);
        LOG.info("http request url: %s, body: %s", requestUrl, paramJson);
        httpRequest.addHeader("from", "sdk");
        httpRequest.addHeader("sdk-version", SDKConstant.SDK_VERSION);
        httpRequest.addHeader("sdk-type", "java");
        if(request.getConfig().getHttpClientConnectTimeout() != null){
            httpRequest.setConnectTimeout(request.getConfig().getHttpClientConnectTimeout());
        }
        if(request.getConfig().getHttpClientReadTimeout() != null){
            httpRequest.setReadTimeout(request.getConfig().getHttpClientReadTimeout());
        }
        if(request.getConfig().getHttpRequestHeader() != null && request.getConfig().getHttpRequestHeader().size() > 0) {
            httpRequest.addHeader(request.getConfig().getHttpRequestHeader());
        }

        HttpResponse httpResponse = httpClient.post(httpRequest);
        if(httpResponse.getStatusCode() != 200){
           throw new DoudianOpException(DoudianOpException.Code.HTTP_RESPONSE_STATUS_CODE_NOT_2XX);
        }

        String httpBody = httpResponse.getBody();
        if (httpBody == null || "".equals(httpBody)) {
            //确保反序列化出来的对象不为null
            httpBody = "{}";
        }
        try {
            DoudianOpResponse<?> response = JsonUtil.fromJson(httpBody, request.getResponseClass());
            //将原始的返回数据设置进去
            response.setOriginResponse(httpBody);
            LOG.info("http response: %s", httpBody);
            return (T)response;
        }catch (Exception e){
            throw new DoudianOpException(DoudianOpException.Code.JSON_ERROR, e);
        }
    }

    @Override
    public <T, R> Future<T> asyncRequest(final DoudianOpRequest<R> request, final AccessToken accessToken) {
        return DoudianOpThreadPool.getInstance(request.getConfig()).submit(new Callable<T>() {
            @Override
            public T call() throws Exception {
                return DefaultDoudianOpClient.this.request(request, accessToken);
            }
        });
    }

    private String getRequestUrlPath(String originUrl){
        if (originUrl == null || originUrl.length() == 0){
            return originUrl;
        }
        if (originUrl.startsWith("/")) {
            return originUrl.substring(1);
        }
        return originUrl;
    }

    public static DefaultDoudianOpClient getDefaultClient(){
        return DEFAULT_DOUDIAN_OP_CLIENT;
    }

    public void setHttpClient(HttpClient httpClient) {
        this.httpClient = httpClient;
    }
}
