package com.els.tso.srm.file;

import com.els.tso.common.dto.FileDTO;
import com.els.tso.srm.model.MultipartFileParam;
import com.els.tso.srm.service.FileChunkUploadService;
import com.els.tso.common.exception.FileManipulaException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Date;

@Component("fileLocalTemplate")
public class FileLocalTemplate implements FileTemplate {
    @Autowired
    private FileChunkUploadService fileChunkUploadService;
    @Value("${els.srm.system.file-path}")
    private String systemPath;

    @Override
    public FileDTO statFile(String filePath) {
        String realFilePath = systemPath + "/" + filePath;
        File file = new File(realFilePath);
        if (!file.exists()){
            throw new FileManipulaException("未找到文件");
        }
        return getFileDto(file);
    }

    @Override
    public FileDTO putFile(String fileName, MultipartFile file) throws IOException {
        if (file == null) {
            return null;
        }
        MultipartFileParam multipartFileParam = getMultipartFileParam(file);
        multipartFileParam = fileChunkUploadService.fileChunkUpload(multipartFileParam, fileName);
        return getFileDTO(multipartFileParam);
    }

    @Override
    public void removeFile(String filePath) {
        String realFilePath = systemPath + "/" + filePath;
        File file = new File(realFilePath);
        if (!file.exists()){
            throw new FileManipulaException("未找到文件");
        }
        boolean delete = file.delete();
        if (!delete){
            throw new FileManipulaException("文件删除失败");
        }
    }
    @Override
    public void downloadFile(String filePath) throws IOException  {
        String realFilePath = systemPath + "/" + filePath;
        File file = new File(realFilePath);
        if (!file.exists()){
            throw new FileManipulaException("未找到文件");
        }
        HttpServletResponse response = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getResponse();
        try (
            InputStream inStream = new FileInputStream(file);
            ServletOutputStream outStream = response.getOutputStream()
        ) {
            byte[] buf = new byte[4096];
            int readLength;
            while (((readLength = inStream.read(buf)) != -1)) {
                outStream.write(buf, 0, readLength);
            }
        }
    }

    private FileDTO getFileDto (File file){
        FileDTO fileDTO = new FileDTO();
        fileDTO.setName(file.getName());
        fileDTO.setType(fileChunkUploadService.getFileExt(file.getName()));
        fileDTO.setPutTime(new Date(getFileCreateTime(file.getPath())));
        String filePath = file.getPath();
        fileDTO.setLink(filePath.replace(systemPath.replaceAll("/", "\\\\"),""));
        fileDTO.setLength(file.length());
        return fileDTO;
    }
    private Long getFileCreateTime(String filePath){
        File file = new File(filePath);
        try {
            Path path= Paths.get(filePath);
            BasicFileAttributeView basicview= Files.getFileAttributeView(path, BasicFileAttributeView.class, LinkOption.NOFOLLOW_LINKS );
            BasicFileAttributes attr = basicview.readAttributes();
            return attr.creationTime().toMillis();
        } catch (Exception e) {
            e.printStackTrace();
            return file.lastModified();
        }
    }
    private MultipartFileParam getMultipartFileParam(MultipartFile file) throws IOException {
        long size = file.getSize();
        String filename = file.getOriginalFilename();
        MultipartFileParam multipartFileParam = new MultipartFileParam();
        multipartFileParam.setAttachment(file.getInputStream());
        multipartFileParam.setIdentifier(size + "-" + fileChunkUploadService.getFileExt(filename));
        multipartFileParam.setTotalSize((int) size);
        multipartFileParam.setFilename(filename);
        multipartFileParam.setChunkSize(size);
        multipartFileParam.setChunkNumber(1);
        multipartFileParam.setCurrentChunkSize(size);
        multipartFileParam.setTotalChunks(1);
        return multipartFileParam;
    }

    private FileDTO getFileDTO(MultipartFileParam multipartFileParam) {
        FileDTO fileDTO = new FileDTO();
        fileDTO.setLength(multipartFileParam.getTotalSize());
        fileDTO.setLink(multipartFileParam.getFileStorePath());
        fileDTO.setName(multipartFileParam.getFilename());
        fileDTO.setPutTime(new Date());
        fileDTO.setType(fileChunkUploadService.getFileExt(fileDTO.getName()));
        return fileDTO;
    }

}
